/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { v4 as uuid } from 'uuid';
import { UsageError, isXtermErrorResponse, cwd as kuiCwd } from '@kui-shell/core';
/** Capture the current working directory */
function cwd() {
    const dir = kuiCwd();
    return dir ? dir.replace(process.env.HOME, '~') : undefined;
}
function isCustomError(response) {
    const err = response;
    return err && typeof err === 'object' && typeof err.name === 'string' && typeof err.message === 'string';
}
export function isError(response) {
    return (response &&
        (response.constructor === Error ||
            response.constructor === UsageError ||
            isXtermErrorResponse(response) ||
            Object.getPrototypeOf(response).constructor === Error ||
            isCustomError(response)));
}
export function isProcessing(block) {
    return block.state === "processing" /* Processing */;
}
export function isActive(block) {
    return block.state === "repl-active" /* Active */;
}
/** @return true if the given `block` is `Active` and differs from the other `oblock` */
export function isActiveAndDifferent(block, oblock) {
    return isActive(block) && (!isActive(oblock) || block.cwd !== oblock.cwd || block.value !== oblock.value);
}
export function isCancelled(block) {
    return block.state === "cancelled" /* Cancelled */;
}
export function isEmpty(block) {
    return block.state === "empty" /* Empty */;
}
export function isOk(block) {
    return block.state === "valid-response" /* ValidResponse */;
}
export function isOops(block) {
    return block.state === "error" /* Error */;
}
export function isFinished(block) {
    return isOops(block) || isCancelled(block) || isOk(block) || isEmpty(block);
}
export function hasCommand(block) {
    return !isActive(block) && (!isEmpty(block) || block.command !== undefined);
}
export function isAnnouncement(block) {
    const blockModel = block;
    return blockModel.state === "valid-response" /* ValidResponse */ && blockModel.isAnnouncement === true;
}
export function hasUUID(block) {
    return block && !isActive(block) && !isEmpty(block) && !isAnnouncement(block);
}
export function hasValue(block) {
    return typeof block.value === 'string';
}
/** Transform to Active */
export function Active(initialValue) {
    return {
        cwd: cwd(),
        state: "repl-active" /* Active */,
        value: initialValue || ''
    };
}
/** Transform to AnnouncementBlock */
export function Announcement(response) {
    return {
        response,
        isAnnouncement: true,
        startTime: Date.now(),
        cwd: cwd(),
        state: "valid-response" /* ValidResponse */
    };
}
export function isRerunable(block) {
    return isOk(block) || isOops(block);
}
export function isBeingRerun(block) {
    return block.isRerun === true;
}
export function hasOriginalUUID(block) {
    return typeof block.originalExecUUID === 'string';
}
export function hasBeenRerun(block) {
    return hasOriginalUUID(block) && hasUUID(block) && block.originalExecUUID !== block.execUUID;
}
/** Transform to Processing */
export function Processing(block, startEvent, isExperimental = false, isReplay = false) {
    return {
        command: startEvent.command,
        isExperimental,
        cwd: block.cwd,
        execUUID: startEvent.execUUID,
        originalExecUUID: (hasOriginalUUID(block) && block.originalExecUUID) || startEvent.execUUID,
        isReplay,
        startEvent,
        startTime: startEvent.startTime,
        state: "processing" /* Processing */
    };
}
/** Transform to Empty */
export function Empty(block, typedSoFar, completeEvent) {
    return {
        cwd: block.cwd,
        command: typedSoFar,
        completeEvent,
        state: "empty" /* Empty */
    };
}
/** Transform to Cancelled */
export function Cancelled(block, typedSoFar, completeEvent) {
    if (isProcessing(block)) {
        return {
            cwd: block.cwd,
            command: block.command,
            execUUID: block.execUUID,
            startTime: block.startTime,
            state: "cancelled" /* Cancelled */
        };
    }
    else {
        return Empty(block, typedSoFar, completeEvent);
    }
}
/** Transform to Finished */
export function Finished(block, event, outputOnly = false, isReplay = false) {
    const response = event.response; // event.responseType === 'ScalarResponse' ? (event.response as ScalarResponse) : true
    const { historyIdx } = event;
    const { startEvent } = block;
    // see Rerun() below; when re-executing a block (which means we
    // re-evaluate a potentially changed command, and want to splice the
    // updated response into an existing block --- in this scenario, we
    // still need a new execUUID so that the view components can know
    // whether or not a re-render is needed; Rerun() which is called
    // onExecStart allocates the execUUID, and then when we get here,
    // onExecEnd, we can swap that new execUUID into place
    if (isBeingRerun(block)) {
        block.execUUID = block.newExecUUID;
    }
    if (event.cancelled) {
        if (!event.command) {
            return Empty(block);
        }
        else {
            return Cancelled(block, event.command);
        }
    }
    else if (isError(response)) {
        return {
            response,
            historyIdx,
            cwd: block.cwd,
            command: block.command,
            startEvent,
            completeEvent: event,
            isExperimental: block.isExperimental,
            isReplay,
            state: "error" /* Error */,
            execUUID: block.execUUID,
            originalExecUUID: (hasOriginalUUID(block) && block.originalExecUUID) || block.execUUID,
            startTime: block.startTime
        };
    }
    else {
        return {
            response,
            historyIdx,
            cwd: block.cwd,
            command: block.command,
            startEvent,
            completeEvent: event,
            isExperimental: block.isExperimental,
            isReplay,
            execUUID: block.execUUID,
            originalExecUUID: (hasOriginalUUID(block) && block.originalExecUUID) || block.execUUID,
            startTime: block.startTime,
            outputOnly,
            state: "valid-response" /* ValidResponse */
        };
    }
}
export function isOutputOnly(block) {
    if (isProcessing(block)) {
        return block.startEvent.echo === false;
    }
    else if (isOk(block)) {
        return ((block.completeEvent && block.completeEvent.echo === false) ||
            (block.startEvent && block.startEvent.echo === false) ||
            block.outputOnly);
    }
}
/** @return whether the block as a startEvent trait */
export function hasStartEvent(block) {
    return !isAnnouncement(block) && (isProcessing(block) || isOk(block) || isOops(block));
}
/** @return whether the block has a completeEvent trait */
export function isWithCompleteEvent(block) {
    return (isOk(block) || isOops(block) || isEmpty(block)) && block.completeEvent !== undefined;
}
/** @return whether the block has pipeStages information; older snapshots may not */
export function hasPipeStages(block) {
    return ((hasStartEvent(block) && block.startEvent.pipeStages !== undefined) ||
        (isWithCompleteEvent(block) && block.completeEvent.pipeStages !== undefined));
}
/** @return whether the block is from replay */
export function isReplay(block) {
    return (isProcessing(block) || isWithCompleteEvent(block)) && block.isReplay;
}
/** Transform a RerunableBlock to one in the BlockBeingRerun state */
export function Rerun(block, newStartEvent = block.startEvent, newCommand = newStartEvent.command, newStartTime = newStartEvent.startTime) {
    return Object.assign(block, {
        isRerun: true,
        startEvent: newStartEvent,
        command: newCommand,
        newExecUUID: uuid(),
        originalExecUUID: (hasOriginalUUID(block) && block.originalExecUUID) || block.execUUID,
        startTime: newStartTime
    });
}
//# sourceMappingURL=BlockModel.js.map