"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.urlFormatterFor = urlFormatterFor;
exports.urlFormatterForArgs = urlFormatterForArgs;

var _pluralize = require("pluralize");

var _util = require("../../../lib/util/util");

var _options = require("../../kubectl/options");

/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = void 0 && (void 0).__awaiter || function (thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};

/** e.g. "apis/apps/v1" for deployments */
function apiOnPathFor(version) {
  return version === 'v1' ? 'api/v1' : `apis/${encodeURIComponent(version)}`;
}
/**
 *  The curl endpoint for accessing kubernetes apiServer needs the resource kind
 *  to be plural and lowercase, so here we transform the user specified kind
 *  to plural and lowercase.
 *  e.g. Pod -> /pods, Ingress -> /ingresses
 *
 */


function kindOnPathFor(_kind) {
  const kind = _kind.toLowerCase();

  return `/${encodeURIComponent((0, _pluralize.isPlural)(kind) ? kind : (0, _pluralize.plural)(kind))}`;
}
/**
 * @return a function that returns a URL string that can be used as a
 * curl endpoint for accessing a kubernetes apiServer
 *
 * @param command e.g. kubectl or oc
 */


function urlFormatterFor(command, namespace, {
  parsedOptions
}, {
  kind,
  version,
  isClusterScoped
}) {
  const kindOnPath = kindOnPathFor(kind); // e.g. "apis/apps/v1" for deployments

  const apiOnPath = apiOnPathFor(version);

  if (!namespace) {
    console.error('Internal oddity with namespace, falling back on "default", and was given:', namespace);
    namespace = 'default';
  } // a bit complex: "kubectl get ns", versus "kubectl get ns foo"
  // the "which" is "foo" in the second case


  const namespaceOnPath = (0, _options.isForAllNamespaces)(parsedOptions) ? '' : kind === 'Namespace' ? '' : isClusterScoped ? '' : `/namespaces/${encodeURIComponent(namespace)}`; // we will accumulate queries

  const queries = []; // labelSelector query

  const label = (0, _options.getLabel)({
    parsedOptions
  });

  if (label) {
    const push = query => queries.push(`labelSelector=${encodeURIComponent(query)}`);

    if (Array.isArray(label)) {
      label.forEach(push);
    } else {
      push(label);
    }
  } // fieldSelector query; chained selectors are comma-separated


  if (parsedOptions['field-selector'] && typeof parsedOptions['field-selector'] === 'string') {
    queries.push(`fieldSelector=${encodeURIComponent(parsedOptions['field-selector'])}`);
  } // limit query


  if (typeof parsedOptions.limit === 'number') {
    queries.push(`limit=${parsedOptions.limit}`);
  } // format a url


  return (includeKind = false, includeQueries = false, name, overrides) => {
    const myApiOnPath = overrides && overrides.version ? apiOnPathFor(overrides.version) : apiOnPath;
    const myKindOnPath = overrides && overrides.kind ? kindOnPathFor(overrides.kind) : kindOnPath;
    const proto = command === 'oc' ? 'openshift' : 'kubernetes';
    return `${proto}:///${myApiOnPath}${namespaceOnPath}${!includeKind ? '' : myKindOnPath}${!name ? '' : `/${encodeURIComponent(name)}`}${!includeQueries || queries.length === 0 ? '' : '?' + queries.join('&')}`;
  };
}

function urlFormatterForArgs(args, explainedKind) {
  return __awaiter(this, void 0, void 0, function* () {
    return urlFormatterFor((0, _util.getCommandFromArgs)(args), yield (0, _options.getNamespace)(args), args, explainedKind);
  });
}