/*
 * Copyright 2019 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { execSync } from 'child_process';
import { readFileSync, writeFileSync } from 'fs';
import { dump, load as parseYAML } from 'js-yaml';
import * as path from 'path';
import * as assert from 'assert';
import { expandHomeDir } from '@kui-shell/core';
import { Common, CLI, ReplExpect, SidecarExpect, Selectors, Util } from '@kui-shell/test';
import { remotePodYaml, waitForGreen, createNS, defaultModeForGet } from '@kui-shell/plugin-kubectl/tests/lib/k8s/utils';
const synonyms = ['kubectl'];
const initialContext = execSync('kubectl config current-context')
    .toString()
    .trim();
var Status;
(function (Status) {
    Status["Offline"] = "red-background";
    Status["Online"] = "green-background";
})(Status || (Status = {}));
// TODO: enable this once proxy can find $HOME on travis
Common.localDescribe('kubectl context switching', function () {
    before(Common.before(this));
    after(Common.after(this, () => {
        console.log(`switching back to initial context ${initialContext}`);
        execSync(`kubectl config use-context ${initialContext}`);
        console.log(`switched to ${execSync('kubectl config current-context')}`);
    }));
    synonyms.forEach(kubectl => {
        /** delete the given namespace */
        const deleteIt = (name, context, kubeconfig) => {
            it(`should delete the namespace ${name} via ${kubectl}`, () => {
                execSync(`kubectl delete namespace ${name} --context ${context} --kubeconfig ${kubeconfig}`);
            });
        };
        /** create the given namespace */
        const createIt = (name) => {
            it(`should create namespace ${name} via ${kubectl}`, () => {
                return CLI.command(`${kubectl} create namespace ${name}`, this.app)
                    .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME(name) }))
                    .then(selector => waitForGreen(this.app, selector))
                    .catch(Common.oops(this, true));
            });
        };
        /** create a pod in the given namespace */
        const createPod = (ns) => {
            it(`should create sample pod in namespace ${ns} from URL via ${kubectl}`, () => {
                return CLI.command(`${kubectl} create -f ${remotePodYaml} -n ${ns}`, this.app)
                    .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME('nginx') }))
                    .then(selector => waitForGreen(this.app, selector))
                    .catch(Common.oops(this, true));
            });
            it(`should show the sample pod in namespace ${ns} in sidecar via ${kubectl}, then close the sidecar`, () => {
                return CLI.command(`${kubectl} get pod nginx -n ${ns} -o yaml`, this.app)
                    .then(ReplExpect.ok)
                    .then(SidecarExpect.open)
                    .then(SidecarExpect.showing('nginx', undefined, undefined, ns))
                    .catch(Common.oops(this, true));
            });
        };
        const defaultFilepath = expandHomeDir('~/.kube/config');
        const getKUBECONFIGFilepath = () => {
            if (process.env.KUBECONFIG) {
                const kconfigEnv = process.env.KUBECONFIG.slice(0);
                return kconfigEnv.split(/:/)[0];
            }
            else {
                return defaultFilepath;
            }
        };
        const getKUBECONFIG = () => {
            if (process.env.KUBECONFIG) {
                return readFileSync(getKUBECONFIGFilepath());
            }
            else {
                return execSync('kubectl config view');
            }
        };
        const addNamespaceToKUBECONFIG = (ns, contextName) => {
            it('should add a new context', () => __awaiter(this, void 0, void 0, function* () {
                try {
                    const kconfig = parseYAML(getKUBECONFIG().toString());
                    const newOnesFilepath = path.join(path.dirname(getKUBECONFIGFilepath()), 'forTesting.yml');
                    kconfig['contexts'][0].context.namespace = ns;
                    kconfig['contexts'][0].name = contextName;
                    writeFileSync(newOnesFilepath, dump(kconfig));
                    yield this.app.client.execute((defaultFilepath, newOnesFilepath) => {
                        process.env.KUBECONFIG = `${process.env.KUBECONFIG || defaultFilepath}:${newOnesFilepath}`;
                    }, defaultFilepath, newOnesFilepath);
                }
                catch (err) {
                    return Common.oops(this, true)(err);
                }
            }));
        };
        /** list contexts and expect the current context */
        const listContextsAndExpectDefault = () => {
            it('should list contexts and show the default context', () => __awaiter(this, void 0, void 0, function* () {
                try {
                    const currentContext = yield CLI.command(`context`, this.app)
                        .then(ReplExpect.okWithCustom({ selector: ' ' }))
                        .then(selector => this.app.client.$(selector))
                        .then(_ => _.getText());
                    const currentContextAsIndicatedByContextsTable = yield CLI.command(`contexts -o wide`, this.app)
                        .then(ReplExpect.okWithCustom({
                        selector: `${Selectors.TABLE_CELL('*', 'NAME')}`
                    }))
                        .then(selector => this.app.client.$(selector))
                        .then(_ => _.getText());
                    assert.strictEqual(currentContextAsIndicatedByContextsTable, currentContext);
                }
                catch (err) {
                    return Common.oops(this, true)(err);
                }
            }));
        };
        /** list contexts and expect the given context */
        const listContextsAndExpectGiven = (contextName) => {
            it(`should list contexts and show the context ${contextName}`, () => __awaiter(this, void 0, void 0, function* () {
                try {
                    const allContextNames = yield CLI.command(`contexts -o wide`, this.app)
                        .then((res) => __awaiter(this, void 0, void 0, function* () {
                        const nameCellSelector = Selectors.BY_KEY('NAME');
                        const selector = yield ReplExpect.okWithCustom({ selector: nameCellSelector })(res);
                        return this.app.client.$$(selector);
                    }))
                        .then(elements => Promise.all(elements.map(_ => _.getText())));
                    assert.ok(allContextNames.find(_ => _ === contextName));
                }
                catch (err) {
                    return Common.oops(this, true)(err);
                }
            }));
        };
        /** list pods and expect an empty list */
        const listPodsAndExpectNone = (ns) => {
            it('should list pods and show nothing', () => {
                return CLI.command(`${kubectl} get pods -n ${ns}`, this.app).then(ReplExpect.error(404));
            });
        };
        /** list pods and expect one entry */
        const listPodsAndExpectOne = (name, ns, kubeconfig = '') => {
            it(`should list pods and show ${name} maybe in namespace ${ns || 'nope'} and kubeconfig ${kubeconfig ||
                'nope'}`, () => {
                return CLI.command(`${kubectl} get pods ${ns ? '-n ' + ns : ''} ${kubeconfig}`, this.app)
                    .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME(name) }))
                    .then(selector => waitForGreen(this.app, selector))
                    .catch(Common.oops(this, true));
            });
        };
        const getPodInSidecar = (name, ns, kubeconfig = '') => {
            it(`should open pod ${name} in sidecar, and maybe in namespace ${ns || 'nope'} and kubeconfig ${kubeconfig ||
                'nope'}`, () => {
                return CLI.command(`${kubectl} get pod ${name} ${ns ? '-n ' + ns : ''} ${kubeconfig} -o yaml`, this.app)
                    .then(ReplExpect.ok)
                    .then(SidecarExpect.open)
                    .then(SidecarExpect.mode(defaultModeForGet))
                    .then(SidecarExpect.yaml({ Name: 'nginx' }))
                    .then(SidecarExpect.showing('nginx', undefined, undefined, ns))
                    .catch(Common.oops(this, true));
            });
        };
        /* switch to the given context */
        const switchToContextByCommand = (contextName) => {
            it(`should switch to the context ${contextName} by command`, () => __awaiter(this, void 0, void 0, function* () {
                try {
                    yield CLI.command(`${kubectl} config use-context ${contextName}`, this.app).then(ReplExpect.ok);
                    // and if we request a new contexts table, it'd better be selected there, too
                    const currentContext = yield CLI.command(`contexts -o wide`, this.app)
                        .then(ReplExpect.okWithCustom({
                        selector: `${Selectors.TABLE_CELL('*', 'NAME')}`
                    }))
                        .then(selector => this.app.client.$(selector))
                        .then(_ => _.getText());
                    assert.strictEqual(contextName, currentContext);
                }
                catch (err) {
                    yield Common.oops(this, true)(err);
                }
            }));
        };
        const showCurrentNamespace = (ns) => {
            it(`should show ${ns} as current namespace`, () => {
                return CLI.command(`namespace current`, this.app)
                    .then(ReplExpect.okWithString(ns))
                    .catch(Common.oops(this, true));
            });
        };
        const createNewTab = () => {
            it('should create a new tab via command', () => CLI.command('tab new', this.app)
                .then(() => this.app.client.$(Selectors.TAB_SELECTED_N(2)))
                .then(_ => _.waitForDisplayed())
                .then(() => CLI.waitForSession(this)) // should have an active repl
                .catch(Common.oops(this, true)));
        };
        const switchToTab1 = () => {
            it(`switch back to first tab via command`, () => CLI.command('tab switch 1', this.app)
                .then(() => this.app.client.$(Selectors.TAB_SELECTED_N(1)))
                .then(_ => _.waitForDisplayed())
                .catch(Common.oops(this, true)));
        };
        const switchToTab2 = () => {
            it(`switch back to the second tab tab via command`, () => CLI.command('tab switch 2', this.app)
                .then(() => this.app.client.$(Selectors.TAB_SELECTED_N(2)))
                .then(_ => _.waitForDisplayed())
                .catch(Common.oops(this, true)));
        };
        const switchNamespaceViaCommand = (ns) => {
            it(`should switch to the namespace ${ns} by command`, () => __awaiter(this, void 0, void 0, function* () {
                try {
                    yield CLI.command(`${kubectl} config set-context --current --namespace=${ns}`, this.app).then(ReplExpect.ok);
                    yield CLI.command(`namespace current`, this.app).then(ReplExpect.okWithString(ns));
                }
                catch (err) {
                    yield Common.oops(this, true)(err);
                }
            }));
        };
        const showCurrentNamespaceViaWidget = (ns) => {
            it(`should show namespace ${ns} in status strip widget`, () => __awaiter(this, void 0, void 0, function* () {
                try {
                    yield this.app.client.waitUntil(() => __awaiter(this, void 0, void 0, function* () {
                        const namespaceText = yield this.app.client
                            .$(Selectors.STATUS_STRIPE_WIDGET('kui--plugin-kubeui--current-namespace'))
                            .then(_ => _.getText());
                        return namespaceText === ns;
                    }), { timeout: CLI.waitTimeout });
                }
                catch (err) {
                    yield Common.oops(this, true)(err);
                }
            }));
        };
        //
        // now start the tests
        //
        Util.closeAllExceptFirstTab.bind(this)();
        const ns = createNS();
        const ns2 = createNS();
        const initialKubeConfig = getKUBECONFIGFilepath();
        listContextsAndExpectDefault();
        createIt(ns);
        addNamespaceToKUBECONFIG(ns, 'holla'); // add holla to the KUBECONFIG contexts
        listContextsAndExpectGiven('holla');
        listPodsAndExpectNone(ns);
        createPod(ns); // create a pod in holla
        listPodsAndExpectOne('nginx', ns);
        listPodsAndExpectOne('nginx', ns, `--kubeconfig ${initialKubeConfig}`);
        getPodInSidecar('nginx', ns, `--kubeconfig ${initialKubeConfig}`);
        switchToContextByCommand('holla');
        listPodsAndExpectOne('nginx');
        // now start tab-state tests
        switchToContextByCommand(initialContext);
        createIt(ns2);
        switchNamespaceViaCommand(ns2);
        listPodsAndExpectNone(ns2);
        let ns2WatcherBadgeInTab1;
        it(`should watch namespace in tab 1 and expect ${ns2} online`, () => CLI.command(`${kubectl} get ns -w`, this.app)
            .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME(ns2) }))
            .then(selector => waitForGreen(this.app, selector))
            .then(selector => {
            ns2WatcherBadgeInTab1 = selector;
        })
            .catch(Common.oops(this, true)));
        createNewTab();
        switchToContextByCommand('holla');
        showCurrentNamespace(ns);
        showCurrentNamespaceViaWidget(ns);
        listPodsAndExpectOne('nginx');
        let nsWatcherBadgeInTab2;
        it(`should watch namespace in tab 2 and expect ${ns} online`, () => CLI.command(`${kubectl} get ns -w`, this.app)
            .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME(ns) }))
            .then(selector => waitForGreen(this.app, selector))
            .then(selector => {
            nsWatcherBadgeInTab2 = selector;
        })
            .catch(Common.oops(this, true)));
        switchToTab1();
        listContextsAndExpectDefault();
        showCurrentNamespace(ns2);
        showCurrentNamespaceViaWidget(ns2);
        listPodsAndExpectNone(ns2);
        switchToTab2();
        showCurrentNamespace(ns);
        showCurrentNamespaceViaWidget(ns);
        listPodsAndExpectOne('nginx');
        deleteIt(ns, initialContext, initialKubeConfig);
        it(`should expect ${ns} to be offline in tab  2`, () => __awaiter(this, void 0, void 0, function* () {
            try {
                const offlineBadge = nsWatcherBadgeInTab2.replace(Status.Online, Status.Offline);
                yield this.app.client.$(offlineBadge).then(_ => _.waitForExist());
            }
            catch (err) {
                return Common.oops(this, true)(err);
            }
        }));
        switchToTab1();
        deleteIt(ns2, initialContext, initialKubeConfig);
        it(`should expect ${ns2} to be offline in tab 1`, () => __awaiter(this, void 0, void 0, function* () {
            try {
                const offlineBadge = ns2WatcherBadgeInTab1.replace(Status.Online, Status.Offline);
                yield this.app.client.$(offlineBadge).then(_ => _.waitForExist());
            }
            catch (err) {
                return Common.oops(this, true)(err);
            }
        }));
    });
});
//# sourceMappingURL=contexts.js.map