/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import React from 'react';
import { v4 as uuid } from 'uuid';
import TurndownService from 'turndown';
import ReactMarkdown from 'react-markdown';
import { dirname, isAbsolute, join, relative } from 'path';
import { List, ListComponent, ListItem } from '@patternfly/react-core';
import { getPrimaryTabId, pexecInCurrentTab } from '@kui-shell/core';
// GitHub Flavored Markdown plugin; see https://github.com/IBM/kui/issues/6563
import gfm from 'remark-gfm';
import Tooltip from '../spi/Tooltip';
import CodeSnippet from '../spi/CodeSnippet';
const SimpleEditor = React.lazy(() => import('./Editor/SimpleEditor'));
export default class Markdown extends React.PureComponent {
    constructor() {
        super(...arguments);
        this._uuid = uuid();
    }
    onCopy(value) {
        navigator.clipboard.writeText(value);
    }
    allContentIsRemote() {
        return typeof this.props.baseUrl === 'string';
    }
    anchorFrom(txt) {
        return `${this._uuid}-${txt}`;
    }
    /** @return markdown source, as string in application/markdown format */
    source() {
        if (this.props.contentType === 'text/html') {
            const { gfm } = require('turndown-plugin-gfm');
            const td = new TurndownService();
            td.use(gfm);
            return td.turndown(this.props.source);
        }
        else {
            return this.props.source;
        }
    }
    handleImage(src, props, key) {
        const isHttp = /^http/i.test(src);
        const isLocal = !isHttp && !this.allContentIsRemote();
        if (isLocal) {
            const absoluteSrc = isAbsolute(src)
                ? src
                : join(this.props.fullpath ? dirname(this.props.fullpath) : this.props.baseUrl || process.cwd(), src);
            src = absoluteSrc;
        }
        else if (!isHttp && this.props.baseUrl) {
            // then this is a relative path against
            src = `${this.props.baseUrl}${!/\/$/.test(this.props.baseUrl) ? '/' : ''}${src}`;
        }
        const style = props ? { float: props.align } : undefined;
        return React.createElement("img", { key: key, src: src, height: props.height, width: props.width, style: style, "data-float": props.align });
    }
    render() {
        if (this.props.onRender) {
            this.props.onRender();
        }
        return (React.createElement(ReactMarkdown, { plugins: [gfm], source: this.source(), "data-is-nested": this.props.nested || undefined, className: this.props.className ||
                'padding-content marked-content page-content' +
                    (!this.props.nested ? ' scrollable scrollable-x scrollable-auto' : ''), renderers: {
                html: props => {
                    if (/<img/.test(props.value)) {
                        const images = props.value.split(/<img/).filter(_ => _);
                        const imageTags = images
                            .map((value, idx) => {
                            const srcMatch = value.match(/src="?([^"\s]+)"?/);
                            const heightMatch = value.match(/height="?(\d+)"?/);
                            const widthMatch = value.match(/width="?(\d+%?)"?/);
                            const alignMatch = value.match(/align="?([^"\s]+)"?/);
                            if (srcMatch) {
                                return this.handleImage(srcMatch[1], {
                                    height: heightMatch && heightMatch[1],
                                    width: widthMatch && widthMatch[1],
                                    align: alignMatch && alignMatch[1]
                                }, idx);
                            }
                        })
                            .filter(_ => _);
                        return React.createElement(React.Fragment, null, imageTags);
                    }
                    // Render the raw string for all other raw html tags
                    return React.createElement("span", null, props.value);
                },
                link: props => {
                    const isKuiCommand = props.href.startsWith('#kuiexec?command=');
                    const isLocal = !/^http/i.test(props.href);
                    const target = !isLocal ? '_blank' : undefined;
                    const onClick = !isLocal && !isKuiCommand
                        ? (evt) => evt.stopPropagation()
                        : (evt) => __awaiter(this, void 0, void 0, function* () {
                            evt.stopPropagation();
                            let file = props.href;
                            if (isKuiCommand) {
                                const raw = props.href.match(/#kuiexec\?command=([^&]+)(&quiet)?/);
                                if (raw) {
                                    const cmdline = decodeURIComponent(raw[1]);
                                    const echo = !raw[2];
                                    if (this.props.repl) {
                                        return this.props.repl.pexec(cmdline, { echo });
                                    }
                                    else {
                                        pexecInCurrentTab(cmdline, undefined, !echo);
                                    }
                                }
                            }
                            else if (props.href.charAt(0) === '#') {
                                const elt = this.props.tab.querySelector(`[data-markdown-anchor="${this.anchorFrom(props.href.slice(1))}"]`);
                                if (elt) {
                                    return elt.scrollIntoView();
                                }
                            }
                            else if (file) {
                                if (this.props.fullpath) {
                                    const absoluteHref = join(dirname(this.props.fullpath), props.href);
                                    const relativeToCWD = relative(process.cwd() || process.env.PWD, absoluteHref);
                                    file = relativeToCWD;
                                }
                                return this.props.repl.pexec(`open ${this.props.repl.encodeComponent(file)}`);
                            }
                        });
                    if (!props.href) {
                        return React.createElement("a", { className: this.props.className }, props.children);
                    }
                    else if (!isLocal && this.props.noExternalLinks) {
                        return React.createElement("span", { className: this.props.className }, props.href);
                    }
                    else {
                        const tip = isKuiCommand
                            ? `### Command Execution\n#### ${decodeURIComponent(props.href.slice(props.href.indexOf('=') + 1))}\n\n\`Link will execute a command\``
                            : `### External Link\n#### ${props.href}\n\n\`Link will open in a separate window\``;
                        return (React.createElement(Tooltip, { markdown: tip },
                            React.createElement("a", Object.assign({}, props, { href: isKuiCommand ? '#' : props.href, target: target, onClick: onClick }))));
                    }
                },
                code: props => {
                    if (this.props.nested) {
                        return (React.createElement("div", { className: "paragraph" },
                            React.createElement(CodeSnippet, { value: props.value, onCopy: this.onCopy.bind(this, props.value) })));
                    }
                    else {
                        return (React.createElement("div", { className: "paragraph" },
                            React.createElement("code", { className: "kui--code--editor" },
                                React.createElement(SimpleEditor, { tabUUID: getPrimaryTabId(this.props.tab), content: props.value, contentType: props.language, fontSize: 12, simple: true, minHeight: 0, readonly: true }))));
                    }
                },
                heading: props => {
                    const valueChild = props.children && props.children.length === 1
                        ? props.children[0]
                        : props.children.find(_ => _.props.value);
                    const anchor = !valueChild || !valueChild.props || !valueChild.props.value
                        ? undefined
                        : this.anchorFrom(valueChild.props.value.toLowerCase().replace(/ /g, '-'));
                    return React.createElement(`h${props.level}`, Object.assign({}, props, {
                        'data-markdown-anchor': anchor,
                        'data-is-href': valueChild && valueChild.props && valueChild.props.href
                    }), props.children);
                },
                image: props => {
                    return this.handleImage(props.src, props) || React.createElement("img", Object.assign({}, props));
                },
                list: props => {
                    return React.createElement(List, { component: props.ordered ? ListComponent.ol : ListComponent.ul }, props.children);
                },
                listItem: props => React.createElement(ListItem, { className: props.className }, props.children),
                table: props => (React.createElement("table", { className: props.className + ' kui--table-like kui--structured-list' }, props.children)),
                tableHead: props => (React.createElement("thead", { className: props.className + ' kui--structured-list-thead' }, props.children)),
                tableBody: props => (React.createElement("tbody", { className: props.className + ' kui--structured-list-tbody' }, props.children)),
                tableRow: props => React.createElement("tr", null, props.children),
                tableCell: props => {
                    if (props.isHeader) {
                        return React.createElement("th", { style: { textAlign: props.align } }, props.children);
                    }
                    else {
                        return React.createElement("td", null, props.children);
                    }
                }
            } }));
    }
}
//# sourceMappingURL=Markdown.js.map