/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import React from 'react';
import { basename } from 'path';
import prettyMillis from 'pretty-ms';
import { flatten, i18n, prettyPrintBytes as safePrettyPrintBytes } from '@kui-shell/core';
import Bar from './Bar';
import ErrorCell from './ErrorCell';
import DefaultColoring from './Coloring';
import trafficLight from './css-for-status';
import { /* renderCell, */ onClickForCell } from './TableCell';
import '../../../../web/scss/components/Table/SequenceDiagram/_index.scss';
const strings = i18n('plugin-client-common');
/** @return pretty-printed bytes */
function safePrettyPrintBytesWithPrefix(bytes, prefix) {
    return prefix + safePrettyPrintBytes(bytes);
}
/** @return bytes per second, formatted to a "N KBps" form */
function prettyPrintThroughput(bytes, durationInMillis) {
    try {
        const numerator = parseInt(bytes, 10); // bytes
        const denominator = durationInMillis / 1000; // seconds
        const ratio = numerator / denominator; // bytes per second
        return safePrettyPrintBytes(ratio) + '/s';
    }
    catch (err) {
        return '';
    }
}
function prettyPrintDuration(duration) {
    try {
        return prettyMillis(duration);
    }
    catch (err) {
        console.error('error formatting duration', duration, err);
        return '';
    }
}
function prettyPrintDateDelta(row, idx1, idx2, numerator) {
    const cell1 = row.attributes[idx1];
    const cell2 = row.attributes[idx2];
    try {
        const startMillis = new Date(cell1.value).getTime();
        if (cell2.value === '<none>') {
            if (cell1.value === '<none>') {
                return '';
            }
            else {
                // then print delta versus now
                const delta = Date.now() - startMillis;
                if (numerator === undefined) {
                    return prettyMillis(delta, { compact: true });
                }
                else {
                    return prettyPrintThroughput(numerator, delta);
                }
            }
        }
        else {
            const endMillis = new Date(cell2.value).getTime();
            const delta = endMillis - startMillis;
            if (numerator === undefined) {
                return prettyMillis(delta);
            }
            else {
                return prettyPrintThroughput(numerator, delta);
            }
        }
    }
    catch (err) {
        console.error('error formatting delta', cell1, cell2, err);
        return '';
    }
}
export default class SequenceDiagram extends React.PureComponent {
    constructor(props) {
        super(props);
        this.state = SequenceDiagram.getDerivedStateFromProps(props);
    }
    componentDidCatch(error, errorInfo) {
        console.error(error, errorInfo);
    }
    static getDerivedStateFromProps(props, state) {
        return Object.assign({ iter: state ? state.iter : 0 }, SequenceDiagram.computeGapModel(props.response, props.denseThreshold));
    }
    componentDidUpdate() {
        if (this.props.isWatching && !this.poller) {
            this.poller = setInterval(() => {
                this.setState(curState => ({ iter: curState.iter + 1 }));
            }, 2000);
        }
        else if (!this.props.isWatching && this.poller) {
            clearInterval(this.poller);
        }
    }
    /** @return numerator/interval formatted */
    getFraction(numerator, interval) {
        if (interval && (!interval.startMillis || !interval.endMillis)) {
            return 0;
        }
        else {
            const fraction = Math.min(1, numerator / (interval ? interval.endMillis - interval.startMillis : this.state.maxIntervalTimeSpan));
            return isNaN(fraction) ? undefined : fraction;
        }
    }
    static sorter(response) {
        if (response.noSort) {
            return () => 0;
        }
        else {
            const idx1 = response.startColumnIdx;
            const idx2 = response.completeColumnIdx;
            return (a, b) => {
                // [SORT]
                const jobNameComparo = a.name.localeCompare(b.name);
                if (jobNameComparo === 0) {
                    // same job name; then compare by startTime
                    const aStartCell = a.attributes[idx1];
                    const bStartCell = b.attributes[idx1];
                    if (!aStartCell || !bStartCell || !aStartCell.value || !bStartCell.value) {
                        return 0;
                    }
                    else {
                        const startDelta = new Date(aStartCell.value).getTime() - new Date(bStartCell.value).getTime();
                        if (startDelta === 0) {
                            const aEndCell = a.attributes[idx2];
                            const bEndCell = b.attributes[idx2];
                            if (!aEndCell || !bEndCell || !aEndCell.value || !bEndCell.value) {
                                return 0;
                            }
                            else {
                                const endDelta = new Date(aEndCell.value).getTime() - new Date(bEndCell.value).getTime();
                                return endDelta;
                            }
                        }
                        else {
                            return startDelta;
                        }
                    }
                }
                else {
                    // different job names
                    return jobNameComparo;
                }
            };
        }
    }
    /**
     * This computes the `State` for this component from the given
     * `response` Table model.
     *
     */
    static computeGapModel(response, denseThreshold = response.colorBy === 'status' ? Number.MAX_SAFE_INTEGER : SequenceDiagram.DEFAULT_DENSE_THRESHOLD) {
        // indices of start and complete attributes
        const idx1 = response.startColumnIdx;
        const idx2 = response.completeColumnIdx;
        // 1) [SLICE]: need to slice so as not to permute the original table model
        // 2) [SORT]: by startTime, then by delta from start of current gap
        // 3) [INTERVALS]: populate a tuple of intervals, where each interval consists of tasks a) from the same job; and b) not too distant in time from the start of the current interval
        const intervals = response.body
            .slice(0) // [SLICE]
            .sort(SequenceDiagram.sorter(response))
            .reduce((intervals, row) => {
            // [INTERVALS]
            const jobName = row.name;
            const startMillis = new Date(row.attributes[idx1].value).getTime();
            const endMillis = !row.attributes[idx2].value || row.attributes[idx2].value === '<none>'
                ? startMillis
                : new Date(row.attributes[idx2].value).getTime();
            // if (isNaN(endMillis)) {
            // }
            if (intervals.length === 0) {
                // case 1: first interval
                return [{ startMillis, endMillis, jobName, rows: [row] }];
            }
            else {
                const currentInterval = intervals[intervals.length - 1];
                const gap = endMillis - currentInterval.startMillis;
                if (gap < denseThreshold && (response.colorBy === 'status' || currentInterval.jobName === jobName)) {
                    // case 2: the gap is small relative to the start of the
                    // current interval and for the same job as the current
                    // interval, in which case we add to current interval
                    currentInterval.endMillis = Math.max(endMillis, currentInterval.endMillis);
                    currentInterval.startMillis = Math.min(startMillis, currentInterval.startMillis);
                    currentInterval.rows.push(row);
                }
                else {
                    // case 3: gap too long, or new job -- create new interval
                    intervals.push({ startMillis, endMillis, jobName, rows: [row] });
                }
                return intervals;
            }
        }, []);
        return {
            maxIntervalTimeSpan: Math.min(denseThreshold, intervals.reduce((max, interval) => Math.max(max, interval.endMillis - interval.startMillis), 0)),
            maxEndMillis: intervals.reduce((max, interval) => Math.max(max, interval.endMillis), 0),
            intervals
        };
    }
    header() {
        return (React.createElement("thead", null,
            React.createElement("tr", null,
                React.createElement("th", { className: "kui--header-cell" }, "Interval"),
                React.createElement("th", { className: "kui--header-cell" }, "Delta"))));
    }
    nSpanCols() {
        return 2; // this.props.response.statusColumnIdx >= 0 ? 4 : 3
    }
    overheads(interval) {
        const idx1 = this.props.response.startColumnIdx;
        const idx2 = this.props.response.completeColumnIdx;
        const idx3 = this.props.response.coldStartColumnIdx;
        const { coldStarts, gaps, denominator } = interval.rows.reduce((sums, row) => {
            const startMillisAttr = row.attributes[idx1];
            const endMillisAttr = row.attributes[idx2];
            const coldAttr = idx3 ? row.attributes[idx3] : undefined;
            let thisStartMillis;
            const { previousEndMillis } = sums;
            if (endMillisAttr) {
                thisStartMillis = new Date(startMillisAttr.value).getTime();
                const thisEndMillis = new Date(endMillisAttr.value).getTime();
                sums.denominator += thisEndMillis - thisStartMillis;
                sums.previousEndMillis = thisEndMillis;
            }
            if (coldAttr) {
                sums.coldStarts += parseInt(coldAttr.value, 10);
            }
            if (previousEndMillis > 0) {
                const gap = thisStartMillis - previousEndMillis;
                if (gap > 0) {
                    sums.gaps += gap;
                }
            }
            return sums;
        }, { coldStarts: 0, gaps: 0, denominator: 0, previousEndMillis: 0 });
        return { coldStartFraction: coldStarts / denominator, gapFraction: gaps / denominator };
    }
    gapRow(startMillis, intervalIdx, onClick) {
        const interval = this.state.intervals[intervalIdx];
        const endMillis = interval.endMillis;
        const hasStart = startMillis !== undefined && !isNaN(startMillis);
        const overheads = this.overheads(interval);
        const gap = [
            React.createElement("tr", { key: `gaprowB-${intervalIdx}`, className: "kui--interval-start" },
                React.createElement("td", { className: "kui--gap-cell", colSpan: this.nSpanCols() },
                    React.createElement("span", { className: "flex-layout" },
                        !hasStart ? '' : new Date(startMillis).toLocaleString(),
                        hasStart && endMillis && (React.createElement("span", { className: "flex-fill flex-align-end left-pad" },
                            endMillis - startMillis === 0 ? '' : `${prettyPrintDuration(endMillis - startMillis)}`,
                            overheads.coldStartFraction > 0 || overheads.gapFraction > 0 ? ' (' : '',
                            overheads.coldStartFraction > 0
                                ? `${Math.round(100 * overheads.coldStartFraction).toFixed(0)}% cold starts`
                                : '',
                            overheads.gapFraction > 0
                                ? `${overheads.coldStartFraction > 0 ? ', ' : ''}${Math.round(100 * overheads.gapFraction).toFixed(0)}% scheduling gaps`
                                : '',
                            overheads.coldStartFraction > 0 || overheads.gapFraction > 0 ? ')' : ''))))),
            React.createElement("tr", { key: `gaprowB-${intervalIdx}-jobName`, className: "kui--interval-start-jobName" },
                React.createElement("td", { className: ['kui--gap-cell', 'kui--table-cell-is-name', onClick ? 'clickable' : ''].join(' '), colSpan: this.nSpanCols(), onClick: onClick }, interval.jobName))
        ];
        return this.blankRow(intervalIdx).concat(gap);
    }
    blankRow(intervalIdx) {
        return intervalIdx === 0
            ? []
            : [
                React.createElement("tr", { key: `gaprowA-${intervalIdx}`, className: "kui--interval-blank" },
                    React.createElement("td", { colSpan: this.nSpanCols() }))
            ];
    }
    colorByStatus() {
        return this.props.response.colorBy === 'status' && this.props.response.statusColumnIdx >= 0;
    }
    findAttrIdx(key) {
        if (this.props.response.body.length >= 0) {
            const rowWithAttr = this.props.response.body.find(row => row.attributes ? row.attributes.find(_ => _.key === key) : false);
            if (rowWithAttr) {
                return rowWithAttr.attributes.findIndex(_ => _.key === key);
            }
        }
        return -1;
    }
    rows() {
        const idx1 = this.props.response.startColumnIdx;
        const idx2 = this.props.response.completeColumnIdx;
        const idx4 = this.props.response.statusColumnIdx;
        const idx5 = this.findAttrIdx('InputFile');
        const idx6 = this.findAttrIdx('InputFileSize');
        const idx7 = this.findAttrIdx('Message');
        const colorByStatus = this.colorByStatus();
        const durationColoring = new DefaultColoring(this.props.response);
        const durationColor = durationColoring.durationCss.bind(durationColoring);
        return flatten(this.state.intervals.map((interval, intervalIdx) => flatten(interval.rows.map((row, rowIdx) => {
            const startDate = new Date(row.attributes[idx1].value);
            const startMillis = startDate.getTime();
            const endMillis = !row.attributes[idx2].value
                ? this.state.maxEndMillis || startMillis
                : new Date(row.attributes[idx2].value).getTime();
            const durationCol = this.props.response.durationColumnIdx >= 0 && row.attributes[this.props.response.durationColumnIdx]
                ? parseInt(row.attributes[this.props.response.durationColumnIdx].value, 10)
                : undefined;
            const duration = durationCol || (!endMillis ? 0 : endMillis - startMillis);
            const left = this.getFraction(startMillis - interval.startMillis, interval);
            let width = !duration ? undefined : this.getFraction(duration, interval);
            if (left + width > 1) {
                /* console.error(
                  'oops',
                  left,
                  width,
                  duration,
                  startMillis - interval.startMillis,
                  interval.endMillis - interval.startMillis,
                  interval.endMillis - duration - interval.startMillis,
                  interval,
                  this.state.maxEndMillis
                ) */
                width = 1 - left;
            }
            const coldStart = this.props.response.coldStartColumnIdx >= 0 && row.attributes[this.props.response.coldStartColumnIdx]
                ? parseInt(row.attributes[this.props.response.coldStartColumnIdx].value, 10)
                : undefined;
            let widthB = coldStart ? this.getFraction(coldStart, interval) : undefined;
            const title = strings('Duration', prettyPrintDuration(duration));
            const titleB = coldStart ? strings('Cold Start', prettyPrintDuration(coldStart), title) : undefined;
            const className = colorByStatus
                ? '' /* trafficLight(row.attributes[idx4]) */
                : durationColor(duration, false);
            if (left + widthB > 1) {
                /* console.error(
                  'oopsB',
                  left,
                  widthB,
                  startMillis - interval.startMillis,
                  interval,
                  interval.endMillis - interval.startMillis,
                  interval
                  ) */
                widthB = 1 - left;
            }
            // time gap since the run
            const gap = startMillis - interval.startMillis;
            const gapText = (intervalIdx === 0 && rowIdx === 0) || gap === 0 || isNaN(gap)
                ? '' // very first row or unfinished task (NaN gap)
                : (gap >= 0 ? '+' : '') + prettyPrintDuration(gap);
            // drilldown to underlying resource, e.g. Pod for Kubernetes Jobs
            const onClick = onClickForCell(row, this.props.tab, this.props.repl, row.attributes[0], this.props.response);
            // rows that help to define the contents of the interval; e.g. jobName
            const interGroupGapRow = rowIdx === 0 && !colorByStatus ? this.gapRow(startMillis, intervalIdx, onClick) : [];
            // does this row represent scheduling overhead?
            const isOverheadRow = /Overhead/i.test(row.attributes[idx4].value);
            const color = trafficLight(row.attributes[idx4]);
            return interGroupGapRow.concat([
                React.createElement("tr", { key: `${intervalIdx}-${rowIdx}`, className: 'kui--sequence-diagram-data-row' +
                        (rowIdx === interval.rows.length - 1 ? ' kui--sequence-diagram-last-row-in-interval' : '') },
                    idx5 >= 0 && (React.createElement("td", { className: "kui--tertiary-text pf-m-fit-content text-right badge-width" },
                        React.createElement("span", { title: (row.attributes[idx5] ? basename(row.attributes[idx5].value) : '') +
                                (idx6 < 0 || !row.attributes[idx6]
                                    ? ''
                                    : safePrettyPrintBytesWithPrefix(row.attributes[idx6].value, ': ')), className: "cell-inner" }, row.attributes[idx5] ? basename(row.attributes[idx5].value) : ''))),
                    React.createElement("td", { className: "kui--sequence-diagram-bar-cell pf-m-nowrap", onClick: onClick },
                        React.createElement(Bar, { left: left, width: width, widthOverlay: widthB, className: className, onClick: onClick, title: title, titleOverlay: titleB })),
                    React.createElement("td", { className: "kui--tertiary-text pf-m-fit-content" }, gapText),
                    colorByStatus && (React.createElement("td", { className: "kui--secondary-text pf-m-fit-content" },
                        React.createElement("div", { "data-tag": "badge", className: "cell-inner" },
                            React.createElement("span", { "data-tag": "badge-circle", className: color }, /red-background/.test(color) ? React.createElement(ErrorCell, null) : undefined),
                            React.createElement("span", { className: 'kui--cell-inner-text' }, !isOverheadRow && row.attributes[idx4].value)))),
                    colorByStatus && idx7 < 0 && (React.createElement("td", { className: "kui--tertiary-text pf-m-fit-content" },
                        React.createElement("span", { className: "cell-inner" }, !isOverheadRow && prettyPrintDateDelta(row, idx1, idx2)))),
                    idx6 >= 0 && idx7 < 0 && (React.createElement("td", { className: "kui--tertiary-text pf-m-fit-content text-right monospace" },
                        React.createElement("span", { className: "cell-inner", title: row.attributes[idx6] && safePrettyPrintBytes(row.attributes[idx6].value) }, row.attributes[idx6] ? prettyPrintDateDelta(row, idx1, idx2, row.attributes[idx6].value) : ''))),
                    idx7 >= 0 && (React.createElement("td", { className: "kui--tertiary-text pf-m-fit-content text-right monospace" },
                        React.createElement("span", { className: "cell-inner", title: row.attributes[idx7] ? row.attributes[idx7].value : '' }, row.attributes[idx7] ? row.attributes[idx7].value : ''))))
            ]);
        }))));
    }
    /** To help with adjusting row height, a data-size attribute */
    size() {
        const nRows = this.props.response.body.length;
        return nRows <= 10 ? 'small' : nRows <= 40 ? 'medium' : nRows <= 200 ? 'large' : 'huge';
    }
    render() {
        if (!this.state) {
            return React.createElement(React.Fragment, null);
        }
        return (React.createElement("div", { className: "kui--data-table-container kui--data-table-container" },
            React.createElement("table", { className: "kui--table-like-wrapper pf-c-table pf-m-compact kui--sequence-diagram", "data-size": this.size(), "data-color-by": "duration" },
                React.createElement("tbody", null, this.rows()))));
    }
}
/**
 * @see Props.denseThreshold
 *
 */
SequenceDiagram.DEFAULT_DENSE_THRESHOLD = 120 * 1000;
//# sourceMappingURL=SequenceDiagram.js.map