"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _core = require("@kui-shell/core");

var _usageHelpers = require("./usage-helpers");

var _vfs = require("../vfs");

/*
 * Copyright 2017 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = void 0 && (void 0).__awaiter || function (thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};
/**
 * This plugin introduces commands that dispatch to a local bash-like
 * shell
 *
 */


const strings = (0, _core.i18n)('plugin-bash-like');
const usage = {
  cd: {
    strict: 'cd',
    command: 'cd',
    title: strings('cdUsageTitle'),
    header: strings('cdUsageHeader'),
    optional: _usageHelpers.localFilepath
  }
};
/**
 * cd command
 *
 */

const cd = args => __awaiter(void 0, void 0, void 0, function* () {
  const dirAsProvided = args.REPL.split(args.command, true, true)[1] || '';

  if (dirAsProvided === '-' && !process.env.OLDPWD) {
    throw new Error(`cd: not a directory: ${dirAsProvided}`);
  }

  const dir = !dirAsProvided ? (0, _core.expandHomeDir)('~') : dirAsProvided === '-' ? process.env.OLDPWD : dirAsProvided;
  const mount = yield (0, _vfs.findMount)(dir, undefined, true);

  try {
    const {
      isDirectory,
      fullpath
    } = !mount ? {
      isDirectory: true,
      fullpath: (0, _vfs.absolute)(dir)
    } : yield mount.fstat(args, (0, _vfs.absolute)(dir));
    const isLocal = mount && mount.isLocal;

    if (isDirectory) {
      if (process.env.OLDPWD === undefined) {
        process.env.OLDPWD = '';
      }

      const OLDPWD = (0, _core.cwd)(); // remember it for when we're done

      const newDir = (0, _core.expandHomeDir)(fullpath);

      if (isLocal && !(0, _core.inBrowser)()) {
        process.chdir(newDir);
      }

      process.env.OLDPWD = OLDPWD;
      process.env.PWD = newDir;

      if (isLocal) {
        delete process.env.VIRTUAL_CWD;
      } else {
        process.env.VIRTUAL_CWD = newDir;
      }

      if (args.tab.state) {
        args.tab.state.capture();
      }

      return newDir;
    } else {
      throw new Error(`cd: not a directory: ${dirAsProvided}`);
    }
  } catch (err) {
    if (err.message && err.message.includes('no such file or directory')) {
      throw new Error(`cd: no such file or directory: ${dirAsProvided}`);
    } else {
      throw err;
    }
  }
});

const bcd = ({
  command,
  execOptions,
  REPL
}) => __awaiter(void 0, void 0, void 0, function* () {
  const pwd = yield REPL.qexec(command.replace(/^cd/, 'kuicd'), undefined, undefined, execOptions);
  process.env.PWD = pwd;
  return pwd;
});
/**
 * Register command handlers
 *
 */


var _default = commandTree => {
  commandTree.listen('/kuicd', cd, {
    requiresLocal: true
  });

  if (!(0, _core.inBrowser)()) {
    commandTree.listen('/cd', cd, {
      usage: usage.cd
    });
  } else {
    commandTree.listen('/cd', bcd, {
      usage: usage.cd
    });
  }
};

exports.default = _default;