"use strict";

var _carbonComponents = require("carbon-components");

/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
var prefix = _carbonComponents.settings.prefix;
describe('Tab', function () {
  describe('renders as expected', function () {
    var React;
    var enzyme;
    var mount;
    var Tab;
    var wrapper;
    beforeEach(function () {
      jest.isolateModules(function () {
        React = require('react');
        enzyme = require('enzyme');
        mount = enzyme.mount;
        Tab = require('../Tab').default;

        var Adapter = require('enzyme-adapter-react-16');

        enzyme.configure({
          adapter: new Adapter()
        });
        wrapper = mount( /*#__PURE__*/React.createElement(Tab, {
          label: "firstTab"
        }));
      });
    });
    it('adds extra classes that are passed via className', function () {
      wrapper.setProps({
        className: 'extra-class'
      });
      expect(wrapper.hasClass('extra-class')).toBe(true);
    });
    it('renders <button> with expected className', function () {
      expect( // TODO: uncomment and replace assertion in next major version
      // wrapper.find('button').hasClass(`${prefix}--tabs__nav-link`)
      wrapper.find('button').hasClass("".concat(prefix, "--tabs--scrollable__nav-link"))).toBe(true);
    });
    it('renders <li> with [role="presentation"]', function () {
      expect(wrapper.props().role).toEqual('presentation');
    });
    it('renders <button> with tabindex set to 0', function () {
      expect(wrapper.find('button').props().tabIndex).toEqual(0);
    });
    it('uses label to set children on <button> when passed via props', function () {
      expect(wrapper.find('button').props().children).toEqual('firstTab');
    });
    it("should not have [className=\"".concat(prefix, "--tabs__nav-item--selected\"] by default"), function () {
      expect(wrapper.hasClass("".concat(prefix, "--tabs__nav-item--selected"))).toBe(false);
    });
    it('has aria-disabled that matches disabled', function () {
      var getDisabledRegion = function getDisabledRegion() {
        return wrapper.find('[aria-disabled]');
      };

      expect(getDisabledRegion().length).toEqual(0);
      wrapper.setProps({
        disabled: true
      });
      expect(getDisabledRegion().prop('aria-disabled')).toEqual(true);
    });
    it("adds [className=\"".concat(prefix, "--tabs__nav-item--selected\"] when selected prop is true"), function () {
      wrapper.setProps({
        selected: true
      });
      expect(wrapper.children().hasClass("".concat(prefix, "--tabs__nav-item--selected"))).toBe(true);
    });
  });
  describe('events', function () {
    var React;
    var enzyme;
    var mount;
    var Tab;
    beforeEach(function () {
      jest.isolateModules(function () {
        React = require('react');
        enzyme = require('enzyme');
        mount = enzyme.mount;
        Tab = require('../Tab').default;

        var Adapter = require('enzyme-adapter-react-16');

        enzyme.configure({
          adapter: new Adapter()
        });
      });
    });
    describe('click', function () {
      var wrapper;
      var onClick;
      var handleTabClick;
      beforeEach(function () {
        wrapper = mount( /*#__PURE__*/React.createElement(Tab, {
          label: "firstTab"
        }));
        onClick = jest.fn();
        handleTabClick = jest.fn();
      });
      it('invokes handleTabClick from onClick prop', function () {
        wrapper.setProps({
          handleTabClick: handleTabClick
        });
        wrapper.simulate('click');
        expect(handleTabClick).toHaveBeenCalled();
      });
      it('invokes onClick when a function is passed to onClick prop', function () {
        wrapper.setProps({
          onClick: onClick
        });
        wrapper.simulate('click');
        expect(onClick).toHaveBeenCalled();
      });
    });
    describe('keydown', function () {
      var onKeyDown;
      var handleTabKeyDown;
      var wrapper;
      beforeEach(function () {
        onKeyDown = jest.fn();
        handleTabKeyDown = jest.fn();
        wrapper = mount( /*#__PURE__*/React.createElement(Tab, {
          label: "firstTab",
          onKeyDown: onKeyDown,
          handleTabKeyDown: handleTabKeyDown
        }));
      });
      it('invokes onKeyDown when a function is passed to onKeyDown prop', function () {
        wrapper.simulate('keyDown', {
          which: 38
        });
        expect(onKeyDown).toHaveBeenCalled();
      });
      it('invokes handleTabAnchorFocus when onKeyDown occurs for appropriate events', function () {
        wrapper.simulate('keyDown', {
          which: 37
        });
        expect(onKeyDown).toHaveBeenCalled();
      });
    });
  });
  describe('deprecated', function () {
    var React;
    var enzyme;
    var mount;
    var Tab;
    beforeEach(function () {
      jest.isolateModules(function () {
        jest.mock('../../prop-types/deprecate', function () {
          return jest.fn().mockImplementation(function () {
            return jest.fn();
          });
        });
        React = require('react');
        enzyme = require('enzyme');
        mount = enzyme.mount;
        Tab = require('../Tab').default;

        var Adapter = require('enzyme-adapter-react-16');

        enzyme.configure({
          adapter: new Adapter()
        });
      });
    });
    test('custom render label', function () {
      var wrapper = mount( /*#__PURE__*/React.createElement(Tab, {
        renderAnchor: function renderAnchor() {
          return /*#__PURE__*/React.createElement("a", {
            id: "custom-label",
            href: "#other-content"
          }, "Content");
        }
      }));
      expect(wrapper.find('#custom-label').props().href).toEqual('#other-content');
    });
    it('sets tabIndex on <button> if one is passed via props', function () {
      // eslint-disable-next-line jsx-a11y/tabindex-no-positive
      var wrapper = mount( /*#__PURE__*/React.createElement(Tab, {
        label: "firstTab",
        tabIndex: 2
      }));
      expect(wrapper.find('button').props().tabIndex).toEqual(2);
    });
  });
});