/*
 * Copyright 2017 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import Debug from 'debug';
import { join } from 'path';
import store from '../models/store';
import expandHomeDir from '../util/home';
import { inBrowser } from '../core/capabilities';
const debug = Debug('core/userdata');
/**
 * Get the userdata directory
 *
 */
export const userDataDir = () => {
    if (inBrowser()) {
        throw new Error('Unsupported operation');
    }
    else {
        // headless
        const name = '@kui-shell/settings';
        switch (process.platform) {
            case 'darwin':
                return join(process.env.HOME, 'Library', 'Application Support', name);
            case 'linux': {
                const home = process.env.XDG_CONFIG_HOME || expandHomeDir('~/.config');
                return join(home, name);
            }
            case 'win32':
                return join(process.env.APPDATA, name);
        }
    }
};
/** filepath to persisted preference model */
const preferencesFilepath = () => join(userDataDir(), 'prefs.json');
/**
 * Sync the preferences to disk
 *
 * @return passes through the preference model
 *
 */
const fsyncPreferences = (prefs) => __awaiter(void 0, void 0, void 0, function* () {
    if (inBrowser()) {
        store().setItem('kui.userprefs', JSON.stringify(prefs));
    }
    else {
        const { mkdirp, writeFile } = yield import('fs-extra');
        yield mkdirp(userDataDir());
        yield writeFile(preferencesFilepath(), JSON.stringify(prefs));
    }
    return prefs;
});
function readFile(filepath) {
    return __awaiter(this, void 0, void 0, function* () {
        const { readFile } = yield import('fs');
        return new Promise((resolve, reject) => {
            readFile(filepath, (err, data) => {
                if (err) {
                    reject(err);
                }
                else {
                    resolve(data.toString());
                }
            });
        });
    });
}
/**
 * Read the preference model
 *
 */
const preferences = () => __awaiter(void 0, void 0, void 0, function* () {
    if (inBrowser()) {
        debug('reading preferences from browser localStorage');
        const prefs = store().getItem('kui.userprefs');
        if (!prefs) {
            return {};
        }
        else {
            try {
                return JSON.parse(prefs);
            }
            catch (err) {
                debug('error parsing preference model', prefs);
                console.error('error parsing preference model', err);
                return {};
            }
        }
    }
    try {
        const filepath = preferencesFilepath();
        debug('reading persisted preference model', filepath);
        const raw = yield readFile(filepath);
        try {
            return JSON.parse(raw);
        }
        catch (err) {
            debug('error parsing preference model', raw);
            console.error('error parsing preference model', err);
            return {};
        }
    }
    catch (err) {
        if (err.code === 'ENOENT') {
            return fsyncPreferences({});
        }
        else {
            throw err;
        }
    }
});
/**
 * Purge the preference model
 *
 */
export const purgePreferences = () => __awaiter(void 0, void 0, void 0, function* () {
    debug('purgePreferences');
    if (inBrowser()) {
        store().removeItem('kui.userprefs');
    }
    else {
        const { unlink } = yield import('fs-extra');
        yield unlink(preferencesFilepath());
    }
});
/**
 * Remove the preference associated with the given key
 *
 * @return the prior value
 *
 */
export const clearPreference = (key) => __awaiter(void 0, void 0, void 0, function* () {
    debug('clearPreference', key);
    const prefs = yield preferences();
    const value = prefs[key];
    delete prefs[key];
    yield fsyncPreferences(prefs);
    return value;
});
/**
 * Get a persisted preference
 *
 * @return the preference value
 *
 */
export const getPreference = (key) => __awaiter(void 0, void 0, void 0, function* () {
    const prefs = yield preferences();
    const value = prefs[key];
    debug('getPreference', key, value);
    return value;
});
/**
 * Set a persisted preference
 *
 * @return the preference value
 *
 */
export const setPreference = (key, value) => __awaiter(void 0, void 0, void 0, function* () {
    debug('setPreference', key, value);
    const prefs = yield preferences();
    prefs[key] = value;
    yield fsyncPreferences(prefs);
    return value;
});
//# sourceMappingURL=userdata.js.map