"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.showingPrevious = showingPrevious;
exports.content = content;
exports.Logs = void 0;

var _debug = _interopRequireDefault(require("debug"));

var _react = _interopRequireDefault(require("react"));

var _core = require("@kui-shell/core");

var _pluginClientCommon = require("@kui-shell/plugin-client-common");

var _util = require("../../util/util");

var _ExecIntoPod = require("./ExecIntoPod");

var _options = require("../../../controller/kubectl/options");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = void 0 && (void 0).__awaiter || function (thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};

const debug = (0, _debug.default)('plugin-kubectl/Logs');
const strings = (0, _core.i18n)('plugin-kubectl', 'logs');
/**
 * Default --tail flag, if the user does not specify one. See
 * https://github.com/IBM/kui/issues/4810
 *
 */

const defaultTail = 1000;

function showingPrevious(args) {
  return args && (!!args.parsedOptions.p || !!args.parsedOptions.previous);
}

class Logs extends _ExecIntoPod.Terminal {
  constructor(props) {
    super(props);
    this.loglines = [];
    this._onFilterChange = this.onFilterChange.bind(this);
    this.state = Object.assign(this.state, {
      nLines: 0,
      filter: undefined,
      showingPrevious: showingPrevious(this.props.args.argsForMode),
      container: this.defaultContainer()
    });
  }

  supportsAllContainers() {
    return true;
  }

  mode() {
    return 'logs';
  }
  /** Which container should we focus on by default? */


  defaultContainer() {
    if (this.props.args.argsForMode) {
      const container = (0, _options.getContainer)(this.props.args.argsForMode, 'logs');

      if (container) {
        // TODO MAYBE? validate container name?
        return container;
      }
    } // undefined means all containers


    return this.props.pod && this.props.pod.spec.containers && this.props.pod.spec.containers.length === 1 ? this.props.pod.spec.containers[0].name : undefined;
  }
  /** Text to display in the Toolbar. */


  toolbarText(status) {
    // i18n message key for toolbar text
    const msgAndType = {
      Live: {
        message: 'Logs are live streaming.',
        type: 'info'
      },
      Idle: {
        message: 'Log streaming is idle.',
        type: 'warning'
      },
      Paused: {
        message: 'Log streaming is paused.',
        type: 'warning'
      },
      Stopped: {
        message: 'Log streaming stopped.',
        type: 'warning'
      },
      Error: {
        message: 'Log streaming stopped abnormally.',
        type: 'error'
      }
    };

    if (!msgAndType) {
      return;
    } else if (!msgAndType[status]) {
      console.error(`Unknown streaming status: ${status}`);
      return;
    }

    const msg1 = msgAndType[status].message;
    const msg2 = this.previousMessage(this.state.isTerminated ? msg1 : `${msg1} ${this.state.container ? 'Showing container X.' : 'Showing all containers.'}`);
    return {
      type: msgAndType[status].type,
      text: this.state.container ? strings(msg2, this.state.container) : strings(msg2)
    };
  }
  /** Addendum to toolbar text to denote whether we are showingPrevious */


  previousMessage(baseMsg) {
    return this.state.showingPrevious ? `${baseMsg} Showing previous instance.` : baseMsg;
  }
  /**
   *
   * @return whether we are showing logs for multiple containers,
   * e.g. via a label selector
   *
   */


  isMulti() {
    return !!(this.props.args.argsForMode && (0, _options.hasLabel)(this.props.args.argsForMode));
  }
  /**
   *
   * @return the command to issue in order to initialize the pty stream
   *
   */


  ptyCommand() {
    const {
      args,
      pod
    } = this.props;
    const {
      container: containerName
    } = this.state;
    const container = containerName ? `-c ${containerName}` : '--all-containers';
    const isMulti = this.isMulti();

    if (args.argsForMode && args.argsForMode.command && (!isMulti || !containerName)) {
      // 1) if the user specified no container, we will inject
      // --all-containers for convenience
      // 2) only use argsForMode once
      // 3) do not add -f unless the user requested it
      const previous = showingPrevious(args.argsForMode) ? '--previous' : '';
      const tail = !args.argsForMode.parsedOptions.tail ? ` --tail ${defaultTail}` : '';
      const command = `${args.argsForMode.command} ${!containerName ? container : ''} ${tail} ${previous}`;

      if (!isMulti) {
        args.argsForMode.command = undefined; // point 2
      }

      return {
        command,
        isLive: args.parsedOptions.f ? 'Live' : 'Paused'
      };
    } else {
      // pod:container? a sign of a multi-pod view
      const previous = this.state.showingPrevious || args.argsForMode && showingPrevious(args.argsForMode) ? '--previous' : '';
      const dashF = !isMulti || args.argsForMode && args.argsForMode.parsedOptions.f ? '-f' : '';
      const isLive = dashF ? 'Live' : undefined;

      if (!containerName && args.argsForMode && (0, _options.hasLabel)(args.argsForMode)) {
        // all container... re-execute label-selector
        return {
          command: args.argsForMode.command,
          isLive
        };
      }

      const split = isMulti && containerName && containerName.split(/:/);
      const possibleMulti = split && split.length === 2 && split;
      const podName = possibleMulti ? possibleMulti[0] : pod.spec._podName || pod.metadata.name;
      const theContainer = possibleMulti ? `-c ${possibleMulti[1]}` : container;
      const command = (0, _options.withKubeconfigFrom)(args, `${(0, _util.getCommandFromArgs)(args)} logs ${podName} -n ${pod.metadata.namespace} ${theContainer} ${dashF} ${previous} --tail ${defaultTail}`);
      debug('log command', command);
      return {
        isLive,
        command
      };
    }
  }

  toolbarButtonsForStreaming(status) {
    if (status === 'Live' || status === 'Paused') {
      const isLive = status === 'Live';
      return [{
        mode: 'toggle-streaming',
        label: isLive ? strings('Pause Streaming') : strings('Resume Streaming'),
        kind: 'view',
        icon: _react.default.createElement(_pluginClientCommon.Icons, {
          icon: isLive ? 'Pause' : 'Play'
        }),
        command: this.toggleStreaming.bind(this, !isLive)
      }];
    } else {
      return [];
    }
  }
  /** Previous logs button */


  previous() {
    return {
      mode: 'kubectl-logs-previous-toggle',
      label: this.state.showingPrevious ? strings('Show Current') : strings('Show Previous'),
      icon: _react.default.createElement(_pluginClientCommon.Icons, {
        icon: this.state.showingPrevious ? 'NextPage' : 'PreviousPage'
      }),
      kind: 'view',
      command: () => this.showContainer(undefined, curState => ({
        showingPrevious: !curState.showingPrevious
      }))
    };
  }

  toolbarButtons(status) {
    return [this.previous()].concat(super.toolbarButtons(status));
  }
  /** The part of toggleStreaming that deals with PTY flow control. */


  doFlowControl(desiredStateIsLive) {
    if (this.state.job) {
      if (desiredStateIsLive) {
        this.state.job.xon();
      } else {
        this.state.job.xoff();
      }
    }
  }
  /** Handler for Pause/Play. */


  toggleStreaming(desiredState) {
    const desiredStatus = desiredState ? 'Live' : 'Paused';

    if (this.state.isLive !== desiredStatus) {
      this.doFlowControl(desiredState);
      this.updateToolbar(desiredState ? 'Live' : 'Paused');
      this.setState(curState => {
        if (curState.isLive !== desiredStatus) {
          return {
            isLive: desiredStatus
          };
        }
      });
    }
  }
  /** Should we wait a bit before proclaiming we have no data? */


  needsHysteresis() {
    return true;
  }
  /** Render the log content in the case we no logs to show. */


  nothingToShow() {
    return _react.default.createElement("div", {
      className: "kui--sidecar-text-content kui--center-absolute"
    }, _react.default.createElement("div", {
      className: "kui--hero-text"
    }, strings('No log data')));
  }

  write(line) {
    this.loglines.push(line);

    if (!this.state.filter && this.state.xterm) {
      this.state.xterm.write(line);
    } else {
      if (this.delay) clearTimeout(this.delay);
      this.delay = setTimeout(() => this.setState({
        nLines: this.loglines.length
      }), 20);
    }
  }

  refill(filter) {
    this.setState(curState => {
      curState.xterm.clear();
      const pattern = new RegExp(filter, /^[a-z0-9]+$/.test(filter) ? 'i' : undefined);
      this.loglines.join().split(/\n/).forEach(_ => {
        if (pattern.test(_)) {
          curState.xterm.writeln(_);
        }
      });
      return {
        filter
      };
    });
  }

  onFilterChange(evt) {
    if (this.delay2) clearTimeout(this.delay2);
    const filter = evt.currentTarget.value;
    this.delay2 = setTimeout(() => this.refill(filter), 20);
  }

  filterPane() {
    return _react.default.createElement("div", {
      className: "flex-layout kui--sidecar-filter-area"
    }, _react.default.createElement("input", {
      className: "flex-fill kui--sidecar-filter-input",
      placeholder: "Enter filter string or regular expression",
      onChange: this._onFilterChange
    }));
  }

  render() {
    if (this.notReady()) {
      return super.render();
    } else {
      return _react.default.createElement(_react.default.Fragment, null, _react.default.createElement("div", {
        className: "flex-fill"
      }, super.render()), this.filterPane());
    }
  }

}
/**
 * The content renderer for the summary tab
 *
 */


exports.Logs = Logs;

function content(tab, pod, args) {
  return __awaiter(this, void 0, void 0, function* () {
    return {
      react: function LogsProvider(toolbarController) {
        return _react.default.createElement(Logs, {
          tab: tab,
          pod: pod,
          args: args,
          toolbarController: toolbarController
        });
      }
    };
  });
}