/*
 * Copyright 2019 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { expandHomeDir } from '@kui-shell/core';
import { getCurrentDefaultNamespace } from './contexts';
import { getCommandFromArgs } from '../../lib/util/util';
/** @return the -f or --filename option */
export function fileOf(args) {
    const filename = args.parsedOptions.f || args.parsedOptions.filename;
    return typeof filename === 'string' ? filename : undefined;
}
/** @return same as fileOf, but also specify whether this came from a -f or --filename option */
export function fileOfWithDetail(args) {
    return {
        filepath: fileOf(args),
        isFor: args.parsedOptions.f ? 'f' : 'filename'
    };
}
/** @return the -R or --recursive option */
export function isRecursive(args) {
    return args.parsedOptions.R || args.parsedOptions.recursive;
}
/** @return the -k or --kustomize option */
export function kustomizeOf(args) {
    return args.parsedOptions.k || args.parsedOptions.kustomize;
}
export function getFileForArgv(args, addSpace = false) {
    const file = fileOf(args);
    if (file) {
        return `-f ${file}${addSpace ? ' ' : ''}`;
    }
    else {
        const kusto = kustomizeOf(args);
        if (kusto) {
            return `-k ${kusto}${addSpace ? ' ' : ''}`;
        }
    }
    return '';
}
export function getFileFromArgv(args) {
    const file = fileOf(args);
    if (file) {
        return file;
    }
    else {
        const kusto = kustomizeOf(args);
        if (kusto) {
            return kusto;
        }
    }
}
export function formatOf({ parsedOptions }) {
    return parsedOptions.o || parsedOptions.output;
}
export function isEntityFormat(format) {
    return format === 'yaml' || format === 'json';
}
export function isEntityRequest(args) {
    return isEntityFormat(formatOf(args));
}
export function isCustomColumns(format) {
    return /^custom-columns=/.test(format);
}
/**
 * Notes: we interpret the lack of an output format designation as a
 * request for tabular output. This seems in keeping with the
 * `kubectl` behavior.
 *
 * @return truthy if the format indicates a desire for tabular output
 *
 */
function isTableFormat(format) {
    return !format || format === 'wide' || isCustomColumns(format) || /^custom-columns-file=/.test(format);
}
export function isDashHelp(args) {
    return args.parsedOptions.help || args.parsedOptions.h;
}
export function isHelpRequest(args) {
    return (isDashHelp(args) || args.argvNoOptions[1] === 'help' || args.argvNoOptions[1] === 'options' // usage: `kubectl options`
    );
}
export function isTableRequest(args) {
    return isTableFormat(formatOf(args));
}
export function isWatchRequest(args) {
    return !!(args.parsedOptions.w || args.parsedOptions.watch || args.parsedOptions['watch-only']);
}
export function watchRequestFrom(args, forceWatch = false) {
    if (forceWatch) {
        return '--watch';
    }
    else if (args.parsedOptions.w) {
        return '-w';
    }
    else if (args.parsedOptions.watch) {
        return '--watch';
    }
    else if (args.parsedOptions['watch-only']) {
        return '--watch-only';
    }
    else {
        return '';
    }
}
export function isTableWatchRequest(args) {
    return isWatchRequest(args) && isTableRequest(args);
}
export function getLabel({ parsedOptions }) {
    const label = parsedOptions.l || parsedOptions.label;
    if (label) {
        return label;
    }
    else {
        // we need to pick the last one, in the case the command line
        // specified more than one; e.g. for `k get pod -lname=nginx
        // -lfoo=bar -n t1`, the kubectl CLI picks foo=bar
        let lastOne;
        // yargs-parser doesn't handle -lname=nginx without the space
        // after -l; or least not the way we've configured it
        for (const key in parsedOptions) {
            if (/^l.+/.test(key) && key !== 'limit') {
                const value = parsedOptions[key];
                if (value) {
                    lastOne = `${key.slice(1)}=${value}`;
                }
            }
        }
        return lastOne;
    }
}
export function getLabelForArgv(args) {
    const label = getLabel(args);
    if (label) {
        return `-l ${label}`;
    }
    else {
        return '';
    }
}
/**
 * Due to deficiencies in yargs-parser (used by @kui-shell/core), the
 * form -lapp=name (i.e. without a whitespace after the -l) is not
 * parsed properly.
 */
export function hasLabel(args) {
    if (args.parsedOptions.l || args.parsedOptions.label) {
        return true;
    }
    for (const key in args.parsedOptions) {
        if (/^l/.test(key)) {
            return true;
        }
    }
    return false;
}
/** @return the namespace as expressed in the command line, or undefined if not */
export function getNamespaceAsExpressed(args) {
    return args.parsedOptions.n || args.parsedOptions.namespace;
}
/** @return the namespace as expressed in the command line, or the default from context */
export function getNamespace(args) {
    return __awaiter(this, void 0, void 0, function* () {
        return args.parsedOptions.n || args.parsedOptions.namespace || (yield getCurrentDefaultNamespace(args));
    });
}
/**
 * A variant of getNamespace where you *only* want to use what was
 * provided by the user in their command line.
 */
export function getNamespaceForArgv({ parsedOptions }) {
    const ns = parsedOptions.n || parsedOptions.namespace;
    return !ns ? '' : `-n ${ns}`;
}
/** @return the resource names array as expressed in the command line */
export function getResourceNamesForArgv(kindFromArgv, args) {
    return args.argvNoOptions.slice(args.argvNoOptions.indexOf(kindFromArgv) + 1);
}
export function getContext(args) {
    return args.parsedOptions.context;
}
/** e.g. for kubectl logs */
export function getContainer(args, verb) {
    const maybe = args.parsedOptions.c || args.parsedOptions.container;
    if (maybe) {
        // specified via -c
        return maybe;
    }
    else {
        // otherwise, specified as a positional parameter
        const idx = args.argvNoOptions.indexOf(verb);
        return args.argvNoOptions[idx + 2];
    }
}
export function getContextForArgv(args) {
    const context = getContext(args);
    if (context) {
        return `--context ${context}`;
    }
    else {
        return '';
    }
}
export function isForAllNamespaces(parsedOptions) {
    return parsedOptions.A || parsedOptions['all-namespaces'];
}
/** Copy over any kubeconfig/context/cluster/namespace specifications from the given args */
export function withKubeconfigFrom(args, cmdline) {
    let extras = ' ';
    if (args.parsedOptions.kubeconfig && !/--kubeconfig/.test(cmdline)) {
        extras += ` --kubeconfig ${expandHomeDir(args.parsedOptions.kubeconfig)}`;
    }
    if (args.parsedOptions.context && !/--context/.test(cmdline)) {
        extras += ` --context ${args.parsedOptions.context}`;
    }
    if (args.parsedOptions.cluster && !/--cluster/.test(cmdline)) {
        extras += ` --cluster ${args.parsedOptions.cluster}`;
    }
    if (!/\s(-n|--namespace)/.test(cmdline) &&
        !/(get|apply|create|delete|explain)\s+(ns|namespace|namespaces|Namespace|Namespaces)\b/.test(cmdline)) {
        // only add namespace option if 1) not already specified; and 2)
        // we aren't fetching a namespace
        extras += ` ${getNamespaceForArgv(args)} `;
    }
    // careful: respect any `--` on the cmdline, and insert our extras
    // *before* that point
    const insertionIndex = cmdline.indexOf(' -- ');
    if (insertionIndex < 0) {
        return cmdline + extras;
    }
    else {
        return cmdline.slice(0, insertionIndex) + extras + cmdline.slice(insertionIndex);
    }
}
function execOptionsHasPrefix(data) {
    return !Buffer.isBuffer(data) && typeof data === 'object' && typeof data.kubectlPrefix === 'string';
}
/** As with `withKubeconfigFrom`, and also copy over the leading command (e.g. `kubectl`) */
export function withKubeconfigAndCommandFrom(args, cmdline) {
    if (execOptionsHasPrefix(args.execOptions.data)) {
        return withKubeconfigFrom(args, `${args.execOptions.data.kubectlPrefix} ${cmdline}`);
    }
    else {
        return withKubeconfigFrom(args, `${getCommandFromArgs(args)} ${cmdline}`);
    }
}
/** Apply --dry-run? */
export function isDryRun(args) {
    const opt = args.parsedOptions['dry-run'];
    return typeof opt === 'boolean' || opt === 'client' || opt === 'server';
}
export function isDiffRequest(args) {
    return args.execOptions.data && args.execOptions.data['diff'];
}
//# sourceMappingURL=options.js.map