import _asyncToGenerator from "@babel/runtime/helpers/asyncToGenerator";
import _regeneratorRuntime from "@babel/runtime/regenerator";

/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import React from 'react';
import { ErrorFilled20 } from '@carbon/icons-react';
import { NotificationButton, ToastNotification, InlineNotification, ActionableNotification } from '../next/Notification';
import { render, screen, waitFor } from '@testing-library/react';
import userEvent from '@testing-library/user-event';
import { settings } from 'carbon-components';
import { it } from 'window-or-global';
var prefix = settings.prefix;
describe('NotificationButton', function () {
  it('should place the `className` prop on the outermost DOM node', function () {
    var _render = render( /*#__PURE__*/React.createElement(NotificationButton, {
      className: "test"
    })),
        container = _render.container;

    expect(container.firstChild).toHaveClass('test');
  });
  it('renders only one icon', function () {
    render( /*#__PURE__*/React.createElement(NotificationButton, null));
    expect(screen.queryAllByRole('img').length).toEqual(1);
  });
  it('supports custom icon', function () {
    var _render2 = render( /*#__PURE__*/React.createElement(NotificationButton, null)),
        rerender = _render2.rerender;

    var defaultIcon = screen.queryByRole('img').innerHTML;
    rerender( /*#__PURE__*/React.createElement(NotificationButton, {
      renderIcon: ErrorFilled20
    }));
    var customIcon = screen.queryByRole('img').innerHTML;
    expect(defaultIcon).not.toEqual(customIcon);
  });
  it('interpolates matching className based on notificationType prop', function () {
    var _render3 = render( /*#__PURE__*/React.createElement(NotificationButton, null)),
        rerender = _render3.rerender,
        container = _render3.container;

    var notificationTypes = ['toast', 'inline'];
    notificationTypes.forEach(function (notificationType) {
      rerender( /*#__PURE__*/React.createElement(NotificationButton, {
        notificationType: notificationType
      }));
      expect(container.firstChild).toHaveClass("".concat(prefix, "--").concat(notificationType, "-notification__close-button"));
      expect(screen.queryByRole('img')).toHaveClass("".concat(prefix, "--").concat(notificationType, "-notification__close-icon"));
    });
  });
});
describe('ToastNotification', function () {
  it('should have role=status by default', function () {
    var _render4 = render( /*#__PURE__*/React.createElement(ToastNotification, null)),
        container = _render4.container;

    expect(container.firstChild).toHaveAttribute('role', 'status');
  });
  it('should place the `className` prop on the outermost DOM node', function () {
    var _render5 = render( /*#__PURE__*/React.createElement(ToastNotification, {
      className: "test"
    })),
        container = _render5.container;

    expect(container.firstChild).toHaveClass('test');
  });
  it('interpolates matching className based on kind prop', function () {
    var _render6 = render( /*#__PURE__*/React.createElement(ToastNotification, null)),
        rerender = _render6.rerender;

    var kinds = ['error', 'info', 'info-square', 'success', 'warning', 'warning-alt'];
    kinds.forEach(function (kind) {
      rerender( /*#__PURE__*/React.createElement(ToastNotification, {
        kind: kind
      }));
      expect(screen.queryByRole('status')).toHaveClass("".concat(prefix, "--toast-notification--").concat(kind));
    });
  });
  it('allows non-interactive elements as children', function () {
    render( /*#__PURE__*/React.createElement(ToastNotification, null, /*#__PURE__*/React.createElement("p", null, "Sample text")));
    expect(screen.queryByText(/Sample text/i)).toBeInTheDocument();
  });
  it('does not allow interactive elements as children', function () {
    var spy = jest.spyOn(console, 'error').mockImplementation(function () {});
    expect(function () {
      render( /*#__PURE__*/React.createElement(ToastNotification, null, /*#__PURE__*/React.createElement("button", {
        type: "button"
      }, "Sample button text")));
    }).toThrow();
    expect(spy).toHaveBeenCalled();
    spy.mockRestore();
  });
  it('close button is rendered by default and includes aria-hidden=true', function () {
    render( /*#__PURE__*/React.createElement(ToastNotification, null));
    var closeButton = screen.queryByRole('button', {
      hidden: true
    });
    expect(closeButton).toBeInTheDocument();
    expect(closeButton).toHaveAttribute('aria-hidden', 'true');
  });
  it('does not render close button when `hideCloseButton` is provided', function () {
    render( /*#__PURE__*/React.createElement(ToastNotification, {
      hideCloseButton: true
    }));
    var closeButton = screen.queryByRole('button', {
      hidden: true
    });
    expect(closeButton).not.toBeInTheDocument();
  });
  it('calls `onClose` when notification is closed', /*#__PURE__*/_asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee() {
    var onClose, closeButton;
    return _regeneratorRuntime.wrap(function _callee$(_context) {
      while (1) {
        switch (_context.prev = _context.next) {
          case 0:
            onClose = jest.fn();
            render( /*#__PURE__*/React.createElement(ToastNotification, {
              onClose: onClose
            }));
            closeButton = screen.queryByRole('button', {
              hidden: true
            });
            userEvent.click(closeButton);
            expect(onClose).toHaveBeenCalledTimes(1);
            _context.next = 7;
            return waitFor(function () {
              expect(screen.queryByRole('status')).not.toBeInTheDocument();
            });

          case 7:
          case "end":
            return _context.stop();
        }
      }
    }, _callee);
  })));
  it('keeps notification open if `onClose` returns false', function () {
    render( /*#__PURE__*/React.createElement(ToastNotification, {
      onClose: function onClose() {
        return false;
      }
    }));
    var closeButton = screen.queryByRole('button', {
      hidden: true
    });
    userEvent.click(closeButton);
    expect(screen.queryByRole('status')).toBeInTheDocument();
  });
  it('calls `onCloseButtonClick` when notification is closed', function () {
    var onCloseButtonClick = jest.fn();
    render( /*#__PURE__*/React.createElement(ToastNotification, {
      onCloseButtonClick: onCloseButtonClick
    }));
    var closeButton = screen.queryByRole('button', {
      hidden: true
    });
    userEvent.click(closeButton);
    expect(onCloseButtonClick).toHaveBeenCalledTimes(1);
  });
  it('closes notification via escape button', /*#__PURE__*/_asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee2() {
    var onCloseButtonClick, onClose;
    return _regeneratorRuntime.wrap(function _callee2$(_context2) {
      while (1) {
        switch (_context2.prev = _context2.next) {
          case 0:
            onCloseButtonClick = jest.fn();
            onClose = jest.fn();
            render( /*#__PURE__*/React.createElement(ToastNotification, {
              onClose: onClose,
              onCloseButtonClick: onCloseButtonClick
            })); // without focus being on/in the notification, it should not close via escape

            userEvent.keyboard('{Escape}');
            expect(onCloseButtonClick).toHaveBeenCalledTimes(0);
            expect(onClose).toHaveBeenCalledTimes(0); // after focus is placed, the notification should close via escape

            userEvent.tab();
            userEvent.keyboard('{Escape}');
            expect(onCloseButtonClick).toHaveBeenCalledTimes(1);
            expect(onClose).toHaveBeenCalledTimes(1);
            _context2.next = 12;
            return waitFor(function () {
              expect(screen.queryByRole('status')).not.toBeInTheDocument();
            });

          case 12:
          case "end":
            return _context2.stop();
        }
      }
    }, _callee2);
  })));
});
describe('InlineNotification', function () {
  it('should have role=status by default', function () {
    var _render7 = render( /*#__PURE__*/React.createElement(InlineNotification, null)),
        container = _render7.container;

    expect(container.firstChild).toHaveAttribute('role', 'status');
  });
  it('should place the `className` prop on the outermost DOM node', function () {
    var _render8 = render( /*#__PURE__*/React.createElement(InlineNotification, {
      className: "test"
    })),
        container = _render8.container;

    expect(container.firstChild).toHaveClass('test');
  });
  it('interpolates matching className based on kind prop', function () {
    var _render9 = render( /*#__PURE__*/React.createElement(InlineNotification, null)),
        rerender = _render9.rerender;

    var kinds = ['error', 'info', 'info-square', 'success', 'warning', 'warning-alt'];
    kinds.forEach(function (kind) {
      rerender( /*#__PURE__*/React.createElement(InlineNotification, {
        kind: kind
      }));
      expect(screen.queryByRole('status')).toHaveClass("".concat(prefix, "--inline-notification--").concat(kind));
    });
  });
  it('allows non-interactive elements as children', function () {
    render( /*#__PURE__*/React.createElement(InlineNotification, null, /*#__PURE__*/React.createElement("p", null, "Sample text")));
    expect(screen.queryByText(/Sample text/i)).toBeInTheDocument();
  });
  it('does not allow interactive elements as children', function () {
    var spy = jest.spyOn(console, 'error').mockImplementation(function () {});
    expect(function () {
      render( /*#__PURE__*/React.createElement(InlineNotification, null, /*#__PURE__*/React.createElement("button", {
        type: "button"
      }, "Sample button text")));
    }).toThrow();
    expect(spy).toHaveBeenCalled();
    spy.mockRestore();
  });
  it('close button is rendered by default and includes aria-hidden=true', function () {
    render( /*#__PURE__*/React.createElement(InlineNotification, null));
    var closeButton = screen.queryByRole('button', {
      hidden: true
    });
    expect(closeButton).toBeInTheDocument();
    expect(closeButton).toHaveAttribute('aria-hidden', 'true');
  });
  it('does not render close button when `hideCloseButton` is provided', function () {
    render( /*#__PURE__*/React.createElement(InlineNotification, {
      hideCloseButton: true
    }));
    var closeButton = screen.queryByRole('button', {
      hidden: true
    });
    expect(closeButton).not.toBeInTheDocument();
  });
  it('calls `onClose` when notification is closed', /*#__PURE__*/_asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee3() {
    var onClose, closeButton;
    return _regeneratorRuntime.wrap(function _callee3$(_context3) {
      while (1) {
        switch (_context3.prev = _context3.next) {
          case 0:
            onClose = jest.fn();
            render( /*#__PURE__*/React.createElement(InlineNotification, {
              onClose: onClose
            }));
            closeButton = screen.queryByRole('button', {
              hidden: true
            });
            userEvent.click(closeButton);
            expect(onClose).toHaveBeenCalledTimes(1);
            _context3.next = 7;
            return waitFor(function () {
              expect(screen.queryByRole('status')).not.toBeInTheDocument();
            });

          case 7:
          case "end":
            return _context3.stop();
        }
      }
    }, _callee3);
  })));
  it('keeps notification open if `onClose` returns false', function () {
    render( /*#__PURE__*/React.createElement(InlineNotification, {
      onClose: function onClose() {
        return false;
      }
    }));
    var closeButton = screen.queryByRole('button', {
      hidden: true
    });
    userEvent.click(closeButton);
    expect(screen.queryByRole('status')).toBeInTheDocument();
  });
  it('calls `onCloseButtonClick` when notification is closed', function () {
    var onCloseButtonClick = jest.fn();
    render( /*#__PURE__*/React.createElement(InlineNotification, {
      onCloseButtonClick: onCloseButtonClick
    }));
    var closeButton = screen.queryByRole('button', {
      hidden: true
    });
    userEvent.click(closeButton);
    expect(onCloseButtonClick).toHaveBeenCalledTimes(1);
  });
  it('closes notification via escape button', /*#__PURE__*/_asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee4() {
    var onCloseButtonClick, onClose;
    return _regeneratorRuntime.wrap(function _callee4$(_context4) {
      while (1) {
        switch (_context4.prev = _context4.next) {
          case 0:
            onCloseButtonClick = jest.fn();
            onClose = jest.fn();
            render( /*#__PURE__*/React.createElement(InlineNotification, {
              onClose: onClose,
              onCloseButtonClick: onCloseButtonClick
            })); // without focus being on/in the notification, it should not close via escape

            userEvent.keyboard('{Escape}');
            expect(onCloseButtonClick).toHaveBeenCalledTimes(0);
            expect(onClose).toHaveBeenCalledTimes(0); // after focus is placed, the notification should close via escape

            userEvent.tab();
            userEvent.keyboard('{Escape}');
            expect(onCloseButtonClick).toHaveBeenCalledTimes(1);
            expect(onClose).toHaveBeenCalledTimes(1);
            _context4.next = 12;
            return waitFor(function () {
              expect(screen.queryByRole('status')).not.toBeInTheDocument();
            });

          case 12:
          case "end":
            return _context4.stop();
        }
      }
    }, _callee4);
  })));
});
describe('ActionableNotification', function () {
  it('uses role=alertdialog', function () {
    var _render10 = render( /*#__PURE__*/React.createElement(ActionableNotification, {
      actionButtonLabel: "My custom action"
    })),
        container = _render10.container;

    expect(container.firstChild).toHaveAttribute('role', 'alertdialog');
  });
  it('renders correct action label', function () {
    render( /*#__PURE__*/React.createElement(ActionableNotification, {
      actionButtonLabel: "My custom action"
    }));
    var actionButton = screen.queryByRole('button', {
      name: 'My custom action'
    });
    expect(actionButton).toBeInTheDocument();
  });
  it('closes notification via escape button when focus is placed on the notification', /*#__PURE__*/_asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee5() {
    var onCloseButtonClick, onClose;
    return _regeneratorRuntime.wrap(function _callee5$(_context5) {
      while (1) {
        switch (_context5.prev = _context5.next) {
          case 0:
            onCloseButtonClick = jest.fn();
            onClose = jest.fn();
            render( /*#__PURE__*/React.createElement(ActionableNotification, {
              onClose: onClose,
              onCloseButtonClick: onCloseButtonClick,
              actionButtonLabel: "My custom action"
            })); // without focus being on/in the notification, it should not close via escape

            userEvent.keyboard('{Escape}');
            expect(onCloseButtonClick).toHaveBeenCalledTimes(0);
            expect(onClose).toHaveBeenCalledTimes(0); // after focus is placed, the notification should close via escape

            userEvent.tab();
            userEvent.keyboard('{Escape}');
            expect(onCloseButtonClick).toHaveBeenCalledTimes(1);
            expect(onClose).toHaveBeenCalledTimes(1);
            _context5.next = 12;
            return waitFor(function () {
              expect(screen.queryByRole('alertdialog')).not.toBeInTheDocument();
            });

          case 12:
          case "end":
            return _context5.stop();
        }
      }
    }, _callee5);
  })));
});