/*
 * Copyright 2019 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import Debug from 'debug';
import { spawn } from 'child_process';
import { expandHomeDir, split, inBrowser } from '@kui-shell/core';
import flags from './flags';
const debug = Debug('plugin-kubeui/controller/kubectl/raw');
/** Part of expandTildes: replace one option with the tilde expansion */
function expand(args, option) {
    const idx = args.argv.indexOf(option.length === 1 ? `-${option}` : `--${option}`);
    if (idx >= 0) {
        const orig = args.argv[idx + 1];
        if (orig) {
            args.argv[idx + 1] = expandHomeDir(orig);
        }
    }
}
/** Expand ~ to the full path of the user's home directory */
function expandTildes(args, env) {
    expand(args, 'cache-dir');
    expand(args, 'certificate-authority');
    expand(args, 'client-key');
    expand(args, 'client-certificate');
    expand(args, 'kubeconfig');
    expand(args, args.parsedOptions.f ? 'f' : 'filename');
    expand(args, args.parsedOptions.k ? 'k' : 'kustomize');
    if (env.HOME) {
        env.HOME = expandHomeDir(env.HOME);
    }
}
/**
 * This is the final bottoming out of the exec/spawn of the native
 * executable.
 *
 */
const doNativeExecOnce = (args) => new Promise((resolve, reject) => {
    const env = Object.assign({}, !inBrowser() ? process.env : {}, args.execOptions.env);
    delete env.DEBUG;
    expandTildes(args, env);
    const executable = args.argv[0].replace(/^_/, '');
    const child = spawn(executable, args.argv.slice(1), { env });
    // this is needed e.g. to handle ENOENT; otherwise the kui process may die with an uncaught exception
    child.on('error', (err) => {
        console.error(`error spawning ${executable}`, err);
        reject(err);
    });
    let stdout = '';
    child.stdout.on('data', data => {
        stdout += data.toString();
    });
    let stderr = '';
    child.stderr.on('data', data => {
        stderr += data.toString();
    });
    child.on('close', (code) => __awaiter(void 0, void 0, void 0, function* () {
        // debug('exec close', code)
        // debug('exec stdout', out)
        if (stderr.length > 0 || code !== 0) {
            debug('exec has stderr with code %s', code);
            debug('exec stderr command', args.command);
            debug('exec stdeout', stdout);
            debug('exec stderr', stderr);
        }
        const noResources = /no resources found/i.test(stderr);
        // const errorFromServer = /Error from server/i.test(stderr)
        if ( /* errorFromServer || */stdout.length === 0 && (code !== 0 || noResources)) {
            const message = stderr;
            const fileNotFound = /error: the path/.test(message);
            const codeForREPL = noResources || /not found/i.test(message) || /doesn't have/i.test(message) || /couldn't find/.test(message)
                ? 404
                : /already exists/i.test(message)
                    ? 409
                    : fileNotFound
                        ? 412
                        : 500;
            if (args.execOptions.failWithUsage) {
                reject(new Error(undefined));
            }
            else {
                const error = new Error(message);
                error.statusCode = code;
                error.code = codeForREPL;
                reject(error);
            }
        }
        else {
            resolve({
                content: {
                    code,
                    stdout,
                    stderr,
                    wasSentToPty: false
                }
            });
        }
    }));
});
export function doNativeExec(args) {
    return __awaiter(this, void 0, void 0, function* () {
        let delay = 1000;
        const maybeRetry = (err) => {
            if (/TLS handshake timeout/.test(err.message)) {
                // retry
                debug(`retrying after handshake timeout with delay=${delay}`);
                console.error(err);
                return new Promise((resolve, reject) => {
                    setTimeout(() => doNativeExecOnce(args).then(resolve, reject), delay);
                    delay += 1000;
                });
            }
            else {
                throw err;
            }
        };
        return doNativeExecOnce(args)
            .catch(maybeRetry)
            .catch(maybeRetry)
            .catch(maybeRetry)
            .catch(maybeRetry)
            .catch(maybeRetry);
    });
}
/**
 * A convenience wrapper over `doNativeExec` that extracts only
 * stdout, and discards the exit code and stderr.
 *
 */
export function doExecRaw(command, parsedOptions, execOptions) {
    return __awaiter(this, void 0, void 0, function* () {
        return (yield doNativeExec({ command, argv: split(command), parsedOptions, execOptions })).content.stdout;
    });
}
export default (registrar) => __awaiter(void 0, void 0, void 0, function* () {
    registrar.listen('/_kubectl', doNativeExec, Object.assign({}, flags, { requiresLocal: true }));
});
//# sourceMappingURL=raw.js.map