import _extends from "@babel/runtime/helpers/extends";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
import _objectWithoutProperties from "@babel/runtime/helpers/objectWithoutProperties";
import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
var _excluded = ["activation", "aria-label", "children", "className", "light", "scrollIntoView", "contained"],
    _excluded2 = ["as", "children", "className", "disabled", "onClick", "onKeyDown"],
    _excluded3 = ["children", "className"];

/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import PropTypes from 'prop-types';
import React, { useState, useRef, useEffect } from 'react';
import cx from 'classnames';
import { keys, match, matches } from '../../../internal/keyboard';
import { usePrefix } from '../../../internal/usePrefix';
import { useId } from '../../../internal/useId';
import { getInteractiveContent } from '../../../internal/useNoInteractiveChildren';
import { useControllableState } from '../../../internal/useControllableState';
import { useMergedRefs } from '../../../internal/useMergedRefs'; // Used to manage the overall state of the Tabs

var TabsContext = /*#__PURE__*/React.createContext(); // Used to keep track of position in a tablist

var TabContext = /*#__PURE__*/React.createContext(); // Used to keep track of position in a list of tab panels

var TabPanelContext = /*#__PURE__*/React.createContext();

function Tabs(_ref) {
  var children = _ref.children,
      _ref$defaultSelectedI = _ref.defaultSelectedIndex,
      defaultSelectedIndex = _ref$defaultSelectedI === void 0 ? 0 : _ref$defaultSelectedI,
      _onChange = _ref.onChange,
      controlledSelectedIndex = _ref.selectedIndex;
  var baseId = useId('ccs'); // The active index is used to track the element which has focus in our tablist

  var _useState = useState(defaultSelectedIndex),
      _useState2 = _slicedToArray(_useState, 2),
      activeIndex = _useState2[0],
      setActiveIndex = _useState2[1]; // The selected index is used for the tab/panel pairing which is "visible"


  var _useControllableState = useControllableState({
    value: controlledSelectedIndex,
    defaultValue: defaultSelectedIndex,
    onChange: function onChange(value) {
      if (_onChange) {
        _onChange({
          selectedIndex: value
        });
      }
    }
  }),
      _useControllableState2 = _slicedToArray(_useControllableState, 2),
      selectedIndex = _useControllableState2[0],
      setSelectedIndex = _useControllableState2[1];

  var value = {
    baseId: baseId,
    activeIndex: activeIndex,
    defaultSelectedIndex: defaultSelectedIndex,
    setActiveIndex: setActiveIndex,
    selectedIndex: selectedIndex,
    setSelectedIndex: setSelectedIndex
  };
  return /*#__PURE__*/React.createElement(TabsContext.Provider, {
    value: value
  }, children);
}

Tabs.propTypes = {
  /**
   * Provide child elements to be rendered inside of the `Tabs`.
   * These elements should render either `TabsList` or `TabsPanels`
   */
  children: PropTypes.node,

  /**
   * Specify which content tab should be initially selected when the component
   * is first rendered
   */
  defaultSelectedIndex: PropTypes.number,

  /**
   * Provide an optional function which is called whenever the state of the
   * `Tabs` changes
   */
  onChange: PropTypes.func,

  /**
   * Control which content panel is currently selected. This puts the component
   * in a controlled mode and should be used along with `onChange`
   */
  selectedIndex: PropTypes.number
};

function useEffectOnce(callback) {
  var savedCallback = useRef(callback);
  var effectGuard = useRef(false);
  useEffect(function () {
    savedCallback.current = callback;
  });
  useEffect(function () {
    if (effectGuard.current !== true) {
      effectGuard.current = true;
      savedCallback.current();
    }
  }, []);
}
/**
 * Get the next index for a given keyboard event given a count of the total
 * items and the current index
 * @param {Event} event
 * @param {number} total
 * @param {number} index
 * @returns {number}
 */


function getNextIndex(event, total, index) {
  if (match(event, keys.ArrowRight)) {
    return (index + 1) % total;
  } else if (match(event, keys.ArrowLeft)) {
    return (total + index - 1) % total;
  } else if (match(event, keys.Home)) {
    return 0;
  } else if (match(event, keys.End)) {
    return total - 1;
  }
}

function TabList(_ref2) {
  var _cx;

  var _ref2$activation = _ref2.activation,
      activation = _ref2$activation === void 0 ? 'automatic' : _ref2$activation,
      label = _ref2['aria-label'],
      children = _ref2.children,
      customClassName = _ref2.className,
      light = _ref2.light,
      scrollIntoView = _ref2.scrollIntoView,
      _ref2$contained = _ref2.contained,
      contained = _ref2$contained === void 0 ? false : _ref2$contained,
      rest = _objectWithoutProperties(_ref2, _excluded);

  var _React$useContext = React.useContext(TabsContext),
      activeIndex = _React$useContext.activeIndex,
      selectedIndex = _React$useContext.selectedIndex,
      setSelectedIndex = _React$useContext.setSelectedIndex,
      setActiveIndex = _React$useContext.setActiveIndex;

  var prefix = usePrefix();
  var ref = useRef(null);
  var className = cx("".concat(prefix, "--tabs"), customClassName, (_cx = {}, _defineProperty(_cx, "".concat(prefix, "--tabs--contained"), contained), _defineProperty(_cx, "".concat(prefix, "--tabs--light"), light), _cx));
  var tabs = [];

  function onKeyDown(event) {
    if (matches(event, [keys.ArrowRight, keys.ArrowLeft, keys.Home, keys.End])) {
      var activeTabs = tabs.filter(function (tab) {
        return !tab.current.disabled;
      });
      var currentIndex = activeTabs.indexOf(tabs[activation === 'automatic' ? selectedIndex : activeIndex]);
      var nextIndex = tabs.indexOf(activeTabs[getNextIndex(event, activeTabs.length, currentIndex)]);

      if (activation === 'automatic') {
        setSelectedIndex(nextIndex);
      } else if (activation === 'manual') {
        setActiveIndex(nextIndex);
      }

      tabs[nextIndex].current.focus();
    }
  }

  useEffectOnce(function () {
    var tab = tabs[selectedIndex];

    if (scrollIntoView && tab) {
      tab.current.scrollIntoView({
        block: 'nearest',
        inline: 'nearest'
      });
    }
  });
  useEffectOnce(function () {
    if (tabs[selectedIndex].current.disabled) {
      var activeTabs = tabs.filter(function (tab) {
        return !tab.current.disabled;
      });

      if (activeTabs.length > 0) {
        var tab = activeTabs[0];
        setSelectedIndex(tabs.indexOf(tab));
      }
    }
  });
  return (
    /*#__PURE__*/
    // eslint-disable-next-line jsx-a11y/interactive-supports-focus
    React.createElement("div", _extends({}, rest, {
      "aria-label": label,
      ref: ref,
      role: "tablist",
      className: className,
      onKeyDown: onKeyDown
    }), React.Children.map(children, function (child, index) {
      var ref = /*#__PURE__*/React.createRef();
      tabs.push(ref);
      return /*#__PURE__*/React.createElement(TabContext.Provider, {
        value: index
      }, /*#__PURE__*/React.cloneElement(child, {
        ref: ref
      }));
    }))
  );
}

TabList.propTypes = {
  /**
   * Specify whether the content tab should be activated automatically or
   * manually
   */
  activation: PropTypes.oneOf(['automatic', 'manual']),

  /**
   * Provide an accessible label to be read when a user interacts with this
   * component
   */
  'aria-label': PropTypes.string.isRequired,

  /**
   * Provide child elements to be rendered inside of `ContentTabs`.
   * These elements should render a `ContentTab`
   */
  children: PropTypes.node,

  /**
   * Specify an optional className to be added to the container node
   */
  className: PropTypes.string,

  /**
   * Specify whether component is contained type
   */
  contained: PropTypes.bool,

  /**
   * Specify whether or not to use the light component variant
   */
  light: PropTypes.bool,

  /**
   * Choose whether or not to automatically scroll to newly selected tabs
   * on component rerender
   */
  scrollIntoView: PropTypes.bool
};
var Tab = /*#__PURE__*/React.forwardRef(function Tab(_ref3, ref) {
  var _cx2;

  var _ref3$as = _ref3.as,
      BaseComponent = _ref3$as === void 0 ? 'button' : _ref3$as,
      children = _ref3.children,
      customClassName = _ref3.className,
      disabled = _ref3.disabled,
      _onClick = _ref3.onClick,
      onKeyDown = _ref3.onKeyDown,
      rest = _objectWithoutProperties(_ref3, _excluded2);

  var prefix = usePrefix();

  var _React$useContext2 = React.useContext(TabsContext),
      selectedIndex = _React$useContext2.selectedIndex,
      setSelectedIndex = _React$useContext2.setSelectedIndex,
      baseId = _React$useContext2.baseId;

  var index = React.useContext(TabContext);
  var id = "".concat(baseId, "-tab-").concat(index);
  var panelId = "".concat(baseId, "-tabpanel-").concat(index);
  var className = cx("".concat(prefix, "--tabs__nav-item"), "".concat(prefix, "--tabs__nav-link"), customClassName, (_cx2 = {}, _defineProperty(_cx2, "".concat(prefix, "--tabs__nav-item--selected"), selectedIndex === index), _defineProperty(_cx2, "".concat(prefix, "--tabs__nav-item--disabled"), disabled), _cx2));
  return /*#__PURE__*/React.createElement(BaseComponent, _extends({}, rest, {
    "aria-controls": panelId,
    "aria-disabled": disabled,
    "aria-selected": selectedIndex === index,
    ref: ref,
    id: id,
    role: "tab",
    className: className,
    disabled: disabled,
    onClick: function onClick(evt) {
      if (disabled) {
        return;
      }

      setSelectedIndex(index);

      if (_onClick) {
        _onClick(evt);
      }
    },
    onKeyDown: onKeyDown,
    tabIndex: selectedIndex === index ? '0' : '-1',
    type: "button"
  }), children);
});
Tab.propTypes = {
  /**
   * Provide a custom element to render instead of the default button
   */
  as: PropTypes.oneOfType([PropTypes.string, PropTypes.elementType]),

  /**
   * Provide child elements to be rendered inside of `Tab`.
   */
  children: PropTypes.node,

  /**
   * Specify an optional className to be added to your Tab
   */
  className: PropTypes.string,

  /**
   * Whether your Tab is disabled.
   */
  disabled: PropTypes.bool,

  /**
   * Provide a handler that is invoked when a user clicks on the control
   */
  onClick: PropTypes.func,

  /**
   * Provide a handler that is invoked on the key down event for the control
   */
  onKeyDown: PropTypes.func,

  /*
   * An optional parameter to allow overriding the anchor rendering.
   * Useful for using Tab along with react-router or other client
   * side router libraries.
   **/
  renderButton: PropTypes.func
};
var TabPanel = /*#__PURE__*/React.forwardRef(function TabPanel(_ref4, forwardRef) {
  var children = _ref4.children,
      customClassName = _ref4.className,
      rest = _objectWithoutProperties(_ref4, _excluded3);

  var prefix = usePrefix();
  var panel = useRef(null);
  var ref = useMergedRefs([forwardRef, panel]);

  var _useState3 = useState('0'),
      _useState4 = _slicedToArray(_useState3, 2),
      tabIndex = _useState4[0],
      setTabIndex = _useState4[1];

  var _React$useContext3 = React.useContext(TabsContext),
      selectedIndex = _React$useContext3.selectedIndex,
      baseId = _React$useContext3.baseId;

  var index = React.useContext(TabPanelContext);
  var id = "".concat(baseId, "-tabpanel-").concat(index);
  var tabId = "".concat(baseId, "-tab-").concat(index);
  var className = cx("".concat(prefix, "--tab-content"), customClassName); // tabindex should only be 0 if no interactive content in children

  useEffect(function () {
    var interactiveContent = getInteractiveContent(panel.current);

    if (interactiveContent) {
      setTabIndex('-1');
    }
  }, []);
  return /*#__PURE__*/React.createElement("div", _extends({}, rest, {
    "aria-labelledby": tabId,
    id: id,
    className: className,
    ref: ref,
    role: "tabpanel",
    tabIndex: tabIndex,
    hidden: selectedIndex !== index
  }), children);
});
TabPanel.propTypes = {
  /**
   * Provide child elements to be rendered inside of `TabPanel`.
   */
  children: PropTypes.node,

  /**
   * Specify an optional className to be added to TabPanel.
   */
  className: PropTypes.string
};

function TabPanels(_ref5) {
  var children = _ref5.children;
  return React.Children.map(children, function (child, index) {
    return /*#__PURE__*/React.createElement(TabPanelContext.Provider, {
      value: index
    }, child);
  });
}

TabPanels.propTypes = {
  /**
   * Provide child elements to be rendered inside of `TabPanels`.
   */
  children: PropTypes.node
};
export { Tabs, Tab, TabPanel, TabPanels, TabList }; // TO DO: implement horizontal scroll and the following props:
// leftOverflowButtonProps
// rightOverflowButtonProps