import _extends from "@babel/runtime/helpers/extends";
import _toConsumableArray from "@babel/runtime/helpers/toConsumableArray";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
import _objectWithoutProperties from "@babel/runtime/helpers/objectWithoutProperties";
var _excluded = ["id", "className", "children", "direction", "disabled", "align", "onClick", "onBlur", "onFocus", "onMouseEnter", "onMouseLeave", "renderIcon", "tooltipText"];

/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import cx from 'classnames';
import React, { useState, useEffect, useRef } from 'react';
import PropTypes from 'prop-types';
import setupGetInstanceId from '../../tools/setupGetInstanceId';
import { composeEventHandlers } from '../../tools/events';
import { keys, matches } from '../../internal/keyboard';
import toggleClass from '../../tools/toggleClass';
import { usePrefix } from '../../internal/usePrefix';
var getInstanceId = setupGetInstanceId();

var TooltipIcon = function TooltipIcon(_ref) {
  var _cx;

  var id = _ref.id,
      className = _ref.className,
      children = _ref.children,
      direction = _ref.direction,
      disabled = _ref.disabled,
      align = _ref.align,
      onClick = _ref.onClick,
      onBlur = _ref.onBlur,
      onFocus = _ref.onFocus,
      onMouseEnter = _ref.onMouseEnter,
      onMouseLeave = _ref.onMouseLeave,
      IconElement = _ref.renderIcon,
      tooltipText = _ref.tooltipText,
      rest = _objectWithoutProperties(_ref, _excluded);

  var prefix = usePrefix();

  var _useState = useState(true),
      _useState2 = _slicedToArray(_useState, 2),
      allowTooltipVisibility = _useState2[0],
      setAllowTooltipVisibility = _useState2[1];

  var _useState3 = useState(false),
      _useState4 = _slicedToArray(_useState3, 2),
      isHovered = _useState4[0],
      setIsHovered = _useState4[1];

  var _useState5 = useState(false),
      _useState6 = _slicedToArray(_useState5, 2),
      isFocused = _useState6[0],
      setIsFocused = _useState6[1];

  var tooltipRef = useRef(null);
  var tooltipTimeout = useRef(null);
  var tooltipId = id || "icon-tooltip-".concat(getInstanceId());
  var tooltipTriggerClasses = cx("".concat(prefix, "--tooltip__trigger"), "".concat(prefix, "--tooltip--a11y"), className, (_cx = {}, _defineProperty(_cx, "".concat(prefix, "--tooltip--").concat(direction), direction), _defineProperty(_cx, "".concat(prefix, "--tooltip--align-").concat(align), align), _defineProperty(_cx, "".concat(prefix, "--tooltip--hidden"), !allowTooltipVisibility || disabled), _defineProperty(_cx, "".concat(prefix, "--tooltip--visible"), isHovered), _cx));

  var closeTooltips = function closeTooltips(evt) {
    var _document;

    var tooltipNode = (_document = document) === null || _document === void 0 ? void 0 : _document.querySelectorAll(".".concat(prefix, "--tooltip--a11y"));

    _toConsumableArray(tooltipNode).map(function (node) {
      toggleClass(node, "".concat(prefix, "--tooltip--hidden"), node !== evt.currentTarget);
    });
  };

  var handleFocus = function handleFocus(evt) {
    closeTooltips(evt);
    setIsFocused(true);
    setAllowTooltipVisibility(true);
  };

  var handleBlur = function handleBlur() {
    setIsHovered(false);
    setIsFocused(false);
    setAllowTooltipVisibility(false);
  };

  var handleMouseEnter = function handleMouseEnter(evt) {
    if (!disabled) {
      tooltipTimeout.current && clearTimeout(tooltipTimeout.current);

      if (evt.target === tooltipRef.current) {
        setAllowTooltipVisibility(true);
        return;
      }

      closeTooltips(evt);
      setAllowTooltipVisibility(true);
    }
  };

  var handleMouseLeave = function handleMouseLeave() {
    if (!isFocused) {
      tooltipTimeout.current = setTimeout(function () {
        setAllowTooltipVisibility(false);
        setIsHovered(false);
      }, 100);
    }
  };

  var handleClick = function handleClick(evt) {
    setAllowTooltipVisibility(false); // Prevent clicks on the tooltip from triggering the button click event

    if (evt.target === tooltipRef.current) {
      evt.preventDefault();
      return;
    }
  };

  useEffect(function () {
    var handleEscKeyDown = function handleEscKeyDown(event) {
      if (matches(event, [keys.Escape])) {
        setAllowTooltipVisibility(false);
        setIsHovered(false);
      }
    };

    document.addEventListener('keydown', handleEscKeyDown);
    return function () {
      return document.removeEventListener('keydown', handleEscKeyDown);
    };
  }, []);
  var cursorStyle;

  if (disabled) {
    cursorStyle = 'not-allowed';
  } else {
    cursorStyle = onClick ? 'pointer' : 'default';
  }

  return /*#__PURE__*/React.createElement("button", _extends({
    disabled: disabled,
    style: {
      cursor: cursorStyle
    }
  }, rest, {
    type: "button",
    className: tooltipTriggerClasses,
    "aria-describedby": tooltipId,
    onMouseEnter: composeEventHandlers([onMouseEnter, handleMouseEnter]),
    onMouseLeave: composeEventHandlers([onMouseLeave, handleMouseLeave]),
    onFocus: composeEventHandlers([onFocus, handleFocus]),
    onBlur: composeEventHandlers([onBlur, handleBlur]),
    onClick: composeEventHandlers([handleClick, onClick])
  }), /*#__PURE__*/React.createElement("span", {
    ref: tooltipRef,
    onMouseEnter: handleMouseEnter,
    className: "".concat(prefix, "--assistive-text"),
    id: tooltipId
  }, tooltipText), IconElement && /*#__PURE__*/React.createElement(IconElement, null), !IconElement && children);
};

TooltipIcon.propTypes = {
  /**
   * Specify the alignment (to the trigger button) of the tooltip.
   * Can be one of: start, center, or end.
   */
  align: PropTypes.oneOf(['start', 'center', 'end']),

  /**
   * Specify an icon as children that will be used as the tooltip trigger. This
   * can be an icon from our Icon component, or a custom SVG element.
   */
  children: PropTypes.node,

  /**
   * Specify an optional className to be applied to the trigger node
   */
  className: PropTypes.string,

  /**
   * Specify the direction of the tooltip. Can be either top or bottom.
   */
  direction: PropTypes.oneOf(['top', 'right', 'left', 'bottom']),

  /**
   * Specify whether the `<TooltipIcon>` should be disabled
   */
  disabled: PropTypes.bool,

  /**
   * Optionally specify a custom id for the tooltip. If one is not provided, we
   * generate a unique id for you.
   */
  id: PropTypes.string,

  /**
   * The event handler for the `blur` event.
   */
  onBlur: PropTypes.func,

  /**
   * The event handler for the `click` event.
   */
  onClick: PropTypes.func,

  /**
   * The event handler for the `focus` event.
   */
  onFocus: PropTypes.func,

  /**
   * The event handler for the `mouseenter` event.
   */
  onMouseEnter: PropTypes.func,

  /**
   * The event handler for the `mouseleave` event.
   */
  onMouseLeave: PropTypes.func,

  /**
   * Function called to override icon rendering.
   */
  renderIcon: PropTypes.oneOfType([PropTypes.func, PropTypes.object]),

  /**
   * Provide the ARIA label for the tooltip.
   * TODO: rename this prop (will be a breaking change)
   */
  tooltipText: PropTypes.node.isRequired
};
TooltipIcon.defaultProps = {
  direction: 'bottom',
  align: 'center'
};
export default TooltipIcon;