import _extends from "@babel/runtime/helpers/extends";
import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
import _objectWithoutProperties from "@babel/runtime/helpers/objectWithoutProperties";
var _excluded = ["align", "className", "children", "label", "description", "enterDelayMs", "leaveDelayMs", "defaultOpen"];

/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import cx from 'classnames';
import PropTypes from 'prop-types';
import React, { useRef } from 'react';
import { Popover, PopoverContent } from '../../Popover';
import { keys, match } from '../../../internal/keyboard';
import { useDelayedState } from '../../../internal/useDelayedState';
import { useId } from '../../../internal/useId';
import { useNoInteractiveChildren } from '../../../internal/useNoInteractiveChildren';
import { usePrefix } from '../../../internal/usePrefix';

function Tooltip(_ref) {
  var _ref$align = _ref.align,
      align = _ref$align === void 0 ? 'top' : _ref$align,
      customClassName = _ref.className,
      children = _ref.children,
      label = _ref.label,
      description = _ref.description,
      _ref$enterDelayMs = _ref.enterDelayMs,
      enterDelayMs = _ref$enterDelayMs === void 0 ? 100 : _ref$enterDelayMs,
      _ref$leaveDelayMs = _ref.leaveDelayMs,
      leaveDelayMs = _ref$leaveDelayMs === void 0 ? 300 : _ref$leaveDelayMs,
      _ref$defaultOpen = _ref.defaultOpen,
      defaultOpen = _ref$defaultOpen === void 0 ? false : _ref$defaultOpen,
      rest = _objectWithoutProperties(_ref, _excluded);

  var containerRef = useRef(null);
  var tooltipRef = useRef(null);

  var _useDelayedState = useDelayedState(defaultOpen),
      _useDelayedState2 = _slicedToArray(_useDelayedState, 2),
      open = _useDelayedState2[0],
      setOpen = _useDelayedState2[1];

  var id = useId('tooltip');
  var prefix = usePrefix();
  var child = React.Children.only(children);
  var triggerProps = {
    onFocus: function onFocus() {
      return setOpen(true);
    },
    onBlur: function onBlur() {
      return setOpen(false);
    },
    // This should be placed on the trigger in case the element is disabled
    onMouseEnter: onMouseEnter
  };

  if (label) {
    triggerProps['aria-labelledby'] = id;
  } else {
    triggerProps['aria-describedby'] = id;
  }

  function onKeyDown(event) {
    if (open && match(event, keys.Escape)) {
      event.stopPropagation();
      setOpen(false);
    }
  }

  function onMouseEnter() {
    setOpen(true, enterDelayMs);
  }

  function onMouseLeave() {
    setOpen(false, leaveDelayMs);
  }

  useNoInteractiveChildren(tooltipRef, 'The Tooltip component must have no interactive content rendered by the' + '`label` or `description` prop');
  return /*#__PURE__*/React.createElement(Popover, _extends({}, rest, {
    align: align,
    className: cx("".concat(prefix, "--tooltip"), customClassName),
    dropShadow: false,
    highContrast: true,
    onKeyDown: onKeyDown,
    onMouseLeave: onMouseLeave,
    open: open,
    ref: containerRef
  }), /*#__PURE__*/React.cloneElement(child, triggerProps), /*#__PURE__*/React.createElement(PopoverContent, {
    "aria-hidden": "true",
    className: "".concat(prefix, "--tooltip-content"),
    id: id,
    ref: tooltipRef,
    role: "tooltip"
  }, label || description));
}

Tooltip.propTypes = {
  /**
   * Specify how the trigger should align with the tooltip
   */
  align: PropTypes.oneOf(['top', 'top-left', 'top-right', 'bottom', 'bottom-left', 'bottom-right', 'left', 'left-bottom', 'left-top', 'right', 'right-bottom', 'right-top']),

  /**
   * Pass in the child to which the tooltip will be applied
   */
  children: PropTypes.node,

  /**
   * Specify an optional className to be applied to the container node
   */
  className: PropTypes.string,

  /**
   * Specify whether the tooltip should be open when it first renders
   */
  defaultOpen: PropTypes.bool,

  /**
   * Provide the description to be rendered inside of the Tooltip. The
   * description will use `aria-describedby` and will describe the child node
   * in addition to the text rendered inside of the child. This means that if you
   * have text in the child node, that it will be announced alongside the
   * description to the screen reader.
   *
   * Note: if label and description are both provided, label will be used and
   * description will not be used
   */
  description: PropTypes.node,

  /**
   * Specify the duration in milliseconds to delay before displaying the tooltip
   */
  enterDelayMs: PropTypes.number,

  /**
   * Provide the label to be rendered inside of the Tooltip. The label will use
   * `aria-labelledby` and will fully describe the child node that is provided.
   * This means that if you have text in the child node, that it will not be
   * announced to the screen reader.
   *
   * Note: if label and description are both provided, description will not be
   * used
   */
  label: PropTypes.node,

  /**
   * Specify the duration in milliseconds to delay before hiding the tooltip
   */
  leaveDelayMs: PropTypes.number
};
export { Tooltip };