import { Kind, typeFromAST, TypeInfo, visit, visitWithTypeInfo, getNamedType, isAbstractType, isObjectType, isInterfaceType, TypeNameMetaFieldDef, getNullableType, isLeafType, isCompositeType, isListType, isSchema, getOperationAST, execute, subscribe, validate, defaultFieldResolver, parse } from 'graphql';
import { applySchemaTransforms, serializeInputValue, updateArgument, collectFields, implementsAbstractType, relocatedError, getErrorsByPathSegment, mergeDeep, ERROR_SYMBOL, setErrors, slicedError, getResponseKeyFromInfo, mapAsyncIterator, getErrors, concatInlineFragments } from '@graphql-tools/utils';
import AggregateError from '@ardatan/aggregate-error';

const OBJECT_SUBSCHEMA_SYMBOL = Symbol('initialSubschema');
const FIELD_SUBSCHEMA_MAP_SYMBOL = Symbol('subschemaMap');

function getSubschema(result, responseKey) {
    const subschema = result[FIELD_SUBSCHEMA_MAP_SYMBOL] && result[FIELD_SUBSCHEMA_MAP_SYMBOL][responseKey];
    return subschema || result[OBJECT_SUBSCHEMA_SYMBOL];
}
function setObjectSubschema(result, subschema) {
    result[OBJECT_SUBSCHEMA_SYMBOL] = subschema;
}
function isSubschemaConfig(value) {
    return Boolean(value.schema);
}
function isSubschema(value) {
    return Boolean(value.transformedSchema);
}
class Subschema {
    constructor(config) {
        var _a;
        this.schema = config.schema;
        this.executor = config.executor;
        this.subscriber = config.subscriber;
        this.createProxyingResolver = config.createProxyingResolver;
        this.transforms = (_a = config.transforms) !== null && _a !== void 0 ? _a : [];
        this.merge = config.merge;
        this.transformedSchema = applySchemaTransforms(this.schema, this.transforms);
    }
}

function getDelegatingOperation(parentType, schema) {
    if (parentType === schema.getMutationType()) {
        return 'mutation';
    }
    else if (parentType === schema.getSubscriptionType()) {
        return 'subscription';
    }
    return 'query';
}
function createRequestFromInfo({ info, operationName, operation = getDelegatingOperation(info.parentType, info.schema), fieldName = info.fieldName, selectionSet, fieldNodes = info.fieldNodes, }) {
    return createRequest({
        sourceSchema: info.schema,
        sourceParentType: info.parentType,
        sourceFieldName: info.fieldName,
        fragments: info.fragments,
        variableDefinitions: info.operation.variableDefinitions,
        variableValues: info.variableValues,
        targetOperationName: operationName,
        targetOperation: operation,
        targetFieldName: fieldName,
        selectionSet,
        fieldNodes,
    });
}
function createRequest({ sourceSchema, sourceParentType, sourceFieldName, fragments, variableDefinitions, variableValues, targetOperationName, targetOperation, targetFieldName, selectionSet, fieldNodes, }) {
    var _a;
    let newSelectionSet;
    let argumentNodeMap;
    if (selectionSet != null) {
        newSelectionSet = selectionSet;
        argumentNodeMap = Object.create(null);
    }
    else {
        const selections = fieldNodes.reduce((acc, fieldNode) => (fieldNode.selectionSet != null ? acc.concat(fieldNode.selectionSet.selections) : acc), []);
        newSelectionSet = selections.length
            ? {
                kind: Kind.SELECTION_SET,
                selections,
            }
            : undefined;
        argumentNodeMap = {};
        const args = (_a = fieldNodes[0]) === null || _a === void 0 ? void 0 : _a.arguments;
        if (args) {
            argumentNodeMap = args.reduce((prev, curr) => ({
                ...prev,
                [curr.name.value]: curr,
            }), argumentNodeMap);
        }
    }
    const newVariables = Object.create(null);
    const variableDefinitionMap = Object.create(null);
    if (sourceSchema != null && variableDefinitions != null) {
        variableDefinitions.forEach(def => {
            const varName = def.variable.name.value;
            variableDefinitionMap[varName] = def;
            const varType = typeFromAST(sourceSchema, def.type);
            const serializedValue = serializeInputValue(varType, variableValues[varName]);
            if (serializedValue !== undefined) {
                newVariables[varName] = serializedValue;
            }
        });
    }
    if (sourceParentType != null) {
        updateArgumentsWithDefaults(sourceParentType, sourceFieldName, argumentNodeMap, variableDefinitionMap, newVariables);
    }
    const rootfieldNode = {
        kind: Kind.FIELD,
        arguments: Object.keys(argumentNodeMap).map(argName => argumentNodeMap[argName]),
        name: {
            kind: Kind.NAME,
            value: targetFieldName || fieldNodes[0].name.value,
        },
        selectionSet: newSelectionSet,
    };
    const operationName = targetOperationName
        ? {
            kind: Kind.NAME,
            value: targetOperationName,
        }
        : undefined;
    const operationDefinition = {
        kind: Kind.OPERATION_DEFINITION,
        name: operationName,
        operation: targetOperation,
        variableDefinitions: Object.keys(variableDefinitionMap).map(varName => variableDefinitionMap[varName]),
        selectionSet: {
            kind: Kind.SELECTION_SET,
            selections: [rootfieldNode],
        },
    };
    let definitions = [operationDefinition];
    if (fragments != null) {
        definitions = definitions.concat(Object.keys(fragments).map(fragmentName => fragments[fragmentName]));
    }
    const document = {
        kind: Kind.DOCUMENT,
        definitions,
    };
    return {
        document,
        variables: newVariables,
    };
}
function updateArgumentsWithDefaults(sourceParentType, sourceFieldName, argumentNodeMap, variableDefinitionMap, variableValues) {
    const sourceField = sourceParentType.getFields()[sourceFieldName];
    sourceField.args.forEach((argument) => {
        const argName = argument.name;
        const sourceArgType = argument.type;
        if (argumentNodeMap[argName] === undefined) {
            const defaultValue = argument.defaultValue;
            if (defaultValue !== undefined) {
                updateArgument(argName, sourceArgType, argumentNodeMap, variableDefinitionMap, variableValues, serializeInputValue(sourceArgType, defaultValue));
            }
        }
    });
}

class VisitSelectionSets {
    constructor(schema, initialType, visitor) {
        this.schema = schema;
        this.initialType = initialType;
        this.visitor = visitor;
    }
    transformRequest(originalRequest) {
        const document = visitSelectionSets(originalRequest, this.schema, this.initialType, this.visitor);
        return {
            ...originalRequest,
            document,
        };
    }
}
function visitSelectionSets(request, schema, initialType, visitor) {
    const { document, variables } = request;
    const operations = [];
    const fragments = Object.create(null);
    document.definitions.forEach(def => {
        if (def.kind === Kind.OPERATION_DEFINITION) {
            operations.push(def);
        }
        else if (def.kind === Kind.FRAGMENT_DEFINITION) {
            fragments[def.name.value] = def;
        }
    });
    const partialExecutionContext = {
        schema,
        variableValues: variables,
        fragments,
    };
    const typeInfo = new TypeInfo(schema, undefined, initialType);
    const newDefinitions = operations.map(operation => {
        const type = operation.operation === 'query'
            ? schema.getQueryType()
            : operation.operation === 'mutation'
                ? schema.getMutationType()
                : schema.getSubscriptionType();
        const fields = collectFields(partialExecutionContext, type, operation.selectionSet, Object.create(null), Object.create(null));
        const newSelections = [];
        Object.keys(fields).forEach(responseKey => {
            const fieldNodes = fields[responseKey];
            fieldNodes.forEach(fieldNode => {
                const selectionSet = fieldNode.selectionSet;
                if (selectionSet == null) {
                    newSelections.push(fieldNode);
                    return;
                }
                const newSelectionSet = visit(selectionSet, visitWithTypeInfo(typeInfo, {
                    [Kind.SELECTION_SET]: node => visitor(node, typeInfo),
                }));
                if (newSelectionSet === selectionSet) {
                    newSelections.push(fieldNode);
                    return;
                }
                newSelections.push({
                    ...fieldNode,
                    selectionSet: newSelectionSet,
                });
            });
        });
        return {
            ...operation,
            selectionSet: {
                kind: Kind.SELECTION_SET,
                selections: newSelections,
            },
        };
    });
    Object.values(fragments).forEach(fragment => {
        newDefinitions.push(visit(fragment, visitWithTypeInfo(typeInfo, {
            [Kind.SELECTION_SET]: node => visitor(node, typeInfo),
        })));
    });
    return {
        ...document,
        definitions: newDefinitions,
    };
}

class AddSelectionSetsByField {
    constructor(sourceSchema, initialType, selectionSetsByType, selectionSetsByField, dynamicSelectionSetsByField) {
        this.transformer = new VisitSelectionSets(sourceSchema, initialType, (node, typeInfo) => visitSelectionSet(node, typeInfo, selectionSetsByType, selectionSetsByField, dynamicSelectionSetsByField));
    }
    transformRequest(originalRequest) {
        return this.transformer.transformRequest(originalRequest);
    }
}
function visitSelectionSet(node, typeInfo, selectionSetsByType, selectionSetsByField, dynamicSelectionSetsByField) {
    const parentType = typeInfo.getParentType();
    if (parentType != null) {
        const parentTypeName = parentType.name;
        let selections = node.selections;
        if (parentTypeName in selectionSetsByType) {
            const selectionSet = selectionSetsByType[parentTypeName];
            if (selectionSet != null) {
                selections = selections.concat(selectionSet.selections);
            }
        }
        if (parentTypeName in selectionSetsByField) {
            node.selections.forEach(selection => {
                if (selection.kind === Kind.FIELD) {
                    const name = selection.name.value;
                    const selectionSet = selectionSetsByField[parentTypeName][name];
                    if (selectionSet != null) {
                        selections = selections.concat(selectionSet.selections);
                    }
                }
            });
        }
        if (parentTypeName in dynamicSelectionSetsByField) {
            node.selections.forEach(selection => {
                if (selection.kind === Kind.FIELD) {
                    const name = selection.name.value;
                    const dynamicSelectionSets = dynamicSelectionSetsByField[parentTypeName][name];
                    if (dynamicSelectionSets != null) {
                        dynamicSelectionSets.forEach(selectionSetFn => {
                            const selectionSet = selectionSetFn(selection);
                            if (selectionSet != null) {
                                selections = selections.concat(selectionSet.selections);
                            }
                        });
                    }
                }
            });
        }
        if (selections !== node.selections) {
            return {
                ...node,
                selections,
            };
        }
    }
}

class ExpandAbstractTypes {
    constructor(sourceSchema, targetSchema) {
        this.targetSchema = targetSchema;
        this.mapping = extractPossibleTypes(sourceSchema, targetSchema);
        this.reverseMapping = flipMapping(this.mapping);
    }
    transformRequest(originalRequest) {
        const document = expandAbstractTypes(this.targetSchema, this.mapping, this.reverseMapping, originalRequest.document);
        return {
            ...originalRequest,
            document,
        };
    }
}
function extractPossibleTypes(sourceSchema, targetSchema) {
    const typeMap = sourceSchema.getTypeMap();
    const mapping = Object.create(null);
    Object.keys(typeMap).forEach(typeName => {
        const type = typeMap[typeName];
        if (isAbstractType(type)) {
            const targetType = targetSchema.getType(typeName);
            if (!isAbstractType(targetType)) {
                const implementations = sourceSchema.getPossibleTypes(type);
                mapping[typeName] = implementations.filter(impl => targetSchema.getType(impl.name)).map(impl => impl.name);
            }
        }
    });
    return mapping;
}
function flipMapping(mapping) {
    const result = Object.create(null);
    Object.keys(mapping).forEach(typeName => {
        const toTypeNames = mapping[typeName];
        toTypeNames.forEach(toTypeName => {
            if (!(toTypeName in result)) {
                result[toTypeName] = [];
            }
            result[toTypeName].push(typeName);
        });
    });
    return result;
}
function expandAbstractTypes(targetSchema, mapping, reverseMapping, document) {
    const operations = document.definitions.filter(def => def.kind === Kind.OPERATION_DEFINITION);
    const fragments = document.definitions.filter(def => def.kind === Kind.FRAGMENT_DEFINITION);
    const existingFragmentNames = fragments.map(fragment => fragment.name.value);
    let fragmentCounter = 0;
    const generateFragmentName = (typeName) => {
        let fragmentName;
        do {
            fragmentName = `_${typeName}_Fragment${fragmentCounter.toString()}`;
            fragmentCounter++;
        } while (existingFragmentNames.indexOf(fragmentName) !== -1);
        return fragmentName;
    };
    const newFragments = [];
    const fragmentReplacements = Object.create(null);
    fragments.forEach((fragment) => {
        newFragments.push(fragment);
        const possibleTypes = mapping[fragment.typeCondition.name.value];
        if (possibleTypes != null) {
            fragmentReplacements[fragment.name.value] = [];
            possibleTypes.forEach(possibleTypeName => {
                const name = generateFragmentName(possibleTypeName);
                existingFragmentNames.push(name);
                const newFragment = {
                    kind: Kind.FRAGMENT_DEFINITION,
                    name: {
                        kind: Kind.NAME,
                        value: name,
                    },
                    typeCondition: {
                        kind: Kind.NAMED_TYPE,
                        name: {
                            kind: Kind.NAME,
                            value: possibleTypeName,
                        },
                    },
                    selectionSet: fragment.selectionSet,
                };
                newFragments.push(newFragment);
                fragmentReplacements[fragment.name.value].push({
                    fragmentName: name,
                    typeName: possibleTypeName,
                });
            });
        }
    });
    const newDocument = {
        ...document,
        definitions: [...operations, ...newFragments],
    };
    const typeInfo = new TypeInfo(targetSchema);
    return visit(newDocument, visitWithTypeInfo(typeInfo, {
        [Kind.SELECTION_SET](node) {
            const newSelections = [...node.selections];
            const maybeType = typeInfo.getParentType();
            if (maybeType != null) {
                const parentType = getNamedType(maybeType);
                node.selections.forEach((selection) => {
                    if (selection.kind === Kind.INLINE_FRAGMENT) {
                        if (selection.typeCondition != null) {
                            const possibleTypes = mapping[selection.typeCondition.name.value];
                            if (possibleTypes != null) {
                                possibleTypes.forEach(possibleType => {
                                    const maybePossibleType = targetSchema.getType(possibleType);
                                    if (maybePossibleType != null &&
                                        implementsAbstractType(targetSchema, parentType, maybePossibleType)) {
                                        newSelections.push({
                                            kind: Kind.INLINE_FRAGMENT,
                                            typeCondition: {
                                                kind: Kind.NAMED_TYPE,
                                                name: {
                                                    kind: Kind.NAME,
                                                    value: possibleType,
                                                },
                                            },
                                            selectionSet: selection.selectionSet,
                                        });
                                    }
                                });
                            }
                        }
                    }
                    else if (selection.kind === Kind.FRAGMENT_SPREAD) {
                        const fragmentName = selection.name.value;
                        if (fragmentName in fragmentReplacements) {
                            fragmentReplacements[fragmentName].forEach(replacement => {
                                const typeName = replacement.typeName;
                                const maybeReplacementType = targetSchema.getType(typeName);
                                if (maybeReplacementType != null && implementsAbstractType(targetSchema, parentType, maybeType)) {
                                    newSelections.push({
                                        kind: Kind.FRAGMENT_SPREAD,
                                        name: {
                                            kind: Kind.NAME,
                                            value: replacement.fragmentName,
                                        },
                                    });
                                }
                            });
                        }
                    }
                });
                if (parentType.name in reverseMapping) {
                    newSelections.push({
                        kind: Kind.FIELD,
                        name: {
                            kind: Kind.NAME,
                            value: '__typename',
                        },
                    });
                }
            }
            if (newSelections.length !== node.selections.length) {
                return {
                    ...node,
                    selections: newSelections,
                };
            }
        },
    }));
}

// For motivation, see https://github.com/ardatan/graphql-tools/issues/751
class WrapConcreteTypes {
    constructor(returnType, targetSchema) {
        this.returnType = returnType;
        this.targetSchema = targetSchema;
    }
    transformRequest(originalRequest) {
        const document = wrapConcreteTypes(this.returnType, this.targetSchema, originalRequest.document);
        return {
            ...originalRequest,
            document,
        };
    }
}
function wrapConcreteTypes(returnType, targetSchema, document) {
    const namedType = getNamedType(returnType);
    if (!isObjectType(namedType)) {
        return document;
    }
    const queryRootType = targetSchema.getQueryType();
    const mutationRootType = targetSchema.getMutationType();
    const subscriptionRootType = targetSchema.getSubscriptionType();
    const typeInfo = new TypeInfo(targetSchema);
    const newDocument = visit(document, visitWithTypeInfo(typeInfo, {
        [Kind.FIELD](node) {
            const maybeType = typeInfo.getParentType();
            if (maybeType == null) {
                return false;
            }
            const parentType = getNamedType(maybeType);
            if (parentType !== queryRootType && parentType !== mutationRootType && parentType !== subscriptionRootType) {
                return false;
            }
            if (!isAbstractType(getNamedType(typeInfo.getType()))) {
                return false;
            }
            return {
                ...node,
                selectionSet: {
                    kind: Kind.SELECTION_SET,
                    selections: [
                        {
                            kind: Kind.INLINE_FRAGMENT,
                            typeCondition: {
                                kind: Kind.NAMED_TYPE,
                                name: {
                                    kind: Kind.NAME,
                                    value: namedType.name,
                                },
                            },
                            selectionSet: node.selectionSet,
                        },
                    ],
                },
            };
        },
    }));
    return newDocument;
}

class FilterToSchema {
    constructor(targetSchema) {
        this.targetSchema = targetSchema;
    }
    transformRequest(originalRequest) {
        return {
            ...originalRequest,
            ...filterToSchema(this.targetSchema, originalRequest.document, originalRequest.variables),
        };
    }
}
function filterToSchema(targetSchema, document, variables) {
    const operations = document.definitions.filter(def => def.kind === Kind.OPERATION_DEFINITION);
    const fragments = document.definitions.filter(def => def.kind === Kind.FRAGMENT_DEFINITION);
    let usedVariables = [];
    let usedFragments = [];
    const newOperations = [];
    let newFragments = [];
    const validFragments = fragments.filter((fragment) => {
        const typeName = fragment.typeCondition.name.value;
        return Boolean(targetSchema.getType(typeName));
    });
    const validFragmentsWithType = validFragments.reduce((prev, fragment) => ({
        ...prev,
        [fragment.name.value]: targetSchema.getType(fragment.typeCondition.name.value),
    }), {});
    let fragmentSet = Object.create(null);
    operations.forEach((operation) => {
        let type;
        if (operation.operation === 'subscription') {
            type = targetSchema.getSubscriptionType();
        }
        else if (operation.operation === 'mutation') {
            type = targetSchema.getMutationType();
        }
        else {
            type = targetSchema.getQueryType();
        }
        const { selectionSet, usedFragments: operationUsedFragments, usedVariables: operationUsedVariables, } = filterSelectionSet(targetSchema, type, validFragmentsWithType, operation.selectionSet);
        usedFragments = union(usedFragments, operationUsedFragments);
        const { usedVariables: collectedUsedVariables, newFragments: collectedNewFragments, fragmentSet: collectedFragmentSet, } = collectFragmentVariables(targetSchema, fragmentSet, validFragments, validFragmentsWithType, usedFragments);
        const operationOrFragmentVariables = union(operationUsedVariables, collectedUsedVariables);
        usedVariables = union(usedVariables, operationOrFragmentVariables);
        newFragments = collectedNewFragments;
        fragmentSet = collectedFragmentSet;
        const variableDefinitions = operation.variableDefinitions.filter((variable) => operationOrFragmentVariables.indexOf(variable.variable.name.value) !== -1);
        newOperations.push({
            kind: Kind.OPERATION_DEFINITION,
            operation: operation.operation,
            name: operation.name,
            directives: operation.directives,
            variableDefinitions,
            selectionSet,
        });
    });
    const newVariables = usedVariables.reduce((acc, variableName) => {
        const variableValue = variables[variableName];
        if (variableValue !== undefined) {
            acc[variableName] = variableValue;
        }
        return acc;
    }, {});
    return {
        document: {
            kind: Kind.DOCUMENT,
            definitions: [...newOperations, ...newFragments],
        },
        variables: newVariables,
    };
}
function collectFragmentVariables(targetSchema, fragmentSet, validFragments, validFragmentsWithType, usedFragments) {
    let remainingFragments = usedFragments.slice();
    let usedVariables = [];
    const newFragments = [];
    while (remainingFragments.length !== 0) {
        const nextFragmentName = remainingFragments.pop();
        const fragment = validFragments.find(fr => fr.name.value === nextFragmentName);
        if (fragment != null) {
            const name = nextFragmentName;
            const typeName = fragment.typeCondition.name.value;
            const type = targetSchema.getType(typeName);
            const { selectionSet, usedFragments: fragmentUsedFragments, usedVariables: fragmentUsedVariables, } = filterSelectionSet(targetSchema, type, validFragmentsWithType, fragment.selectionSet);
            remainingFragments = union(remainingFragments, fragmentUsedFragments);
            usedVariables = union(usedVariables, fragmentUsedVariables);
            if (!(name in fragmentSet)) {
                fragmentSet[name] = true;
                newFragments.push({
                    kind: Kind.FRAGMENT_DEFINITION,
                    name: {
                        kind: Kind.NAME,
                        value: name,
                    },
                    typeCondition: fragment.typeCondition,
                    selectionSet,
                });
            }
        }
    }
    return {
        usedVariables,
        newFragments,
        fragmentSet,
    };
}
function filterSelectionSet(schema, type, validFragments, selectionSet) {
    const usedFragments = [];
    const usedVariables = [];
    const typeInfo = new TypeInfo(schema, undefined, type);
    const filteredSelectionSet = visit(selectionSet, visitWithTypeInfo(typeInfo, {
        [Kind.FIELD]: {
            enter(node) {
                const parentType = typeInfo.getParentType();
                if (isObjectType(parentType) || isInterfaceType(parentType)) {
                    const fields = parentType.getFields();
                    const field = node.name.value === '__typename' ? TypeNameMetaFieldDef : fields[node.name.value];
                    if (!field) {
                        return null;
                    }
                    const argNames = (field.args != null ? field.args : []).map(arg => arg.name);
                    if (node.arguments != null) {
                        const args = node.arguments.filter((arg) => argNames.indexOf(arg.name.value) !== -1);
                        if (args.length !== node.arguments.length) {
                            return {
                                ...node,
                                arguments: args,
                            };
                        }
                    }
                }
            },
            leave(node) {
                const resolvedType = getNamedType(typeInfo.getType());
                if (isObjectType(resolvedType) || isInterfaceType(resolvedType)) {
                    const selections = node.selectionSet != null ? node.selectionSet.selections : null;
                    if (selections == null || selections.length === 0) {
                        // need to remove any added variables. Is there a better way to do this?
                        visit(node, {
                            [Kind.VARIABLE](variableNode) {
                                const index = usedVariables.indexOf(variableNode.name.value);
                                if (index !== -1) {
                                    usedVariables.splice(index, 1);
                                }
                            },
                        });
                        return null;
                    }
                }
            },
        },
        [Kind.FRAGMENT_SPREAD](node) {
            if (node.name.value in validFragments) {
                const parentType = typeInfo.getParentType();
                const innerType = validFragments[node.name.value];
                if (!implementsAbstractType(schema, parentType, innerType)) {
                    return null;
                }
                usedFragments.push(node.name.value);
                return;
            }
            return null;
        },
        [Kind.INLINE_FRAGMENT]: {
            enter(node) {
                if (node.typeCondition != null) {
                    const parentType = typeInfo.getParentType();
                    const innerType = schema.getType(node.typeCondition.name.value);
                    if (!implementsAbstractType(schema, parentType, innerType)) {
                        return null;
                    }
                }
            },
        },
        [Kind.VARIABLE](node) {
            usedVariables.push(node.name.value);
        },
    }));
    return {
        selectionSet: filteredSelectionSet,
        usedFragments,
        usedVariables,
    };
}
function union(...arrays) {
    const cache = Object.create(null);
    const result = [];
    arrays.forEach(array => {
        array.forEach(item => {
            if (!(item in cache)) {
                cache[item] = true;
                result.push(item);
            }
        });
    });
    return result;
}

class AddFragmentsByField {
    constructor(targetSchema, mapping) {
        this.targetSchema = targetSchema;
        this.mapping = mapping;
    }
    transformRequest(originalRequest) {
        const document = addFragmentsByField(this.targetSchema, originalRequest.document, this.mapping);
        return {
            ...originalRequest,
            document,
        };
    }
}
function addFragmentsByField(targetSchema, document, mapping) {
    const typeInfo = new TypeInfo(targetSchema);
    return visit(document, visitWithTypeInfo(typeInfo, {
        [Kind.SELECTION_SET](node) {
            const parentType = typeInfo.getParentType();
            if (parentType != null) {
                const parentTypeName = parentType.name;
                let selections = node.selections;
                if (parentTypeName in mapping) {
                    node.selections.forEach(selection => {
                        if (selection.kind === Kind.FIELD) {
                            const name = selection.name.value;
                            const fragment = mapping[parentTypeName][name];
                            if (fragment != null) {
                                selections = selections.concat(fragment);
                            }
                        }
                    });
                }
                if (selections !== node.selections) {
                    return {
                        ...node,
                        selections,
                    };
                }
            }
        },
    }));
}

class AddTypenameToAbstract {
    constructor(targetSchema) {
        this.targetSchema = targetSchema;
    }
    transformRequest(originalRequest) {
        const document = addTypenameToAbstract(this.targetSchema, originalRequest.document);
        return {
            ...originalRequest,
            document,
        };
    }
}
function addTypenameToAbstract(targetSchema, document) {
    const typeInfo = new TypeInfo(targetSchema);
    return visit(document, visitWithTypeInfo(typeInfo, {
        [Kind.SELECTION_SET](node) {
            const parentType = typeInfo.getParentType();
            let selections = node.selections;
            if (parentType != null && isAbstractType(parentType)) {
                selections = selections.concat({
                    kind: Kind.FIELD,
                    name: {
                        kind: Kind.NAME,
                        value: '__typename',
                    },
                });
            }
            if (selections !== node.selections) {
                return {
                    ...node,
                    selections,
                };
            }
        },
    }));
}

function handleNull(errors) {
    if (errors.length) {
        if (errors.some(error => !error.path || error.path.length < 2)) {
            if (errors.length > 1) {
                const combinedError = new AggregateError(errors);
                return combinedError;
            }
            const error = errors[0];
            return error.originalError || relocatedError(error, null);
        }
        else if (errors.some(error => typeof error.path[1] === 'string')) {
            const childErrors = getErrorsByPathSegment(errors);
            const result = {};
            Object.keys(childErrors).forEach(pathSegment => {
                result[pathSegment] = handleNull(childErrors[pathSegment]);
            });
            return result;
        }
        const childErrors = getErrorsByPathSegment(errors);
        const result = [];
        Object.keys(childErrors).forEach(pathSegment => {
            result.push(handleNull(childErrors[pathSegment]));
        });
        return result;
    }
    return null;
}

function memoizeInfoAnd2Objectsand1Primitive(fn) {
    let cache1;
    function memoized(a1, a2, a3, a4) {
        if (!cache1) {
            cache1 = new WeakMap();
            const cache2 = new WeakMap();
            cache1.set(a1.fieldNodes, cache2);
            const cache3 = new WeakMap();
            cache2.set(a2, cache3);
            const cache4 = Object.create(null);
            cache3.set(a3, cache4);
            const newValue = fn(a1, a2, a3, a4);
            cache4[a4] = newValue;
            return newValue;
        }
        let cache2 = cache1.get(a1.fieldNodes);
        if (!cache2) {
            cache2 = new WeakMap();
            cache1.set(a1.fieldNodes, cache2);
            const cache3 = new WeakMap();
            cache2.set(a2, cache3);
            const cache4 = Object.create(null);
            cache3.set(a3, cache4);
            const newValue = fn(a1, a2, a3, a4);
            cache4[a4] = newValue;
            return newValue;
        }
        let cache3 = cache2.get(a2);
        if (!cache3) {
            cache3 = new WeakMap();
            cache2.set(a2, cache3);
            const cache4 = Object.create(null);
            cache3.set(a3, cache4);
            const newValue = fn(a1, a2, a3, a4);
            cache4[a4] = newValue;
            return newValue;
        }
        let cache4 = cache3.get(a3);
        if (!cache4) {
            cache4 = Object.create(null);
            cache3.set(a3, cache4);
            const newValue = fn(a1, a2, a3, a4);
            cache4[a4] = newValue;
            return newValue;
        }
        const cachedValue = cache4[a4];
        if (cachedValue === undefined) {
            const newValue = fn(a1, a2, a3, a4);
            cache4[a4] = newValue;
            return newValue;
        }
        return cachedValue;
    }
    return memoized;
}
function memoize3(fn) {
    let cache1;
    function memoized(a1, a2, a3) {
        if (!cache1) {
            cache1 = new WeakMap();
            const cache2 = new WeakMap();
            cache1.set(a1, cache2);
            const cache3 = new WeakMap();
            cache2.set(a2, cache3);
            const newValue = fn(a1, a2, a3);
            cache3.set(a3, newValue);
            return newValue;
        }
        let cache2 = cache1.get(a1);
        if (!cache2) {
            cache2 = new WeakMap();
            cache1.set(a1, cache2);
            const cache3 = new WeakMap();
            cache2.set(a2, cache3);
            const newValue = fn(a1, a2, a3);
            cache3.set(a3, newValue);
            return newValue;
        }
        let cache3 = cache2.get(a2);
        if (!cache3) {
            cache3 = new WeakMap();
            cache2.set(a2, cache3);
            const newValue = fn(a1, a2, a3);
            cache3.set(a3, newValue);
            return newValue;
        }
        const cachedValue = cache3.get(a3);
        if (cachedValue === undefined) {
            const newValue = fn(a1, a2, a3);
            cache3.set(a3, newValue);
            return newValue;
        }
        return cachedValue;
    }
    return memoized;
}
function memoize2(fn) {
    let cache1;
    function memoized(a1, a2) {
        if (!cache1) {
            cache1 = new WeakMap();
            const cache2 = new WeakMap();
            cache1.set(a1, cache2);
            const newValue = fn(a1, a2);
            cache2.set(a2, newValue);
            return newValue;
        }
        let cache2 = cache1.get(a1);
        if (!cache2) {
            cache2 = new WeakMap();
            cache1.set(a1, cache2);
            const newValue = fn(a1, a2);
            cache2.set(a2, newValue);
            return newValue;
        }
        const cachedValue = cache2.get(a2);
        if (cachedValue === undefined) {
            const newValue = fn(a1, a2);
            cache2.set(a2, newValue);
            return newValue;
        }
        return cachedValue;
    }
    return memoized;
}

function mergeProxiedResults(target, ...sources) {
    const results = sources.filter(source => !(source instanceof Error));
    const fieldSubschemaMap = results.reduce((acc, source) => {
        const subschema = source[OBJECT_SUBSCHEMA_SYMBOL];
        Object.keys(source).forEach(key => {
            acc[key] = subschema;
        });
        return acc;
    }, {});
    const result = results.reduce(mergeDeep, target);
    result[FIELD_SUBSCHEMA_MAP_SYMBOL] = target[FIELD_SUBSCHEMA_MAP_SYMBOL]
        ? Object.assign({}, target[FIELD_SUBSCHEMA_MAP_SYMBOL], fieldSubschemaMap)
        : fieldSubschemaMap;
    const errors = sources.map((source) => (source instanceof Error ? source : source[ERROR_SYMBOL]));
    result[ERROR_SYMBOL] = target[ERROR_SYMBOL].concat(...errors);
    return result;
}

const sortSubschemasByProxiability = memoize3(function (mergedTypeInfo, sourceSubschemaOrSourceSubschemas, targetSubschemas) {
    // 1.  calculate if possible to delegate to given subschema
    //    TODO: change logic so that required selection set can be spread across multiple subschemas?
    const proxiableSubschemas = [];
    const nonProxiableSubschemas = [];
    const sourceSubschemas = Array.isArray(sourceSubschemaOrSourceSubschemas)
        ? sourceSubschemaOrSourceSubschemas
        : [sourceSubschemaOrSourceSubschemas];
    targetSubschemas.forEach(t => {
        if (sourceSubschemas.some(s => mergedTypeInfo.containsSelectionSet.get(s).get(t))) {
            proxiableSubschemas.push(t);
        }
        else {
            nonProxiableSubschemas.push(t);
        }
    });
    return {
        proxiableSubschemas,
        nonProxiableSubschemas,
    };
});
const buildDelegationPlan = memoize3(function (mergedTypeInfo, fieldNodes, proxiableSubschemas) {
    const { uniqueFields, nonUniqueFields } = mergedTypeInfo;
    const unproxiableFieldNodes = [];
    // 2. for each selection:
    const delegationMap = new Map();
    fieldNodes.forEach(fieldNode => {
        if (fieldNode.name.value === '__typename') {
            return;
        }
        // 2a. use uniqueFields map to assign fields to subschema if one of possible subschemas
        const uniqueSubschema = uniqueFields[fieldNode.name.value];
        if (uniqueSubschema != null) {
            if (!proxiableSubschemas.includes(uniqueSubschema)) {
                unproxiableFieldNodes.push(fieldNode);
                return;
            }
            const existingSubschema = delegationMap.get(uniqueSubschema);
            if (existingSubschema != null) {
                existingSubschema.push(fieldNode);
            }
            else {
                delegationMap.set(uniqueSubschema, [fieldNode]);
            }
            return;
        }
        // 2b. use nonUniqueFields to assign to a possible subschema,
        //     preferring one of the subschemas already targets of delegation
        let nonUniqueSubschemas = nonUniqueFields[fieldNode.name.value];
        if (nonUniqueSubschemas == null) {
            unproxiableFieldNodes.push(fieldNode);
            return;
        }
        nonUniqueSubschemas = nonUniqueSubschemas.filter(s => proxiableSubschemas.includes(s));
        if (nonUniqueSubschemas == null) {
            unproxiableFieldNodes.push(fieldNode);
            return;
        }
        const subschemas = Array.from(delegationMap.keys());
        const existingSubschema = nonUniqueSubschemas.find(s => subschemas.includes(s));
        if (existingSubschema != null) {
            delegationMap.get(existingSubschema).push(fieldNode);
        }
        else {
            delegationMap.set(nonUniqueSubschemas[0], [fieldNode]);
        }
    });
    const finalDelegationMap = new Map();
    delegationMap.forEach((selections, subschema) => {
        finalDelegationMap.set(subschema, {
            kind: Kind.SELECTION_SET,
            selections,
        });
    });
    return {
        delegationMap: finalDelegationMap,
        unproxiableFieldNodes,
    };
});
const combineSubschemas = memoize2(function (subschemaOrSubschemas, additionalSubschemas) {
    return Array.isArray(subschemaOrSubschemas)
        ? subschemaOrSubschemas.concat(additionalSubschemas)
        : [subschemaOrSubschemas].concat(additionalSubschemas);
});
function mergeFields(mergedTypeInfo, typeName, object, fieldNodes, sourceSubschemaOrSourceSubschemas, targetSubschemas, context, info) {
    if (!fieldNodes.length) {
        return object;
    }
    const { proxiableSubschemas, nonProxiableSubschemas } = sortSubschemasByProxiability(mergedTypeInfo, sourceSubschemaOrSourceSubschemas, targetSubschemas);
    const { delegationMap, unproxiableFieldNodes } = buildDelegationPlan(mergedTypeInfo, fieldNodes, proxiableSubschemas);
    if (!delegationMap.size) {
        return object;
    }
    let containsPromises = false;
    const maybePromises = [];
    delegationMap.forEach((selectionSet, s) => {
        const maybePromise = s.merge[typeName].resolve(object, context, info, s, selectionSet);
        maybePromises.push(maybePromise);
        if (!containsPromises && maybePromise instanceof Promise) {
            containsPromises = true;
        }
    });
    return containsPromises
        ? Promise.all(maybePromises).then(results => mergeFields(mergedTypeInfo, typeName, mergeProxiedResults(object, ...results), unproxiableFieldNodes, combineSubschemas(sourceSubschemaOrSourceSubschemas, proxiableSubschemas), nonProxiableSubschemas, context, info))
        : mergeFields(mergedTypeInfo, typeName, mergeProxiedResults(object, ...maybePromises), unproxiableFieldNodes, combineSubschemas(sourceSubschemaOrSourceSubschemas, proxiableSubschemas), nonProxiableSubschemas, context, info);
}

function collectSubFields(info, typeName) {
    let subFieldNodes = Object.create(null);
    const visitedFragmentNames = Object.create(null);
    const type = info.schema.getType(typeName);
    const partialExecutionContext = {
        schema: info.schema,
        variableValues: info.variableValues,
        fragments: info.fragments,
    };
    info.fieldNodes.forEach(fieldNode => {
        subFieldNodes = collectFields(partialExecutionContext, type, fieldNode.selectionSet, subFieldNodes, visitedFragmentNames);
    });
    const stitchingInfo = info.schema.extensions.stitchingInfo;
    const selectionSetsByType = stitchingInfo.selectionSetsByType;
    const selectionSetsByField = stitchingInfo.selectionSetsByField;
    Object.keys(subFieldNodes).forEach(responseName => {
        var _a;
        const fieldName = subFieldNodes[responseName][0].name.value;
        const typeSelectionSet = selectionSetsByType[typeName];
        if (typeSelectionSet != null) {
            subFieldNodes = collectFields(partialExecutionContext, type, typeSelectionSet, subFieldNodes, visitedFragmentNames);
        }
        const fieldSelectionSet = (_a = selectionSetsByField === null || selectionSetsByField === void 0 ? void 0 : selectionSetsByField[typeName]) === null || _a === void 0 ? void 0 : _a[fieldName];
        if (fieldSelectionSet != null) {
            subFieldNodes = collectFields(partialExecutionContext, type, fieldSelectionSet, subFieldNodes, visitedFragmentNames);
        }
    });
    return subFieldNodes;
}
const getFieldsNotInSubschema = memoizeInfoAnd2Objectsand1Primitive(function (info, subschema, mergedTypeInfo, typeName) {
    const typeMap = isSubschemaConfig(subschema) ? mergedTypeInfo.typeMaps.get(subschema) : subschema.getTypeMap();
    const fields = typeMap[typeName].getFields();
    const subFieldNodes = collectSubFields(info, typeName);
    let fieldsNotInSchema = [];
    Object.keys(subFieldNodes).forEach(responseName => {
        const fieldName = subFieldNodes[responseName][0].name.value;
        if (!(fieldName in fields)) {
            fieldsNotInSchema = fieldsNotInSchema.concat(subFieldNodes[responseName]);
        }
    });
    return fieldsNotInSchema;
});

function handleObject(type, object, errors, subschema, context, info, skipTypeMerging) {
    var _a;
    const stitchingInfo = (_a = info === null || info === void 0 ? void 0 : info.schema.extensions) === null || _a === void 0 ? void 0 : _a.stitchingInfo;
    setErrors(object, errors.map(error => slicedError(error)));
    setObjectSubschema(object, subschema);
    if (skipTypeMerging || !stitchingInfo) {
        return object;
    }
    const typeName = isAbstractType(type) ? info.schema.getTypeMap()[object.__typename].name : type.name;
    const mergedTypeInfo = stitchingInfo.mergedTypes[typeName];
    let targetSubschemas;
    if (mergedTypeInfo != null) {
        targetSubschemas = mergedTypeInfo.targetSubschemas.get(subschema);
    }
    if (!targetSubschemas) {
        return object;
    }
    const fieldNodes = getFieldsNotInSubschema(info, subschema, mergedTypeInfo, typeName);
    return mergeFields(mergedTypeInfo, typeName, object, fieldNodes, subschema, targetSubschemas, context, info);
}

function handleList(type, list, errors, subschema, context, info, skipTypeMerging) {
    const childErrors = getErrorsByPathSegment(errors);
    return list.map((listMember, index) => handleListMember(getNullableType(type.ofType), listMember, index in childErrors ? childErrors[index] : [], subschema, context, info, skipTypeMerging));
}
function handleListMember(type, listMember, errors, subschema, context, info, skipTypeMerging) {
    if (listMember == null) {
        return handleNull(errors);
    }
    if (isLeafType(type)) {
        return type.parseValue(listMember);
    }
    else if (isCompositeType(type)) {
        return handleObject(type, listMember, errors, subschema, context, info, skipTypeMerging);
    }
    else if (isListType(type)) {
        return handleList(type, listMember, errors, subschema, context, info, skipTypeMerging);
    }
}

function handleResult(result, errors, subschema, context, info, returnType = info.returnType, skipTypeMerging) {
    const type = getNullableType(returnType);
    if (result == null) {
        return handleNull(errors);
    }
    if (isLeafType(type)) {
        return type.parseValue(result);
    }
    else if (isCompositeType(type)) {
        return handleObject(type, result, errors, subschema, context, info, skipTypeMerging);
    }
    else if (isListType(type)) {
        return handleList(type, result, errors, subschema, context, info, skipTypeMerging);
    }
}

class CheckResultAndHandleErrors {
    constructor(info, fieldName, subschema, context, returnType = info.returnType, typeMerge) {
        this.context = context;
        this.info = info;
        this.fieldName = fieldName;
        this.subschema = subschema;
        this.returnType = returnType;
        this.typeMerge = typeMerge;
    }
    transformResult(result) {
        return checkResultAndHandleErrors(result, this.context != null ? this.context : {}, this.info, this.fieldName, this.subschema, this.returnType, this.typeMerge);
    }
}
function checkResultAndHandleErrors(result, context, info, responseKey = getResponseKeyFromInfo(info), subschema, returnType = info.returnType, skipTypeMerging) {
    const errors = result.errors != null ? result.errors : [];
    const data = result.data != null ? result.data[responseKey] : undefined;
    return handleResult(data, errors, subschema, context, info, returnType, skipTypeMerging);
}

class AddArgumentsAsVariables {
    constructor(targetSchema, args) {
        this.targetSchema = targetSchema;
        this.args = Object.entries(args).reduce((prev, [key, val]) => ({
            ...prev,
            [key]: val,
        }), {});
    }
    transformRequest(originalRequest) {
        const { document, variables } = addVariablesToRootField(this.targetSchema, originalRequest, this.args);
        return {
            ...originalRequest,
            document,
            variables,
        };
    }
}
function addVariablesToRootField(targetSchema, originalRequest, args) {
    const document = originalRequest.document;
    const variableValues = originalRequest.variables;
    const operations = document.definitions.filter(def => def.kind === Kind.OPERATION_DEFINITION);
    const fragments = document.definitions.filter(def => def.kind === Kind.FRAGMENT_DEFINITION);
    const newOperations = operations.map((operation) => {
        const variableDefinitionMap = operation.variableDefinitions.reduce((prev, def) => ({
            ...prev,
            [def.variable.name.value]: def,
        }), {});
        let type;
        if (operation.operation === 'subscription') {
            type = targetSchema.getSubscriptionType();
        }
        else if (operation.operation === 'mutation') {
            type = targetSchema.getMutationType();
        }
        else {
            type = targetSchema.getQueryType();
        }
        const newSelectionSet = [];
        operation.selectionSet.selections.forEach((selection) => {
            if (selection.kind === Kind.FIELD) {
                const argumentNodes = selection.arguments;
                const argumentNodeMap = argumentNodes.reduce((prev, argument) => ({
                    ...prev,
                    [argument.name.value]: argument,
                }), {});
                const targetField = type.getFields()[selection.name.value];
                // excludes __typename
                if (targetField != null) {
                    updateArguments(targetField, argumentNodeMap, variableDefinitionMap, variableValues, args);
                }
                newSelectionSet.push({
                    ...selection,
                    arguments: Object.keys(argumentNodeMap).map(argName => argumentNodeMap[argName]),
                });
            }
            else {
                newSelectionSet.push(selection);
            }
        });
        return {
            ...operation,
            variableDefinitions: Object.keys(variableDefinitionMap).map(varName => variableDefinitionMap[varName]),
            selectionSet: {
                kind: Kind.SELECTION_SET,
                selections: newSelectionSet,
            },
        };
    });
    return {
        document: {
            ...document,
            definitions: [...newOperations, ...fragments],
        },
        variables: variableValues,
    };
}
function updateArguments(targetField, argumentNodeMap, variableDefinitionMap, variableValues, newArgs) {
    targetField.args.forEach((argument) => {
        const argName = argument.name;
        const argType = argument.type;
        if (argName in newArgs) {
            updateArgument(argName, argType, argumentNodeMap, variableDefinitionMap, variableValues, serializeInputValue(argType, newArgs[argName]));
        }
    });
}

function defaultDelegationBinding(delegationContext) {
    var _a;
    const { subschema: schemaOrSubschemaConfig, targetSchema, fieldName, args, context, info, returnType, transforms = [], skipTypeMerging, } = delegationContext;
    const stitchingInfo = (_a = info === null || info === void 0 ? void 0 : info.schema.extensions) === null || _a === void 0 ? void 0 : _a.stitchingInfo;
    let transformedSchema = stitchingInfo === null || stitchingInfo === void 0 ? void 0 : stitchingInfo.transformedSchemas.get(schemaOrSubschemaConfig);
    if (transformedSchema != null) {
        delegationContext.transformedSchema = transformedSchema;
    }
    else {
        transformedSchema = delegationContext.transformedSchema;
    }
    let delegationTransforms = [
        new CheckResultAndHandleErrors(info, fieldName, schemaOrSubschemaConfig, context, returnType, skipTypeMerging),
    ];
    if (stitchingInfo != null) {
        delegationTransforms = delegationTransforms.concat([
            new AddSelectionSetsByField(info.schema, returnType, stitchingInfo.selectionSetsByType, stitchingInfo.selectionSetsByField, stitchingInfo.dynamicSelectionSetsByField),
            new WrapConcreteTypes(returnType, transformedSchema),
            new ExpandAbstractTypes(info.schema, transformedSchema),
        ]);
    }
    else if (info != null) {
        delegationTransforms = delegationTransforms.concat([
            new WrapConcreteTypes(returnType, transformedSchema),
            new ExpandAbstractTypes(info.schema, transformedSchema),
        ]);
    }
    else {
        delegationTransforms.push(new WrapConcreteTypes(returnType, transformedSchema));
    }
    delegationTransforms = delegationTransforms.concat(transforms.slice().reverse());
    if (stitchingInfo != null) {
        delegationTransforms.push(new AddFragmentsByField(targetSchema, stitchingInfo.fragmentsByField));
    }
    if (args != null) {
        delegationTransforms.push(new AddArgumentsAsVariables(targetSchema, args));
    }
    delegationTransforms = delegationTransforms.concat([
        new FilterToSchema(targetSchema),
        new AddTypenameToAbstract(targetSchema),
    ]);
    return delegationTransforms;
}

class Transformer {
    constructor(context, binding = defaultDelegationBinding) {
        this.transformations = [];
        this.delegationContext = context;
        const delegationTransforms = binding(this.delegationContext);
        delegationTransforms.forEach(transform => this.addTransform(transform, {}));
    }
    addTransform(transform, context = {}) {
        this.transformations.push({ transform, context });
    }
    transformRequest(originalRequest) {
        return this.transformations.reduce((request, transformation) => transformation.transform.transformRequest != null
            ? transformation.transform.transformRequest(request, this.delegationContext, transformation.context)
            : request, originalRequest);
    }
    transformResult(originalResult) {
        return this.transformations.reduceRight((result, transformation) => transformation.transform.transformResult != null
            ? transformation.transform.transformResult(result, this.delegationContext, transformation.context)
            : result, originalResult);
    }
}

function delegateToSchema(options) {
    if (isSchema(options)) {
        throw new Error('Passing positional arguments to delegateToSchema is deprecated. ' + 'Please pass named parameters instead.');
    }
    const { info, operationName, operation = getDelegatingOperation(info.parentType, info.schema), fieldName = info.fieldName, returnType = info.returnType, selectionSet, fieldNodes, } = options;
    const request = createRequestFromInfo({
        info,
        operation,
        fieldName,
        selectionSet,
        fieldNodes,
        operationName,
    });
    return delegateRequest({
        ...options,
        request,
        operation,
        fieldName,
        returnType,
    });
}
function getDelegationReturnType(targetSchema, operation, fieldName) {
    let rootType;
    if (operation === 'query') {
        rootType = targetSchema.getQueryType();
    }
    else if (operation === 'mutation') {
        rootType = targetSchema.getMutationType();
    }
    else {
        rootType = targetSchema.getSubscriptionType();
    }
    return rootType.getFields()[fieldName].type;
}
function delegateRequest({ request, schema: subschemaOrSubschemaConfig, rootValue, info, operation, fieldName, args, returnType, context, transforms = [], transformedSchema, skipValidation, skipTypeMerging, binding, }) {
    var _a, _b;
    let operationDefinition;
    let targetOperation;
    let targetFieldName;
    if (operation == null) {
        operationDefinition = getOperationAST(request.document, undefined);
        targetOperation = operationDefinition.operation;
    }
    else {
        targetOperation = operation;
    }
    if (fieldName == null) {
        operationDefinition = operationDefinition !== null && operationDefinition !== void 0 ? operationDefinition : getOperationAST(request.document, undefined);
        targetFieldName = operationDefinition.selectionSet.selections[0].name.value;
    }
    else {
        targetFieldName = fieldName;
    }
    let targetSchema;
    let targetRootValue;
    let subschemaConfig;
    let allTransforms;
    if (isSubschemaConfig(subschemaOrSubschemaConfig)) {
        subschemaConfig = subschemaOrSubschemaConfig;
        targetSchema = subschemaConfig.schema;
        targetRootValue = (_a = rootValue !== null && rootValue !== void 0 ? rootValue : subschemaConfig === null || subschemaConfig === void 0 ? void 0 : subschemaConfig.rootValue) !== null && _a !== void 0 ? _a : info === null || info === void 0 ? void 0 : info.rootValue;
        allTransforms =
            subschemaOrSubschemaConfig.transforms != null
                ? subschemaOrSubschemaConfig.transforms.concat(transforms)
                : transforms;
    }
    else {
        targetSchema = subschemaOrSubschemaConfig;
        targetRootValue = rootValue !== null && rootValue !== void 0 ? rootValue : info === null || info === void 0 ? void 0 : info.rootValue;
        allTransforms = transforms;
    }
    const delegationContext = {
        subschema: subschemaOrSubschemaConfig,
        targetSchema,
        operation: targetOperation,
        fieldName: targetFieldName,
        args,
        context,
        info,
        returnType: (_b = returnType !== null && returnType !== void 0 ? returnType : info === null || info === void 0 ? void 0 : info.returnType) !== null && _b !== void 0 ? _b : getDelegationReturnType(targetSchema, targetOperation, targetFieldName),
        transforms: allTransforms,
        transformedSchema: transformedSchema !== null && transformedSchema !== void 0 ? transformedSchema : targetSchema,
        skipTypeMerging,
    };
    const transformer = new Transformer(delegationContext, binding);
    const processedRequest = transformer.transformRequest(request);
    if (!skipValidation) {
        validateRequest(targetSchema, processedRequest.document);
    }
    if (targetOperation === 'query' || targetOperation === 'mutation') {
        const executor = (subschemaConfig === null || subschemaConfig === void 0 ? void 0 : subschemaConfig.executor) || createDefaultExecutor(targetSchema, (subschemaConfig === null || subschemaConfig === void 0 ? void 0 : subschemaConfig.rootValue) || targetRootValue);
        const executionResult = executor({
            document: processedRequest.document,
            variables: processedRequest.variables,
            context,
            info,
        });
        if (executionResult instanceof Promise) {
            return executionResult.then(originalResult => transformer.transformResult(originalResult));
        }
        return transformer.transformResult(executionResult);
    }
    const subscriber = (subschemaConfig === null || subschemaConfig === void 0 ? void 0 : subschemaConfig.subscriber) || createDefaultSubscriber(targetSchema, (subschemaConfig === null || subschemaConfig === void 0 ? void 0 : subschemaConfig.rootValue) || targetRootValue);
    return subscriber({
        document: processedRequest.document,
        variables: processedRequest.variables,
        context,
        info,
    }).then((subscriptionResult) => {
        if (Symbol.asyncIterator in subscriptionResult) {
            // "subscribe" to the subscription result and map the result through the transforms
            return mapAsyncIterator(subscriptionResult, originalResult => ({
                [targetFieldName]: transformer.transformResult(originalResult),
            }));
        }
        return transformer.transformResult(subscriptionResult);
    });
}
function validateRequest(targetSchema, document) {
    const errors = validate(targetSchema, document);
    if (errors.length > 0) {
        if (errors.length > 1) {
            const combinedError = new AggregateError(errors);
            throw combinedError;
        }
        const error = errors[0];
        throw error.originalError || error;
    }
}
function createDefaultExecutor(schema, rootValue) {
    return ({ document, context, variables, info }) => execute({
        schema,
        document,
        contextValue: context,
        variableValues: variables,
        rootValue: rootValue !== null && rootValue !== void 0 ? rootValue : info === null || info === void 0 ? void 0 : info.rootValue,
    });
}
function createDefaultSubscriber(schema, rootValue) {
    return ({ document, context, variables, info }) => subscribe({
        schema,
        document,
        contextValue: context,
        variableValues: variables,
        rootValue: rootValue !== null && rootValue !== void 0 ? rootValue : info === null || info === void 0 ? void 0 : info.rootValue,
    });
}

/**
 * Resolver that knows how to:
 * a) handle aliases for proxied schemas
 * b) handle errors from proxied schemas
 * c) handle external to internal enum coversion
 */
function defaultMergedResolver(parent, args, context, info) {
    if (!parent) {
        return null;
    }
    const responseKey = getResponseKeyFromInfo(info);
    const errors = getErrors(parent, responseKey);
    // check to see if parent is not a proxied result, i.e. if parent resolver was manually overwritten
    // See https://github.com/apollographql/graphql-tools/issues/967
    if (!errors) {
        return defaultFieldResolver(parent, args, context, info);
    }
    const result = parent[responseKey];
    const subschema = getSubschema(parent, responseKey);
    return handleResult(result, errors, subschema, context, info);
}

function unwrapResult(parent, path) {
    let newParent = parent;
    const pathLength = path.length;
    for (let i = 0; i < pathLength; i++) {
        const responseKey = path[i];
        const errors = getErrors(newParent, responseKey);
        const subschema = getSubschema(newParent, responseKey);
        const object = newParent[responseKey];
        if (object == null) {
            return handleNull(errors);
        }
        setErrors(object, errors.map(error => relocatedError(error, error.path != null ? error.path.slice(1) : undefined)));
        setObjectSubschema(object, subschema);
        newParent = object;
    }
    return newParent;
}
function dehoistResult(parent, delimeter = '__gqltf__') {
    const result = Object.create(null);
    Object.keys(parent).forEach(alias => {
        let obj = result;
        const fieldNames = alias.split(delimeter);
        const fieldName = fieldNames.pop();
        fieldNames.forEach(key => {
            obj = obj[key] = obj[key] || Object.create(null);
        });
        obj[fieldName] = parent[alias];
    });
    result[ERROR_SYMBOL] = parent[ERROR_SYMBOL].map((error) => {
        if (error.path != null) {
            const path = error.path.slice();
            const pathSegment = path.shift();
            const expandedPathSegment = pathSegment.split(delimeter);
            return relocatedError(error, expandedPathSegment.concat(path));
        }
        return error;
    });
    result[OBJECT_SUBSCHEMA_SYMBOL] = parent[OBJECT_SUBSCHEMA_SYMBOL];
    return result;
}
function createMergedResolver({ fromPath, dehoist, delimeter = '__gqltf__', }) {
    const parentErrorResolver = (parent, args, context, info) => parent instanceof Error ? parent : defaultMergedResolver(parent, args, context, info);
    const unwrappingResolver = fromPath != null
        ? (parent, args, context, info) => parentErrorResolver(unwrapResult(parent, fromPath), args, context, info)
        : parentErrorResolver;
    const dehoistingResolver = dehoist
        ? (parent, args, context, info) => unwrappingResolver(dehoistResult(parent, delimeter), args, context, info)
        : unwrappingResolver;
    const noParentResolver = (parent, args, context, info) => parent ? dehoistingResolver(parent, args, context, info) : {};
    return noParentResolver;
}

class AddSelectionSetsByField$1 {
    constructor(schema, mapping) {
        this.schema = schema;
        this.mapping = mapping;
    }
    transformRequest(originalRequest) {
        const document = addSelectionSetsByField(this.schema, originalRequest.document, this.mapping);
        return {
            ...originalRequest,
            document,
        };
    }
}
function addSelectionSetsByField(schema, document, mapping) {
    const typeInfo = new TypeInfo(schema);
    return visit(document, visitWithTypeInfo(typeInfo, {
        [Kind.SELECTION_SET](node) {
            const parentType = typeInfo.getParentType();
            if (parentType != null) {
                const parentTypeName = parentType.name;
                let selections = node.selections;
                if (parentTypeName in mapping) {
                    node.selections.forEach(selection => {
                        if (selection.kind === Kind.FIELD) {
                            const name = selection.name.value;
                            const selectionSet = mapping[parentTypeName][name];
                            if (selectionSet != null) {
                                selections = selections.concat(selectionSet.selections);
                            }
                        }
                    });
                }
                if (selections !== node.selections) {
                    return {
                        ...node,
                        selections,
                    };
                }
            }
        },
    }));
}

class AddSelectionSetsByType {
    constructor(targetSchema, mapping) {
        this.targetSchema = targetSchema;
        this.mapping = mapping;
    }
    transformRequest(originalRequest) {
        const document = addSelectionSetsByType(this.targetSchema, originalRequest.document, this.mapping);
        return {
            ...originalRequest,
            document,
        };
    }
}
function addSelectionSetsByType(targetSchema, document, mapping) {
    const typeInfo = new TypeInfo(targetSchema);
    return visit(document, visitWithTypeInfo(typeInfo, {
        [Kind.SELECTION_SET](node) {
            const parentType = typeInfo.getParentType();
            if (parentType != null) {
                const parentTypeName = parentType.name;
                let selections = node.selections;
                if (parentTypeName in mapping) {
                    const selectionSet = mapping[parentTypeName];
                    if (selectionSet != null) {
                        selections = selections.concat(selectionSet.selections);
                    }
                }
                if (selections !== node.selections) {
                    return {
                        ...node,
                        selections,
                    };
                }
            }
        },
    }));
}

class ReplaceFieldWithFragment {
    constructor(targetSchema, fragments) {
        this.targetSchema = targetSchema;
        this.mapping = {};
        for (const { field, fragment } of fragments) {
            const parsedFragment = parseFragmentToInlineFragment(fragment);
            const actualTypeName = parsedFragment.typeCondition.name.value;
            if (!(actualTypeName in this.mapping)) {
                this.mapping[actualTypeName] = Object.create(null);
            }
            const typeMapping = this.mapping[actualTypeName];
            if (!(field in typeMapping)) {
                typeMapping[field] = [parsedFragment];
            }
            else {
                typeMapping[field].push(parsedFragment);
            }
        }
    }
    transformRequest(originalRequest) {
        const document = replaceFieldsWithFragments(this.targetSchema, originalRequest.document, this.mapping);
        return {
            ...originalRequest,
            document,
        };
    }
}
function replaceFieldsWithFragments(targetSchema, document, mapping) {
    const typeInfo = new TypeInfo(targetSchema);
    return visit(document, visitWithTypeInfo(typeInfo, {
        [Kind.SELECTION_SET](node) {
            const parentType = typeInfo.getParentType();
            if (parentType != null) {
                const parentTypeName = parentType.name;
                let selections = node.selections;
                if (parentTypeName in mapping) {
                    node.selections.forEach(selection => {
                        if (selection.kind === Kind.FIELD) {
                            const name = selection.name.value;
                            const fragments = mapping[parentTypeName][name];
                            if (fragments != null && fragments.length > 0) {
                                const fragment = concatInlineFragments(parentTypeName, fragments);
                                selections = selections.concat(fragment);
                            }
                        }
                    });
                }
                if (selections !== node.selections) {
                    return {
                        ...node,
                        selections,
                    };
                }
            }
        },
    }));
}
function parseFragmentToInlineFragment(definitions) {
    if (definitions.trim().startsWith('fragment')) {
        const document = parse(definitions);
        for (const definition of document.definitions) {
            if (definition.kind === Kind.FRAGMENT_DEFINITION) {
                return {
                    kind: Kind.INLINE_FRAGMENT,
                    typeCondition: definition.typeCondition,
                    selectionSet: definition.selectionSet,
                };
            }
        }
    }
    const query = parse(`{${definitions}}`).definitions[0];
    for (const selection of query.selectionSet.selections) {
        if (selection.kind === Kind.INLINE_FRAGMENT) {
            return selection;
        }
    }
    throw new Error('Could not parse fragment');
}

export { AddArgumentsAsVariables, AddFragmentsByField, AddSelectionSetsByType as AddMergedTypeSelectionSets, AddSelectionSetsByField as AddSelectionSets, AddSelectionSetsByField$1 as AddSelectionSetsByField, AddTypenameToAbstract, CheckResultAndHandleErrors, ExpandAbstractTypes, FilterToSchema, ReplaceFieldWithFragment, Subschema, VisitSelectionSets, checkResultAndHandleErrors, createMergedResolver, createRequest, createRequestFromInfo, defaultDelegationBinding, defaultMergedResolver, delegateRequest, delegateToSchema, getSubschema, handleResult, isSubschema, isSubschemaConfig };
//# sourceMappingURL=index.esm.js.map
