/* Copyright (c) 2020 Red Hat, Inc. */

import supertest from 'supertest';
import nock from 'nock';
import server, { GRAPHQL_PATH } from '../index';
import {
  kubeGetMock, mockAPIResourceList,
} from '../mocks/GenericResources';
import {
  mockManagedClusters, mockManagedClusterInfos,
  mockClusterList, mockNonClusterList,
} from '../mocks/Clusters';
import { mockPolicyListResponse } from '../mocks/PolicyList';
import ApiGroup from '../lib/ApiGroup';
import mockApplicationsList from '../mocks/Applications';

describe('Topology Filters Resolver', () => {
  beforeAll(() => {
    // specify the url to be intercepted
    const APIServer = nock('http://0.0.0.0/kubernetes');

    // define the method to be intercepted
    APIServer.get('/').reply(200, kubeGetMock);
    APIServer.get(`/apis/${ApiGroup.policiesGroup}/${ApiGroup.version}`).reply(200, mockAPIResourceList);
    APIServer.persist().get(/(\/apis\/internal.open-cluster-management.io\/v1beta1\/namespaces\/[\s\S]+\/managedclusterinfos\/)/)
      .reply(200, mockManagedClusterInfos);
    APIServer.persist().get(/(\/apis\/internal.open-cluster-management.io\/v1beta1\/namespaces\/[\s\S]+-[\s\S]+\/managedclusterinfos\/)/)
      .reply(200, mockManagedClusterInfos);
    APIServer.persist().get(/(\/apis\/cluster.open-cluster-management.io\/v1\/managedclusters\/[\s\S]+)/)
      .reply(200, mockManagedClusters);
    APIServer.persist().get(/(\/apis\/clusterregistry.k8s.io\/v1alpha1\/namespaces\/[\s\S]+\/clusters\/)/)
      .reply(200, mockNonClusterList);
    APIServer.persist().get(/(\/apis\/clusterregistry.k8s.io\/v1alpha1\/namespaces\/[\s\S]+-[\s\S]+\/clusters\/)/)
      .reply(200, mockClusterList);
    APIServer.get(/(\/apis\/app.k8s.io\/v1beta1\/namespaces\/[\s\S]+\/applications)/)
      .reply(200, mockApplicationsList);
    APIServer.get(/(\/apis\/app.k8s.io\/v1beta1\/namespaces\/[\s\S]+-[\s\S]+\/applications)/)
      .reply(200, mockApplicationsList);
    APIServer.persist().get(/(\/apis\/policy.open-cluster-management.io\/v1\/namespaces\/[\s\S]+\/policies)/)
      .reply(200, mockPolicyListResponse);
    APIServer.persist().get(/(\/apis\/policy.open-cluster-management.io\/v1\/namespaces\/[\s\S]+-[\s\S]+\/policies)/)
      .reply(200, mockPolicyListResponse);
  });

  test('Correctly Resolves HCMTopology case', () => new Promise((done) => {
    supertest(server)
      .post(GRAPHQL_PATH)
      .send({
        query: `
          query getTopologyFilters($topologyResourceType: String) {
            topologyFilters(topologyResourceType: $topologyResourceType) {
              filters {
                name
                namespace
                uid
                labels
                annotations
              }
            }
          }
        `,
        variables: {
          topologyResourceType: 'HCMTopology',
        },
      })
      .end((err, res) => {
        expect(JSON.parse(res.text)).toMatchSnapshot();
        done();
      });
  }));
  // this case is called from the ui, but doesn't actually do anything
  test('Correctly Resolves HCMCluster case', () => new Promise((done) => {
    supertest(server)
      .post(GRAPHQL_PATH)
      .send({
        query: `
          query getTopologyFilters($topologyResourceType: String) {
            topologyFilters(topologyResourceType: $topologyResourceType) {
              filters {
                name
                namespace
                uid
                labels
                annotations
              }
            }
          }
        `,
        variables: {
          topologyResourceType: 'HCMCluster',
        },
      })
      .end((err, res) => {
        expect(JSON.parse(res.text)).toMatchSnapshot();
        done();
      });
  }));
  test('Correctly Resolves HCMApplication case', () => new Promise((done) => {
    supertest(server)
      .post(GRAPHQL_PATH)
      .send({
        query: `
          query getTopologyFilters($topologyResourceType: String) {
            topologyFilters(topologyResourceType: $topologyResourceType) {
              filters {
                name
                namespace
                uid
                labels
                annotations
              }
            }
          }
        `,
        variables: {
          topologyResourceType: 'HCMApplication',
        },
      })
      .end((err, res) => {
        expect(JSON.parse(res.text)).toMatchSnapshot();
        done();
      });
  }));
  test('Correctly Resolves HCMPolicy case', () => new Promise((done) => {
    supertest(server)
      .post(GRAPHQL_PATH)
      .send({
        query: `
          query getTopologyFilters($topologyResourceType: String) {
            topologyFilters(topologyResourceType: $topologyResourceType) {
              filters {
                name
                namespace
                uid
                labels
                annotations
              }
            }
          }
        `,
        variables: {
          topologyResourceType: 'HCMPolicy',
        },
      })
      .end((err, res) => {
        expect(JSON.parse(res.text)).toMatchSnapshot();
        done();
      });
  }));
});
