/* Copyright (c) 2020 Red Hat, Inc. */

import supertest from 'supertest';
import nock from 'nock';
import server, { GRAPHQL_PATH } from '../index';
import {
  kubeGetMock, mockAPIResourceList,
  mockCreateResourcesResponse, mockUpdateResourcesResponse,
} from '../mocks/GenericResources';
import { mockManagedClusters, mockManagedClusterInfos, mockClusterList } from '../mocks/Clusters';
import { mockPolicyListResponse, mockSinglePolicyResponse, mockPlacementRuleResponse } from '../mocks/PolicyList';
import ApiGroup from '../lib/ApiGroup';

describe('Topology Resolver', () => {
  beforeAll(() => {
    // specify the url to be intercepted
    const APIServer = nock('http://0.0.0.0/kubernetes');

    // define the method to be intercepted
    APIServer.get('/').reply(200, kubeGetMock);
    APIServer.get(`/apis/${ApiGroup.policiesGroup}/${ApiGroup.version}`).reply(200, mockAPIResourceList);
    APIServer.persist().get(/(\/apis\/internal.open-cluster-management.io\/v1beta1\/namespaces\/[\s\S]+\/managedclusterinfos\/)/)
      .reply(200, mockManagedClusterInfos);
    APIServer.persist().get(/(\/apis\/internal.open-cluster-management.io\/v1beta1\/namespaces\/[\s\S]+-[\s\S]+\/managedclusterinfos\/)/)
      .reply(200, mockManagedClusterInfos);
    APIServer.persist().get(/(\/apis\/cluster.open-cluster-management.io\/v1\/managedclusters\/[\s\S]+)/)
      .reply(200, mockManagedClusters);
    APIServer.persist().get(/(\/apis\/clusterregistry.k8s.io\/v1alpha1\/namespaces\/[\s\S]+\/clusters\/)/)
      .reply(200, mockClusterList);
    APIServer.persist().get(/(\/apis\/clusterregistry.k8s.io\/v1alpha1\/namespaces\/[\s\S]+-[\s\S]+\/clusters\/)/)
      .reply(200, mockClusterList);
    APIServer.get(/(\/apis\/policy.open-cluster-management.io\/v1\/namespaces\/[\s\S]+\/policies\/policy-pod-1592668711636)/)
      .reply(200, mockSinglePolicyResponse);
    APIServer.persist().get(/(\/apis\/apps.open-cluster-management.io\/v1\/namespaces\/[\s\S]+\/placementrules)/)
      .reply(200, mockPlacementRuleResponse);
    APIServer.persist().get(/(\/apis\/apps.open-cluster-management.io\/v1\/namespaces\/[\s\S]+-[\s\S]+\/placementrules)/)
      .reply(200, mockPlacementRuleResponse);
    APIServer.post(`/apis/${ApiGroup.policiesGroup}/${ApiGroup.version}/namespaces/mcm/policies`)
      .reply(200, mockCreateResourcesResponse);
    APIServer.put(`/apis/${ApiGroup.policiesGroup}/${ApiGroup.version}/mcm/compliances/test-policy`)
      .reply(200, mockUpdateResourcesResponse);
    APIServer.persist().get(/(\/apis\/policy.open-cluster-management.io\/v1\/namespaces\/[\s\S]+\/policies)/)
      .reply(200, mockPolicyListResponse);
    APIServer.persist().get(/(\/apis\/policy.open-cluster-management.io\/v1\/namespaces\/[\s\S]+-[\s\S]+\/policies)/)
      .reply(200, mockPolicyListResponse);
  });

  test('Correctly Resolves HCMCluster case', () => new Promise((done) => {
    supertest(server)
      .post(GRAPHQL_PATH)
      .send({
        query: `
          query getTopology($topologyResourceType: String) {
            topology(topologyResourceType: $topologyResourceType) {
              resources {
                id
                uid
                name
                shortName
                cluster
                clusterName
                type
                specs
                namespace
                topology
                labels {
                  name
                  value
                }
              }
              relationships {
                type
                specs
                to {
                  uid
                }
                from {
                  uid
                }
              }
            }
          }
        `,
        variables: {
          topologyResourceType: 'HCMCluster',
        },
      })
      .end((err, res) => {
        expect(JSON.parse(res.text)).toMatchSnapshot();
        done();
      });
  }));

  test('Correctly Resolves HCMPolicy case', () => new Promise((done) => {
    supertest(server)
      .post(GRAPHQL_PATH)
      .send({
        query: `
          query getTopology($filter: TopologyFilter, $topologyResourceType: String) {
            topology(filter: $filter, topologyResourceType: $topologyResourceType) {
              resources {
                id
                uid
                name
                shortName
                cluster
                clusterName
                type
                specs
                namespace
                topology
                labels {
                  name
                  value
                }
              }
              relationships {
                type
                specs
                to {
                  uid
                }
                from {
                  uid
                }
              }
            }
          }
        `,
        variables: {
          topologyResourceType: 'HCMPolicy',
          filter: {
            policy: [
              {
                annotations: {},
                name: 'policy-pod-1592668711636',
                namespace: 'default',
                uid: '89d773e9-52d1-4b4d-9e51-2dad355f33fc',
              },
            ],
          },
        },
      })
      .end((err, res) => {
        expect(JSON.parse(res.text)).toMatchSnapshot();
        done();
      });
  }));
});
