'use strict'
/* eslint quote-props:0 */
Object.defineProperty(exports, '__esModule', { value: true })
exports.getType = exports.typeMap = exports.isEmptyKeys = exports.isEmptyWeakMap = exports.isEmptyMap = exports.isEmptyPlainObject = exports.isEmptyArray = exports.isNullish = exports.isWeakMap = exports.isMap = exports.isUndefined = exports.isNull = exports.isBoolean = exports.isString = exports.isNumber = exports.isArray = exports.isRegExp = exports.isFunction = exports.isAsyncFunction = exports.isSyncFunction = exports.isArguments = exports.isDate = exports.isError = exports.isClass = exports.isConventionalClass = exports.isNativeClass = exports.isPlainObject = exports.isObject = exports.getObjectType = void 0
// Prepare
const isClassRegex = /^class\s|^function\s+[A-Z]/
const isConventionalClassRegex = /^function\s+[A-Z]/
const isNativeClassRegex = /^class\s/
// -----------------------------------
// Values
/** Get the object type string */
function getObjectType(value) {
	return Object.prototype.toString.call(value)
}
exports.getObjectType = getObjectType
/** Checks to see if a value is an object */
function isObject(value) {
	// null is object, hence the extra check
	return value !== null && typeof value === 'object'
}
exports.isObject = isObject
/** Checks to see if a value is an object and only an object */
function isPlainObject(value) {
	/* eslint no-proto:0 */
	return isObject(value) && value.__proto__ === Object.prototype
}
exports.isPlainObject = isPlainObject
/**
 * Is ES6+ class */
function isNativeClass(value) {
	// NOTE TO DEVELOPER: If any of this changes, isClass must also be updated
	return (
		typeof value === 'function' && isNativeClassRegex.test(value.toString())
	)
}
exports.isNativeClass = isNativeClass
/**
 * Is Conventional Class
 export * Looks for function with capital first letter MyClass
 * First letter is the 9th character
 * If changed, isClass must also be updated */
function isConventionalClass(value) {
	return (
		typeof value === 'function' &&
		isConventionalClassRegex.test(value.toString())
	)
}
exports.isConventionalClass = isConventionalClass
function isClass(value) {
	return typeof value === 'function' && isClassRegex.test(value.toString())
}
exports.isClass = isClass
/** Checks to see if a value is an error */
function isError(value) {
	return value instanceof Error
}
exports.isError = isError
/** Checks to see if a value is a date */
function isDate(value) {
	return getObjectType(value) === '[object Date]'
}
exports.isDate = isDate
/** Checks to see if a value is an arguments object */
function isArguments(value) {
	return getObjectType(value) === '[object Arguments]'
}
exports.isArguments = isArguments
/**
 export * Checks to see if a value is a function but not an asynchronous function */
function isSyncFunction(value) {
	return getObjectType(value) === '[object Function]'
}
exports.isSyncFunction = isSyncFunction
/** Checks to see if a value is an asynchronous function */
function isAsyncFunction(value) {
	return getObjectType(value) === '[object AsyncFunction]'
}
exports.isAsyncFunction = isAsyncFunction
/** Checks to see if a value is a function */
function isFunction(value) {
	return isSyncFunction(value) || isAsyncFunction(value)
}
exports.isFunction = isFunction
/** Checks to see if a value is an regex */
function isRegExp(value) {
	return getObjectType(value) === '[object RegExp]'
}
exports.isRegExp = isRegExp
/** Checks to see if a value is an array */
function isArray(value) {
	return (
		(typeof Array.isArray === 'function' && Array.isArray(value)) ||
		getObjectType(value) === '[object Array]'
	)
}
exports.isArray = isArray
/** Checks to see if a valule is a number */
function isNumber(value) {
	return typeof value === 'number' || getObjectType(value) === '[object Number]'
}
exports.isNumber = isNumber
/** Checks to see if a value is a string */
function isString(value) {
	return typeof value === 'string' || getObjectType(value) === '[object String]'
}
exports.isString = isString
/** Checks to see if a valule is a boolean */
function isBoolean(value) {
	return (
		value === true ||
		value === false ||
		getObjectType(value) === '[object Boolean]'
	)
}
exports.isBoolean = isBoolean
/** Checks to see if a value is null */
function isNull(value) {
	return value === null
}
exports.isNull = isNull
/** Checks to see if a value is undefined */
function isUndefined(value) {
	return typeof value === 'undefined'
}
exports.isUndefined = isUndefined
/** Checks to see if a value is a Map */
function isMap(value) {
	return getObjectType(value) === '[object Map]'
}
exports.isMap = isMap
/** Checks to see if a value is a WeakMap */
function isWeakMap(value) {
	return getObjectType(value) === '[object WeakMap]'
}
exports.isWeakMap = isWeakMap
// -----------------------------------
// Empty
/** Checks to see if a value is nullish */
function isNullish(value) {
	return value == null
}
exports.isNullish = isNullish
/**
 * Is empty array
 * Will throw if the value was not an array
 */
function isEmptyArray(value) {
	if (!isArray(value)) throw new Error('value was not an array')
	return value.length === 0
}
exports.isEmptyArray = isEmptyArray
/**
 * Is empty plain object
 * Will throw if the value was not a plain object
 */
function isEmptyPlainObject(value) {
	if (!isPlainObject(value)) throw new Error('value was not a plain object')
	// We could use Object.keys, but this is more effecient
	for (const key in value) {
		if (value.hasOwnProperty(key)) {
			return false
		}
	}
	return true
}
exports.isEmptyPlainObject = isEmptyPlainObject
/**
 * Is empty map
 * Will throw if the value was not a map
 */
function isEmptyMap(value) {
	if (!isMap(value)) throw new Error('value was not a map')
	for (const key of value) {
		return false
	}
	return true
}
exports.isEmptyMap = isEmptyMap
/**
 * Is empty weak map
 * Will throw if the value was not a weak map
 */
function isEmptyWeakMap(value) {
	if (!isWeakMap(value)) throw new Error('value was not a weak map')
	for (const key of value) {
		return false
	}
	return true
}
exports.isEmptyWeakMap = isEmptyWeakMap
/**
 * Is empty keys
 */
function isEmptyKeys(value) {
	return Object.keys(value).length === 0
}
exports.isEmptyKeys = isEmptyKeys
// -----------------------------------
// General
/**
 * The default {@link TypeMap} for {@link getType}.
 export * AsyncFunction and SyncFunction are missing, as they are more specific types that people can detect afterwards.
 * @readonly
 */
exports.typeMap = Object.freeze({
	array: isArray,
	boolean: isBoolean,
	date: isDate,
	error: isError,
	class: isClass,
	function: isFunction,
	null: isNull,
	number: isNumber,
	regexp: isRegExp,
	string: isString,
	undefined: isUndefined,
	map: isMap,
	weakmap: isWeakMap,
	object: isObject,
})
/**
 * Cycle through the passed {@link TypeMap} testing the value, returning the first type that passes, otherwise `null`.
 * @param value the value to test
 * @param customTypeMap defaults to {@link typeMap}
 */
function getType(value, customTypeMap = exports.typeMap) {
	// Cycle through our type map
	for (const key in customTypeMap) {
		if (customTypeMap.hasOwnProperty(key)) {
			if (customTypeMap[key](value)) {
				return key
			}
		}
	}
	// No type was successful
	return null
}
exports.getType = getType
