/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

import _ from 'lodash'

export const getInitialBuffer = (fetchFilters, nodes, links) => {
  const nodeMap = {}
  nodes.forEach(n=>{
    delete n.latency
    nodeMap[n.uid] = n
  })
  return {
    nodes: _.cloneDeep(nodes),
    links: links.filter(l=>{
      delete l.latency
      return l.from && l.to && nodeMap[l.from.uid] && nodeMap[l.to.uid]
    }),
    fetchFilters: _.cloneDeep(fetchFilters),
    hadNodes: nodes.length>0
  }
}
// compensate for spotty weave data
export const getBufferedResponse = (state, action) => {
  const { fetchFilters } = action
  const { topologyResourceType } = state
  let { buffer={} } = state
  let { nodes=[], links=[] } = action
  // if no weave, no buffer needed
  if (topologyResourceType!=='HCMTopology') {
    return { nodes, links }
  }

  if (!_.isEqual(buffer.fetchFilters, fetchFilters)) {
    buffer = {fetchFilters}
  }
  // weave scans can miss all nodes between scans
  buffer.hadNodes = buffer.hadNodes || nodes.length>0
  if (buffer.hadNodes && nodes.length===0) {
    if (buffer.latency===undefined) {
      buffer.latency = 6
    }
    buffer.latency -= 1
    // give it 3 scans where all objects are missing before we refresh topology with nothing
    if (buffer.latency>0) {
      return { nodes: buffer.nodes, links: buffer.links, buffer }
    }
  }
  delete buffer.latency
  // buffer nodes
  const nodeMap = _.keyBy(nodes, 'uid')
  bufferElements(nodeMap, _.keyBy(buffer.nodes, 'uid'), 2)
  nodes = Object.values(nodeMap)
  // buffer links
  const linkMap = {}
  links.forEach(l=>{
    l.uid = l.from.uid+l.to.uid
    linkMap[l.uid] = l
  })
  bufferElements(linkMap, _.keyBy(buffer.links, 'uid'), 10)
  links = Object.values(linkMap)
  // update buffer
  buffer.nodes = _.cloneDeep(nodes)
  buffer.links = _.cloneDeep(links)
  return { nodes, links, buffer }
}

// weave scans can:
//  1) include multiple copies of the same node
//  2) miss some or ALL nodes between scans
// previous nodes can come from state buffer or from sessionStore
const bufferElements = (map, prevMap, latency) => {
  for (const uid in prevMap) {
    if (Object.prototype.hasOwnProperty.call(prevMap, uid)) {
      const prev = prevMap[uid]
      const isLive = !prev.specs || prev.specs.isLive
      if (!map[uid] && isLive) {
        // if missing in this scan, see if it reappears in the next 2 minutes
        if (prev.latency===undefined) {
          prev.latency = {count: 3, time: (new Date()).getTime()}
        }
        prev.latency.count--
        if (prev.latency.count>0 || ((new Date()).getTime() - prev.latency.time < latency*60*1000)) {
          map[uid] = prev
        }
      } else {
        // if it's back, forget it was ever gone
        delete prev.latency
      }
    }
  }
}
