/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2018. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
import lodash from 'lodash'
import {RESOURCE_TYPES} from '../../lib/shared/constants'
import GrcApolloClient from '../../lib/client/apollo-client'
import {
  RESOURCE_DETAILS_REQUEST, RESOURCE_DETAILS_RECEIVE_SUCCESS,
  RESOURCE_RECEIVE_SUCCESS, REQUEST_STATUS, RESOURCE_RECEIVE_FAILURE,
  RESOURCE_REQUEST, TOPOLOGY_RESTORE_SAVED_STATE, TOPOLOGY_FILTERS_UPDATE,
  TOPOLOGY_RESOURCE_TYPE_UPDATE, TOPOLOGY_FILTERS_RECEIVE_ERROR,
  TOPOLOGY_FILTERS_REQUEST, TOPOLOGY_FILTERS_RECEIVE_SUCCESS
} from './index'

export const requestResource = (resourceType, fetchFilters) => ({
  type: RESOURCE_REQUEST,
  status: REQUEST_STATUS.IN_PROGRESS,
  resourceType,
  fetchFilters
})

export const receiveResourceError = (err, resourceType) => ({
  type: RESOURCE_RECEIVE_FAILURE,
  status: REQUEST_STATUS.ERROR,
  err,
  resourceType
})


export const receiveTopologySuccess = (  response,  resourceType,  fetchFilters,  willLoadDetails) => ({
  type: RESOURCE_RECEIVE_SUCCESS,
  status: REQUEST_STATUS.DONE,
  nodes: response.resources || [],
  links: response.relationships || [],
  filters: {
    clusters: response.clusters,
    labels: response.labels,
    namespaces: response.namespaces,
    types: response.resourceTypes
  },
  resourceType,
  fetchFilters,
  willLoadDetails,
})

export const requestResourceDetails = (resourceType, detailsFetchFilters, reloading) => ({
  type: RESOURCE_DETAILS_REQUEST,
  status: REQUEST_STATUS.IN_PROGRESS,
  resourceType,
  detailsFetchFilters,
  reloading
})

export const receiveTopologyDetailsSuccess = (
  response,
  resourceType,
  detailsFetchFilters
) => ({
  type: RESOURCE_DETAILS_RECEIVE_SUCCESS,
  pods: response.pods || [],
  resourceType,
  detailsFetchFilters
})

export const fetchTopology = (vars, fetchFilters, reloading) => {
  const resourceType = RESOURCE_TYPES.HCM_TOPOLOGY
  return (dispatch) => {
    dispatch(requestResource(resourceType, fetchFilters))
    return GrcApolloClient.get(resourceType, vars)
      .then(response => {
        if (response.errors) {
          return dispatch(receiveResourceError(response.errors[0], resourceType))
        }
        const topology = {
          clusters: lodash.cloneDeep(response.data.clusters),
          labels: lodash.cloneDeep(response.data.labels),
          namespaces: lodash.cloneDeep(response.data.namespaces),
          resourceTypes: lodash.cloneDeep(response.data.resourceTypes),
          resources: lodash.cloneDeep(response.data.topology.resources),
          relationships: lodash.cloneDeep(response.data.topology.relationships),
        }

        // if there's a cluster, also fetch details
        let clusterNames=[]
        const clusterName = lodash.get(fetchFilters, 'cluster[0].name')
        if (clusterName) {
          // else if fetching from weave, get cluster name from fetchFilters
          clusterNames = [clusterName]
        } else {
          const topologyResIndex = topology.resources.findIndex(
            ({type})=>{
              return type==='pod'
            })
          if (topologyResIndex !== -1) {
            topology.resources.forEach(res=>{
              if (res.type==='cluster') {
                clusterNames = [...clusterNames, ...lodash.get(res, 'specs.clusterNames', [])]
              }
            })
          }
        }
        const loadDetails = clusterNames.length>0
        dispatch(receiveTopologySuccess(topology, resourceType, fetchFilters, loadDetails))
        if (loadDetails) {
          const detailsFetchFilters = lodash.cloneDeep(fetchFilters)
          dispatch(requestResourceDetails(resourceType, detailsFetchFilters, reloading))
          GrcApolloClient
            .getResource({name: 'HCMTopologyDetails'}, {filter: {clusterNames}})
            .then(responseTemp => {
              if (!responseTemp.errors) {
                topology.pods = lodash.cloneDeep(responseTemp.data.topologyDetails.pods)
                dispatch(receiveTopologyDetailsSuccess(topology, resourceType, detailsFetchFilters))
              }
            })
            .catch(err => (err))
        }
      })
      .catch(err => dispatch(receiveResourceError(err, resourceType)))
  }
}

export const restoreSavedTopologyFilters = (namespace, name) => ({
  type: TOPOLOGY_RESTORE_SAVED_STATE,
  namespace,
  name,
})

export const updateTopologyFilters = (filterType, filters) => ({
  type: TOPOLOGY_FILTERS_UPDATE,
  filterType,
  filters,
})

export const updateTopologyResourceType = (topologyResourceType) => ({
  type: TOPOLOGY_RESOURCE_TYPE_UPDATE,
  topologyResourceType,
})

const  receiveFiltersError = (err) => ({
  type: TOPOLOGY_FILTERS_RECEIVE_ERROR,
  err
})

export const fetchTopologyFilters = (topologyResourceType) => {
  return (dispatch) => {
    dispatch({
      type: TOPOLOGY_FILTERS_REQUEST,
    })
    return GrcApolloClient.getTopologyFilters({topologyResourceType})
      .then(response => {
        if (response.errors) {
          return dispatch(receiveFiltersError(response.errors[0]))
        }
        dispatch({
          type: TOPOLOGY_FILTERS_RECEIVE_SUCCESS,
          filters: lodash.cloneDeep(response.data.topologyFilters),
          topologyResourceType
        })
      })
      .catch(err => {dispatch(receiveFiltersError(err))})
  }
}
