/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

import React from 'react'
import PropTypes from 'prop-types'
import { connect } from 'react-redux'
import { Breadcrumb, Tabs, Tab } from 'carbon-components-react'
import resources from '../../lib/shared/resources'
import { withRouter, Link } from 'react-router-dom'
import msgs from '../../nls/platform.properties'

resources(() => {
  require('../../scss/secondary-header.scss')
})

export class SecondaryHeader extends React.Component {
  constructor(props) {
    super(props)
    this.renderBreadCrumb = this.renderBreadCrumb.bind(this)
    this.renderTabs = this.renderTabs.bind(this)

    this.state = {
      shadowPresent: false
    }
  }

  componentDidMount() {
    window.addEventListener('scroll', this.listenToScroll)
  }

  componentWillUnmount() {
    window.removeEventListener('scroll', this.listenToScroll)
  }

  listenToScroll = () => {
    if (window.scrollY > .1 && this.state.shadowPresent === false) {
      this.setState({shadowPresent:true})
    }
    else if (window.scrollY <= .1 && this.state.shadowPresent === true) {
      this.setState({shadowPresent:false})
    }
  }

  render() {
    const { tabs, title, breadcrumbItems, description } = this.props
    const { locale } = this.context
    if ((tabs && tabs.length > 0) || (breadcrumbItems && breadcrumbItems.length > 0)) {
      return (
        <div className='secondary-header-wrapper' role='region' aria-label={title}>
          <div
            className={`secondary-header simple-header${
              this.state.shadowPresent ? '-with-shadow' : ''
            }${description ? ' special-layout': ''}`}
          >
            <header>
              <div className='bx--detail-page-header-content'>
                {breadcrumbItems &&
                  (
                    <Breadcrumb>
                      {this.renderBreadCrumb()}
                    </Breadcrumb>
                  )
                }
                {this.renderHeader()}
                {tabs && tabs.length > 0 &&
                  <Tabs selected={this.getSelectedTab() || 0} aria-label={`${title} ${msgs.get('tabs.label', locale)}`}>
                    {this.renderTabs()}
                  </Tabs>
                }
              </div>
            </header>
          </div>
        </div>
      )
    } else {
      return (
        <div
          className='secondary-header-wrapper-min'
          role='region'
          aria-label={`${title} ${msgs.get('secondaryHeader', locale)}`}
        >
          <div className={`secondary-header simple-header${this.state.shadowPresent ? '-with-shadow' : ''}`}>
            <h1 className='bx--detail-page-header-title'>{decodeURIComponent(title)}</h1>
          </div>
        </div>
      )
    }
  }

  renderHeader() {
    const { title:headerTitle, description, information} = this.props
    if (description) {
      return (
        <div className='bx--detail-page-header-title-container'>
          <h1 className='bx--detail-page-header-title'>{headerTitle}</h1>
          <div className='detail-page-header-title-button'
            tabIndex={0}
            role={'button'}
            onClick={description.action}
            onKeyPress={description.action}>
            <p>
              {description.display}
            </p>
          </div>
        </div>
      )
    } else {
      return (
        <div className='bx--detail-page-header-title-container'>
          <h1 className='bx--detail-page-header-title'>{headerTitle}</h1>
          {information &&<svg className='info-icon'>
            <use href={'#diagramIcons_info'} ></use>
          </svg>}
        </div>
      )
    }
  }

  renderBreadCrumb() {
    const { locale } = this.context
    const { breadcrumbItems } = this.props
    return breadcrumbItems && breadcrumbItems.map((breadcrumb, index) => {
      const key = `${breadcrumb}-${index}`
      return (
        <div key={key} className='bx--breadcrumb-item' title={decodeURIComponent(breadcrumb.label)}>
          {
            breadcrumb.handleClick
              ? <div className='breadcrumb-link'
                tabIndex={0}
                role={'button'}
                onClick={breadcrumb.handleClick}
                onKeyPress={breadcrumb.handleClick}>
                {msgs.get(breadcrumb.label, locale)}
              </div>
              : <Link to={breadcrumb.url || ''} className='bx--link'>
                {decodeURIComponent(breadcrumb.label)}
              </Link>
          }
        </div>
      )
    })
  }

  renderTabs() {
    const { tabs } = this.props,
          { locale } = this.context
    return tabs.map(tab => {
      return <Tab
        label={msgs.get(tab.label, locale)}
        key={tab.id}
        id={tab.id}
        href={tab.url}
        onClick={tab.handleClick ? tab.handleClick : this.clickTab.bind(this, tab.url)}
      />
    })
  }

  getSelectedTab() {
    const { tabs, location } = this.props
    const selectedTab = tabs.map((tab, index) => {
      tab.index = index
      return tab
    }).filter((tab, index) => {
      if (index !== 0) {
        return location.pathname.startsWith(tab.url)
      }
      return undefined
    })
    return selectedTab[0] && selectedTab[0].index
  }

  clickTab(url) {
    this.props.history.push(url)
    return url
  }
}

SecondaryHeader.contextTypes = {
  locale: PropTypes.string
}

SecondaryHeader.propTypes = {
  breadcrumbItems: PropTypes.array,
  description: PropTypes.oneOfType([PropTypes.object, PropTypes.string]),
  history: PropTypes.object.isRequired,
  information: PropTypes.oneOfType([PropTypes.object, PropTypes.string]),
  location: PropTypes.object,
  tabs: PropTypes.array,
  title: PropTypes.string,
}

const mapStateToProps = (state) => {
  return {
    title: state.secondaryHeader.title,
    tabs: state.secondaryHeader.tabs,
    breadcrumbItems: state.secondaryHeader.breadcrumbItems,
    links: state.secondaryHeader.links,
    refresh: state.secondaryHeader.refresh,
    role: state.role && state.role.role,
    description: state.secondaryHeader.description,
    information: state.secondaryHeader.information,
  }
}

export default withRouter(connect(mapStateToProps)(SecondaryHeader))
