/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2020. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

import React from 'react'
import ReactDOM from 'react-dom'
import PropTypes from 'prop-types'
import { connect } from 'react-redux'
import { withRouter } from 'react-router-dom'
import { Button, Icon, Loading } from 'carbon-components-react'
import { fetchResources } from '../../actions/common'
import { RESOURCE_TYPES } from '../../../lib/shared/constants'
import { REQUEST_STATUS } from '../../actions/index'
import { getClusterData, getPodData } from './utils'
import SearchName from '../Topology/viewer/SearchName'
import resources from '../../../lib/shared/resources'
import msgs from '../../../nls/platform.properties'
import config from '../../../lib/shared/config'

const topoBannerKey = 'acm-topology-banner-tip'
const topoBannerValue = 'topology_banner_tip'

resources(() => {
  require('./style.scss')
})

class TopologyCards extends React.Component {
  static propTypes = {
    QueryApplicationList: PropTypes.object,
    fetchApplications: PropTypes.func,
    locale: PropTypes.string.isRequired,
    portals: PropTypes.object,
    queryStatus: PropTypes.string
  }

  constructor (props) {
    super(props)
    this.state = {
      searchName: '',
    }
  }

  componentDidMount() {
    const {
      fetchApplications
    } = this.props

    fetchApplications()
  }

  render() {
    const { queryStatus, QueryApplicationList, locale } = this.props
    let appList = QueryApplicationList.items

    // Show search bar, banner and cards if there are apps
    if (appList &&
      appList.length > 0 &&
      queryStatus !== REQUEST_STATUS.INCEPTION &&
      queryStatus !== REQUEST_STATUS.IN_PROGRESS
    ) {
      const { searchName='' } = this.state

      // Filter the list of apps to display based on search bar input
      if (searchName !== '') {
        appList = appList.filter(app => {
          return app.name.includes(searchName)
        })
      }

      const createBannerStorage = (e, target) => {
        // create topoBannerValue to keep banner closed
        localStorage.setItem(topoBannerKey, topoBannerValue)
        // close banner
        document.getElementById(target).style.display = 'none'
      }

      const getBannerStorage = (key) => {
        return localStorage.getItem(key)
      }

      const openAppDetails = (e, appName, appNamespace) => {
        const redirectLink = (window.location.href).replace('/topology/', '/applications/')
        window.open(`${redirectLink}${appNamespace}/${appName}`, '_blank')
      }

      return (
        <div className="topologyContainer">
          {this.renderSearchName()}
          { (getBannerStorage(topoBannerKey) && getBannerStorage(topoBannerKey) === topoBannerValue)
            ? ''
            : <div className="infoBanner" id="banner">
              <img className='launch-banner-icon'
                src={`${config.contextPath}/graphics/launch-banner.svg`} alt="launch-banner-icon" />
              <div className="bannerTitle">{msgs.get('topology.banner.title', locale)}</div>
              <div className="bannerDescription">
                {msgs.get('topology.banner.description.line1', locale)}
                <br />
                {msgs.get('topology.banner.description.line2', locale)}
              </div>
              <div
                className="bannerDismissText"
                role="button"
                tabIndex="0"
                onClick={e => createBannerStorage(e, 'banner')}
                onKeyPress={e => createBannerStorage(e, 'banner')}
              >
                {msgs.get('topology.banner.dismiss.text', locale)}
              </div>
            </div>
            // show banner if there is no topoBannerValue in storage
          }

          {appList.length > 0 ?
            <div className="topologyCardsContainer">
              {Object.keys(appList).map(appIndex => {
                const appName = appList[appIndex].name
                const appNamespace = appList[appIndex].namespace
                let hasFailedClusters = false
                let hasFailedPods = false
                let failedClusterCount = 0
                let totalClusterCount = 0
                let failedPodCount = 0
                let totalPodCount = 0

                if (appList[appIndex].remoteSubscriptionStatusCount) {
                  const clusterData = getClusterData(
                    appList[appIndex].remoteSubscriptionStatusCount,
                    hasFailedClusters,
                    failedClusterCount,
                    totalClusterCount
                  )
                  hasFailedClusters = clusterData[0]
                  failedClusterCount = clusterData[1]
                  totalClusterCount = clusterData[2]
                }

                if (appList[appIndex].podStatusCount) {
                  const podData = getPodData(
                    appList[appIndex].podStatusCount,
                    hasFailedPods,
                    failedPodCount,
                    totalPodCount
                  )
                  hasFailedPods = podData[0]
                  failedPodCount = podData[1]
                  totalPodCount = podData[2]
                }

                return (
                  <div
                    className="appCard"
                    key={`card_${appName}_${appNamespace}`}
                    onClick={e => openAppDetails(e, appName, appNamespace)}
                    onKeyPress={e => openAppDetails(e, appName, appNamespace)}
                    tabIndex={0}
                    role="button"
                  >
                    <div className="cardHeader">
                      <div className="cardName">{appName}</div>
                      {hasFailedClusters || hasFailedPods
                        ? <Icon className="failedIcon" name="icon--warning--glyph" />
                        : ''}
                      <div className="cardNamespace">
                        {msgs.get('topology.card.namespace.label', locale)} {appNamespace}
                      </div>
                    </div>

                    <div className="cardContent">
                      <div className="cardClusterCount">
                        <span
                          className={hasFailedClusters ? 'failedCount failedResources' : 'failedCount'}
                        >
                          {failedClusterCount}
                        </span>
                        <span className="totalCount">/{totalClusterCount}</span>
                        <div className="cardClusterText">{msgs.get('topology.card.clusters.failed', locale)}</div>
                      </div>

                      <div className="cardPodCount">
                        <span
                          className={hasFailedPods ? 'failedCount failedResources' : 'failedCount'}
                        >
                          {failedPodCount}
                        </span>
                        <span className="totalCount">/{totalPodCount}</span>
                        <div className="cardPodText">{msgs.get('topology.card.pods.failed', locale)}</div>
                      </div>
                    </div>
                  </div>
                )
              })}
            </div> : ''
          }
        </div>
      )
    }
    // Show empty state if there are no apps
    else if (appList &&
      appList.length === 0  &&
      queryStatus !== REQUEST_STATUS.INCEPTION &&
      queryStatus !== REQUEST_STATUS.IN_PROGRESS
    ) {
      const openAppsPage = (e, params) => {
        window.open((window.location.href).replace('/topology/', `/${params}/`), '_blank')
      }
      const buttonName = msgs.get('topology.apps.button.text', locale)
      const descriptionLine2Start = msgs.get('topology.apps.empty.description.line2.1', locale)
      const descriptionLine2End = msgs.get('topology.apps.empty.description.line2.2', locale)

      return (
        <div className="topologyContainer">
          <div className="emptyTopoAppsContainer">
            <img className='empty-apps-icon'
              src={`${config.contextPath}/graphics/no-applications.png`} alt="empty-apps-icon" />
            <div className="emptyAppsTitle">{msgs.get('topology.apps.empty.title', locale)}</div>
            <div className="emptyAppsDescription">
              {msgs.get('topology.apps.empty.description.line1', locale)}
              <br />
              {descriptionLine2Start} <span id="text-highlight">{buttonName}</span> {descriptionLine2End}
            </div>
            <Button
              small
              className="apps-page-redirect"
              onClick={e => openAppsPage(e, 'applications')}
            >
              {buttonName}
            </Button>
          </div>
        </div>
      )
    }
    // Show loading spinner if data is being fetched
    else {
      return (
        <div className="topologyContainer">
          <Loading withOverlay={false} className="content-spinner" />
        </div>
      )
    }
  }

  renderSearchName() {
    const { portals={}, locale } = this.props
    const { searchTextbox } = portals
    if (searchTextbox) {
      const portal = document.getElementById(searchTextbox)
      if (portal) {
        const { searchName } = this.state
        return ReactDOM.createPortal(
          <SearchName
            searchName={searchName}
            onNameSearch={this.onNameSearch.bind(this)}
            locale={locale}
          />,
          portal
        )
      }
    }
    return null
  }

  onNameSearch(searchName) {
    this.setState({searchName})
  }
}

const mapStateToProps = state => {
  const { QueryApplicationList = {} } = state
  const queryStatus = QueryApplicationList.status

  return {
    queryStatus,
    QueryApplicationList
  }
}

const mapDispatchToProps = dispatch => {
  return {
    fetchApplications: () =>
      dispatch(fetchResources(RESOURCE_TYPES.QUERY_APPLICATIONS)),
  }
}

export default withRouter(connect(mapStateToProps, mapDispatchToProps)(TopologyCards))
