/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2018. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */

'use strict'

import React from 'react'
import PropTypes from 'prop-types'
import { Portals } from './constants.js'
import { connect } from 'react-redux'
import { withRouter } from 'react-router-dom'
import FilterableSelect from '../common/FilterableSelect'
import { updateActiveFilters } from '../../actions/common'
import resources from '../../../lib/shared/resources'
import { RESOURCE_TYPES } from '../../../lib/shared/constants'
import { getActiveTopologyFilters } from './helpers/topology-helper'
import msgs from '../../../nls/platform.properties'
import { fetchTopologyFilters, updateTopologyFilters } from '../../actions/topology'
import * as Actions from '../../actions'
import _ from 'lodash'
import _uniqueId from 'lodash/uniqueId'

resources(() => {
  require('../../../scss/topology-controls.scss')
  require('../../../scss/topology-diagram.scss')
})


class TopologyToolbar extends React.Component {
  static propTypes = {
    activeFilters: PropTypes.object,
    availableFilters: PropTypes.object,
    failure: PropTypes.bool,
    fetchFilters: PropTypes.func,
    fetching: PropTypes.bool,
    onSelectedFilterChange: PropTypes.func,
    topologyResourceType: PropTypes.string,
  }

  constructor (props) {
    super(props)
    this.onServerFilterChange = this.onServerFilterChange.bind(this)
  }

  UNSAFE_componentWillMount() {
    this.props.fetchFilters(this.props.topologyResourceType)
  }

  shouldComponentUpdate(nextProps){
    return !_.isEqual(this.props.activeFilters, nextProps.activeFilters) ||
    !_.isEqual(this.props.availableFilters, nextProps.availableFilters) ||
    !_.isEqual(this.props.fetching, nextProps.fetching) ||
    !_.isEqual(this.props.failure, nextProps.failure)
  }

  render() {
    const { locale } = this.context
    const { activeFilters, availableFilters, fetching, failure, topologyResourceType } = this.props

    let activeKey, availableKey
    switch (topologyResourceType) {
    default:
    case RESOURCE_TYPES.HCM_TOPOLOGY.name:
      activeKey='cluster'
      availableKey='clusters'
      break

    case RESOURCE_TYPES.HCM_APPLICATIONS.name:
      activeKey='application'
      availableKey='applications'
      break

    case RESOURCE_TYPES.HCM_POLICIES.name:
      activeKey='policy'
      availableKey='policies'
      break
    }
    const title = msgs.get(`topology.select.${activeKey}`, locale)
    const availableServerFilters = availableFilters[availableKey]
    let selectedServerFilter = activeFilters[activeKey] ? activeFilters[activeKey][0] : null
    if (selectedServerFilter) {
      selectedServerFilter = selectedServerFilter.name
    }

    // clear filters
    const clearFilters = []
    Object.keys(activeFilters).forEach(key=>{
      const activeSet = activeFilters[key]
      if (activeSet.size>0) {
        clearFilters.push(key)
      }
    })

    // no selection/filater bar on clusters tab
    const policiesTab = topologyResourceType === RESOURCE_TYPES.HCM_POLICIES.name
    return (
      <div className='topology-controls'>
        <div className='topology-control-container' >
          {/*pick cluster/application/policy/etc */}
          {policiesTab&&<FilterableSelect
            key={_uniqueId('key')}
            title={title}
            selectedFilter={selectedServerFilter}
            availableFilters={availableServerFilters}
            onChange={this.onServerFilterChange}
            fetching={fetching}
            failure={failure}
          />}

          {/*name search*/}
          <React.Fragment>
            <div id={Portals.searchTextbox} />
          </React.Fragment>
        </div>

        {/*type filter bar*/}
        {policiesTab&&<React.Fragment>
          <div id={Portals.typeFilterBar} />
        </React.Fragment>}
      </div>
    )
  }

  onServerFilterChange = (selection) => {
    const { topologyResourceType, onSelectedFilterChange } = this.props
    switch (topologyResourceType) {
    default:
    case RESOURCE_TYPES.HCM_TOPOLOGY.name:
      onSelectedFilterChange('cluster', [selection])
      break

    case RESOURCE_TYPES.HCM_APPLICATIONS.name:
      onSelectedFilterChange('application', [selection])
      break

    case RESOURCE_TYPES.HCM_POLICIES.name:
      onSelectedFilterChange('policy', [selection])
      break
    }
  }

}

TopologyToolbar.contextTypes = {
  locale: PropTypes.string
}

const mapStateToProps = (state) =>{
  const activeFilters = getActiveTopologyFilters(state)
  const { availableFilters = {}, filtersStatus } = state.topology
  const { activeFilters:activeViewFilters } = state.resourceToolbar
  return {
    activeFilters,  // other persisted filters
    activeViewFilters, // unpersisted filter view filters
    availableFilters,
    fetching: filtersStatus !== Actions.REQUEST_STATUS.DONE,
    failure: filtersStatus === Actions.REQUEST_STATUS.ERROR
  }
}
const mapDispatchToProps = (dispatch) => {
  return {
    fetchFilters: (topologyResourceType) => dispatch(fetchTopologyFilters(topologyResourceType)),
    updateActiveFilters: (activeFilters) => dispatch(updateActiveFilters(activeFilters)),
    onSelectedFilterChange: (filterType, filter) => {
      dispatch(updateTopologyFilters(filterType, filter))
    },
  }
}

export default withRouter(connect(mapStateToProps, mapDispatchToProps)(TopologyToolbar))
