/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2017, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

import React from 'react'
import PropTypes from 'prop-types'
import { connect } from 'react-redux'
import { Tabs, Tab } from 'carbon-components-react'
import { withRouter } from 'react-router-dom'
import classNames from 'classnames'
import * as Actions from '../../actions'
import { updateModal, updateSecondaryHeader } from '../../actions/common'
import msgs from '../../../nls/platform.properties'
import { RESOURCE_TYPES } from '../../../lib/shared/constants'
import { fetchTopology,  updateTopologyResourceType } from '../../actions/topology'
import { Portals } from './constants.js'
import {
  ResourceTopologyTabs,
  getDiagramName,
  getDiagramOptions,
  getDiagramStyles,
  hasServerFilters,
  getServerFilters,
  requiresServerReload,
  getActiveTopologyFilters } from './helpers/topology-helper'
import { getTopologyElements } from './helpers/fetch-helper'
import { Topology } from '../Topology'
import { TopologyCards } from '../TopologyCards'
import TopologyToolbar from './TopologyToolbar'
import config from '../../../lib/shared/config'
import _ from 'lodash'

class TopologyView extends React.Component {
  static propTypes = {
    activeChannel: PropTypes.string,
    activeFilters: PropTypes.object,
    canSaveTopologyState: PropTypes.bool,
    channels: PropTypes.array,
    detailsStatus: PropTypes.string,
    fetchTopology: PropTypes.func,
    links: PropTypes.array,
    nodes: PropTypes.array,
    onTopologyResourceTypeChange: PropTypes.func,
    params: PropTypes.object,
    restoreSavedTopologyFilters: PropTypes.func,
    role: PropTypes.string,
    serverFilterSelected: PropTypes.bool,
    showLogs: PropTypes.func,
    status: PropTypes.string,
    topologyResourceType: PropTypes.string,
    updateSecondaryHeader: PropTypes.func,
  }

  constructor (props) {
    super(props)
    this.state = {
      loaded:false,
      firstLoad:false,
      isAccountAdministrator: props.role==='AccountAdministrator',
      changingChannel: false,
      otherTypeFilters:[],
    }
    this.refetch = this.refetch.bind(this)
    this.onTopologyResourceTypeChange = this.onTopologyResourceTypeChange.bind(this)
  }

  UNSAFE_componentWillMount() {
    const { updateSecondaryHeader:updateSecondaryHeaderTemp } = this.props
    updateSecondaryHeaderTemp(msgs.get('routes.topology', this.context.locale))
    // changing active filters will then load the toplogy diagram
    this.props.restoreSavedTopologyFilters()
  }

  refetch() {
    const { changingChannel } = this.state
    if (!changingChannel) {
      const { activeFilters, topologyResourceType, activeChannel } = this.props
      this.props.fetchTopology(activeFilters, topologyResourceType, activeChannel, true)
    }
  }

  UNSAFE_componentWillReceiveProps(nextProps) {
    // load topology
    this.setState(prevState => {
      let { loaded, firstLoad, reloading, changingChannel } = prevState
      const { nodes, activeFilters, activeChannel, topologyResourceType,
        canSaveTopologyState, status, detailsStatus, serverFilterSelected} = nextProps

      // need to reload if user asks for different clusters
      // else active filter just affects what shapes whe show
      const needReload = requiresServerReload(topologyResourceType, activeFilters, this.props.activeFilters)
      if (changingChannel) {
        changingChannel = _.get(activeFilters, 'application[0].channel') !== activeChannel
      }
      if (topologyResourceType !== this.props.topologyResourceType ||
          canSaveTopologyState !== this.props.canSaveTopologyState || needReload) {
        if (serverFilterSelected) {
          this.props.fetchTopology(activeFilters, topologyResourceType, activeChannel)
          loaded = false
        } else {
          loaded = true
        }
      } else {
        // prevent loading... message when just doing a live update
        loaded = (loaded
              || status === Actions.REQUEST_STATUS.DONE
              || status === Actions.REQUEST_STATUS.CACHE
              || status === Actions.REQUEST_STATUS.ERROR)
            && !needReload
      }
      firstLoad = firstLoad || loaded

      // reloading??
      if (!_.isEqual(nodes, this.props.nodes) || !_.isEqual(activeFilters, this.props.activeFilters) ||
          topologyResourceType!==this.props.topologyResourceType
          || status!==this.props.status || detailsStatus!==this.props.detailsStatus) {
        reloading = status===Actions.REQUEST_STATUS.CACHE ||
                 status===Actions.REQUEST_STATUS.IN_PROGRESS ||
                    detailsStatus===Actions.REQUEST_STATUS.IN_PROGRESS
      }
      //reloading = reloading || willLoadDetails || !detailsLoaded ||detailsReloading
      return {
        loaded, firstLoad, reloading, changingChannel
      }
    })
  }

  shouldComponentUpdate(nextProps, nextState){
    return !_.isEqual(this.props.activeFilters, nextProps.activeFilters) ||
    this.state.changingChannel!==nextState.changingChannel ||
    nextProps.activeChannel !==  this.props.activeChannel ||
    nextProps.topologyResourceType !== this.props.topologyResourceType ||
    nextProps.status !== this.props.status ||
    nextState.hasActiveRequest !== this.state.hasActiveRequest ||
    nextState.reloading !== this.state.reloading ||
    nextState.loaded !== this.state.loaded
  }

  render() {
    const { locale } = this.context
    const { loaded, firstLoad, reloading, changingChannel,
      otherTypeFilters, isAccountAdministrator } = this.state
    const { activeFilters, links, nodes, channels, activeChannel, params,
      topologyResourceType, serverFilterSelected, status, showLogs} = this.props
    // local storage hasn't determined what type of topology we're showing yet
    if (!topologyResourceType) {
      return null
    }

    const fetchControl = {
      isLoaded: loaded,
      isFailed: status === Actions.REQUEST_STATUS.ERROR,
      isReloading: reloading,
      refetch: this.refetch,
    }
    const channelControl = {
      allChannels: channels,
      activeChannel: activeChannel,
      isChangingChannel: changingChannel,
      changeTheChannel: this.changeTheChannel.bind(this)
    }

    const selectedTabIndex = ResourceTopologyTabs.findIndex(tab=>{
      return topologyResourceType===tab.name
    })

    const classnames = classNames('topologyTab', {'first-load diagram-shown': firstLoad})
    const renderTab = (topologyResourceTab, namePath) => {
      if (topologyResourceType===topologyResourceTab) {
        const {name} = getDiagramName(topologyResourceType, activeFilters, locale)
        const options = getDiagramOptions(topologyResourceType)
        const styles = getDiagramStyles(topologyResourceType)
        return (
          <div className={classnames} >
            <TopologyToolbar params={params||{}}
              otherTypeFilters={otherTypeFilters}
              topologyResourceType={topologyResourceType}
            />
            {!serverFilterSelected ? this.renderNothingSelected(namePath) : ''}
            {serverFilterSelected && (selectedTabIndex === 0 || selectedTabIndex === 2)?
              <Topology
                title={name}
                links={links}
                nodes={nodes}
                options={options}
                styles={styles}
                showLogs={showLogs}
                portals={Portals}
                fetchControl={fetchControl}
                channelControl = {channelControl}
                searchUrl = {config.contextPath.replace(new RegExp('/topology$'), '/search')}
                selectedTab = {selectedTabIndex}
                locale={locale}
              /> : ''}
            {serverFilterSelected && selectedTabIndex === 1 ?
              <TopologyCards
                portals={Portals}
                locale={locale}
              /> : ''
            }
          </div>
        )
      }
      return null
    }

    return (
      <Tabs className='topologyDiagramTabs'
        selected={selectedTabIndex}
        onSelectionChange={this.onTopologyResourceTypeChange} >
        <Tab label={msgs.get('tabs.clusters', locale)} >
          {renderTab(RESOURCE_TYPES.HCM_CLUSTERS.name)}
        </Tab>
        {/*   // hiding Network tab
          !isAccountAdministrator && <Tab label={msgs.get('tabs.networking', locale)} >
          {renderTab(RESOURCE_TYPES.HCM_TOPOLOGY.name, 'topology')}
        </Tab>*/}
        <Tab label={msgs.get('tabs.applications', locale)} >
          {renderTab(RESOURCE_TYPES.HCM_APPLICATIONS.name, 'application')}
        </Tab>
        {!isAccountAdministrator && <Tab label={msgs.get('tabs.policies', locale)} >
          {renderTab(RESOURCE_TYPES.HCM_POLICIES.name, 'policy')}
        </Tab>}
      </Tabs>
    )
  }

  renderNothingSelected(type) {
    const { locale } = this.context
    const title = msgs.get(`overview.no.information.title.${type}`, locale)
    const tabName = msgs.get('tabs.govern.risk', locale)
    const descriptionLine1 = msgs.get(`overview.no.information.description.${type}.line1`, locale)
    const descriptionLine2Start = msgs.get(`overview.no.information.description.${type}.line2.1`, locale)
    const descriptionLine2End = msgs.get(`overview.no.information.description.${type}.line2.2`, locale)
    return (
      <div className='no-information'>
        <img className='no-information-icon'
          src={`${config.contextPath}/graphics/no-selection.png`} alt={title} />
        <div className='no-information-title'>{title}</div>
        <div className='no-information-description'>
          {descriptionLine1}
          <br />
          {descriptionLine2Start} <span id="text-highlight">{tabName}</span> {descriptionLine2End}
        </div>
      </div>
    )
  }

  changeTheChannel(fetchChannel) {
    this.setState({ changingChannel: true})
    const { activeFilters, topologyResourceType } = this.props
    this.props.fetchTopology(activeFilters, topologyResourceType, fetchChannel)
  }

  onTopologyResourceTypeChange = (selection) => {
    this.props.onTopologyResourceTypeChange(ResourceTopologyTabs[selection].name)
  }

}

TopologyView.contextTypes = {
  locale: PropTypes.string
}

const mapStateToProps = (state) =>{
  const activeFilters = getActiveTopologyFilters(state)
  const { availableFilters, topologyResourceType, canSaveTopologyState, status } = state.topology
  const { willLoadDetails, detailsLoaded, detailsReloading, detailsStatus } = state.topology
  const serverFilterSelected = hasServerFilters(topologyResourceType, activeFilters)
  const {links=[], nodes=[], channels, activeChannel} =
    getTopologyElements(topologyResourceType, state.topology, activeFilters)
  return {
    links,
    nodes,
    channels,
    activeChannel,
    activeFilters,
    availableFilters,
    canSaveTopologyState,
    topologyResourceType,
    serverFilterSelected,
    status,
    willLoadDetails,
    detailsLoaded,
    detailsReloading,
    detailsStatus,
  }
}

const mapDispatchToProps = (dispatch) => {
  return {
    fetchTopology: (filters, topologyResourceType, fetchChannel, reloading) => {
      const fetchFilters = getServerFilters(topologyResourceType, filters, fetchChannel)
      dispatch(fetchTopology({ filter: {...fetchFilters}, topologyResourceType}, fetchFilters, reloading))
    },
    restoreSavedTopologyFilters: () => {
      dispatch({type: Actions.TOPOLOGY_RESTORE_SAVED_STATE})
    },
    onTopologyResourceTypeChange: (topologyResourceType) => {
      dispatch(updateTopologyResourceType(topologyResourceType))
      dispatch({type: Actions.TOPOLOGY_RESTORE_SAVED_STATE})
    },
    showLogs: ( data) => {
      dispatch(updateModal({ open: true, type: 'view-logs', resourceType:RESOURCE_TYPES.HCM_PODS, data }))
    },
    updateSecondaryHeader: (title, tabs, breadcrumbItems, links) =>
      dispatch(updateSecondaryHeader(title, tabs, breadcrumbItems, links)),
  }
}

export default withRouter(connect(mapStateToProps, mapDispatchToProps)(TopologyView))
