/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

import { RESOURCE_TYPES } from '../../../../lib/shared/constants'
import { getStoredKey, getStoredObject } from '../helpers/topology-helper'
import _ from 'lodash'

export const getTopologyElements = (topologyResourceType, topology, activeFilters) => {
  switch (topologyResourceType) {
  case RESOURCE_TYPES.HCM_CLUSTERS.name:
    return getTopologyClusterElements(topology)
  case RESOURCE_TYPES.HCM_APPLICATIONS.name:
    return getTopologyApplicationElements(topology, activeFilters)
  default:
  case RESOURCE_TYPES.HCM_TOPOLOGY.name:
  case RESOURCE_TYPES.HCM_POLICIES.name:
    return getTopologyRelationshipElements(topology)
  }
}

const getTopologyRelationshipElements = (topology, stripPodName) => {
  const { nodes=[] } = topology
  let { links=[] } = topology

  if (nodes.length>0) {
    // create maps
    const nodeMap = {}
    const podMap = {}
    const deploymentMap = {}
    nodes.forEach(node=> {
      const { clusterName='', name, namespace='<none>', type, uid } = node
      switch (type) {
      case 'deployment':
        node.specs = Object.assign(node.specs||{}, {
          hasPods: false,
          pods: [],
        })
        deploymentMap[namespace+'/'+name] = node
        break
      case 'pod':
        podMap[`${clusterName}/${name}`] = node
      }
      nodeMap[uid] = node
    })
    // convert "to/from" to "source/target"
    // filter out self links
    links = links.map((l)=>({
      source: l.from.uid,
      target: l.to.uid,
      label: l.type,
      type: l.type,
      specs: l.specs,
      uid: l.from.uid + l.to.uid,
    })).filter(l => {
      if (l.source === l.target) {
        if (nodeMap[l.source]) {
          nodeMap[l.source].selfLink = l
        }
        return false
      }
      return true
    })
    // add details
    addDiagramDetails(topology, nodes, podMap, stripPodName)
  }
  return {
    nodes,
    links,
  }
}

const getTopologyApplicationElements = (topology, activeFilters) => {
  const elements =  getTopologyRelationshipElements(topology, true)
  // determine active channel
  let {activeChannel='first', channels} = topology
  const currentAppId = _.get(topology, 'fetchFilters.application[0].uid')
  const requestedAppId = _.get(activeFilters, 'application[0].uid')
  if (!channels || currentAppId!==requestedAppId) {
    const application = _.get(activeFilters, 'application')
    const storedKey = getStoredKey({application})
    const storedObject = getStoredObject(storedKey)
    if (storedObject) {
      channels = storedObject.channels
      activeChannel = storedObject.activeChannel
    }
  }
  return Object.assign(elements, {channels, activeChannel})
}

const getTopologyClusterElements = (topology) => {
  const { nodes = [], links = [] } = topology
  return {
    nodes,
    links,
  }
}

function addDiagramDetails(topology, nodes, podMap, stripPodName) {
  const { detailsLoaded, detailsReloading } = topology
  // get extra details from topology or from localstore
  if (detailsLoaded || detailsReloading) {
    const {pods} = topology
    // associate pods with status
    if (pods) {
      pods.forEach(pod=>{
        let {name} = pod
        if (name) {
          // if coming from application, we only know the pod w/o the uid
          if (stripPodName) {
            name = pod.name.replace(/-[0-9a-fA-F]{8,10}-[0-9a-zA-Z]{4,5}$/, '')
            if (name===pod.name) {
              const idx = name.lastIndexOf('-')
              if (idx!==-1) {
                name = name.substr(0, idx)
              }
            }
          }
          // match up status with pod node
          const podName = `${_.get(pod, 'cluster.metadata.name')}/${name}`
          if (podMap[podName]) {
            const podModel = _.get(podMap[podName], 'specs.podModel', {})
            podModel[pod.name] = pod
            _.set(podMap[podName], 'specs.podModel', podModel)
          }
        }
      })
    }
  }
}
