/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

import * as Actions from '../../../../src-web/actions'
import { RESOURCE_TYPES, MCM_TOPOLOGY_STATE_COOKIE,
  MCM_TOPOLOGY_ELEMENTS_COOKIE  } from '../../../../lib/shared/constants'
import msgs from '../../../../nls/platform.properties'
import _ from 'lodash'

/*
* UI helpers to help with data transformations
* */

export const ResourceTopologyTabs = Object.freeze([
  RESOURCE_TYPES.HCM_CLUSTERS,
  // hiding Network tab
  //RESOURCE_TYPES.HCM_TOPOLOGY,
  RESOURCE_TYPES.HCM_APPLICATIONS,
  RESOURCE_TYPES.HCM_POLICIES,
])


export const getDiagramName = (topologyResourceType, activeFilters, locale) => {
  let name=''
  let namespace=''
  switch (topologyResourceType) {
  case RESOURCE_TYPES.HCM_CLUSTERS.name:
    name = msgs.get('topology.title.clusters', locale)
    break

  case RESOURCE_TYPES.HCM_TOPOLOGY.name:
    name = _.get(activeFilters, 'cluster[0].name','')
    namespace = _.get(activeFilters, 'cluster[0].namespace','')
    break

  case RESOURCE_TYPES.HCM_APPLICATIONS.name:
    name = _.get(activeFilters, 'application[0].name','')
    namespace = _.get(activeFilters, 'application[0].namespace','')
    break

  case RESOURCE_TYPES.HCM_POLICIES.name:
    name = _.get(activeFilters, 'policy[0].name','')
    namespace = _.get(activeFilters, 'policy[0].namespace','')
    break
  }
  return {name, namespace}
}

export const getDiagramOptions = (topologyResourceType) => {
  let options={}
  switch (topologyResourceType) {
  case RESOURCE_TYPES.HCM_CLUSTERS.name:
    options = {
      filtering: 'cluster',
      showLineLabels: true, // show labels on lines
      showGroupTitles: true // show titles over sections
    }
    break

  case RESOURCE_TYPES.HCM_TOPOLOGY.name:
    options = {
      filtering: 'weave',
      availableTypes: ['internet', 'host', 'service', 'deployment', 'daemonset', 'statefulset', 'pod', 'other'],
      initialActiveTypes: ['internet', 'host', 'deployment', 'pod'],
      showHubs: true,
      consolidateSmallGroups: true,
      embedPodsInDeployments: true,
    }
    break

  case RESOURCE_TYPES.HCM_APPLICATIONS.name:
    options = {
      filtering: 'application',
      layout: 'application',
      showLineLabels: true, // show labels on lines
      showGroupTitles: false, // show titles over sections
    }
    break

  case RESOURCE_TYPES.HCM_POLICIES.name:
    options = {
      filtering: 'policy',
      layout: 'policy'
    }
    break
  }
  return options
}

export const getDiagramStyles = (topologyResourceType) => {
  let styles={}
  switch (topologyResourceType) {
  case RESOURCE_TYPES.HCM_APPLICATIONS.name:
    styles = {
      shapes: {
        'deployment': {
          shape: 'ideployment',
          className: 'deployment'
        },
        'pod': {
          shape: 'ipod',
          className: 'pod'
        },
      }
    }
    break
  }
  return styles
}

export const hasServerFilters = (topologyResourceType=RESOURCE_TYPES.HCM_CLUSTERS.name, filters) => {
  switch (topologyResourceType) {
  default:
  case RESOURCE_TYPES.HCM_CLUSTERS.name:
    return true

  case RESOURCE_TYPES.HCM_APPLICATIONS.name:
    return true

  case RESOURCE_TYPES.HCM_TOPOLOGY.name:
    return filters.cluster && filters.cluster.length>0

  case RESOURCE_TYPES.HCM_POLICIES.name:
    return filters.policy  && filters.policy.length>0
  }
}

export const getServerFilters = (topologyResourceType=RESOURCE_TYPES.HCM_CLUSTERS.name, filters, fetchChannel) => {
  switch (topologyResourceType) {
  default:
  case RESOURCE_TYPES.HCM_CLUSTERS.name:
    return {}

  case RESOURCE_TYPES.HCM_TOPOLOGY.name:
    return {cluster: _.cloneDeep(filters.cluster)}

  case RESOURCE_TYPES.HCM_APPLICATIONS.name: {
    if (filters.application) {
      const application = _.cloneDeep(filters.application)
      application[0].channel = fetchChannel
      return {application}
    }
    return {}
  }

  case RESOURCE_TYPES.HCM_POLICIES.name:
    return {policy: _.cloneDeep(filters.policy)}
  }
}

export const requiresServerReload = (topologyResourceType=RESOURCE_TYPES.HCM_CLUSTERS.name,
  requestedFilters, currentFilters) => {
  switch (topologyResourceType) {
  default:
  case RESOURCE_TYPES.HCM_CLUSTERS.name:
    return Object.keys(requestedFilters).length>0

  case RESOURCE_TYPES.HCM_TOPOLOGY.name: {
    const requested =_.get(requestedFilters, 'cluster')
    const current = _.get(currentFilters, 'cluster')
    return !_.isEqual(requested, current)
  }

  case RESOURCE_TYPES.HCM_APPLICATIONS.name: {
    let application = _.get(requestedFilters, 'application[0]', {})
    const requested = `${application.uid}//${application.channel || '__ALL__/__ALL__//__ALL__/__ALL__'}`
    application = _.get(currentFilters, 'application[0]', {})
    const current = `${application.uid}//${application.channel || '__ALL__/__ALL__//__ALL__/__ALL__'}`
    return requested!==current
  }

  case RESOURCE_TYPES.HCM_POLICIES.name:
    return !_.isEqual(requestedFilters.policy, currentFilters.policy)
  }
}

export const getActiveTopologyFilters = (state) => {
  const { topologyResourceType=RESOURCE_TYPES.HCM_CLUSTERS.name } = state.topology
  // if application, also include current channel
  let activeFilters = state.topology.activeFilters
  if (topologyResourceType === RESOURCE_TYPES.HCM_APPLICATIONS.name) {
    const application = _.get(activeFilters, 'application[0]')
    if (application) {
      activeFilters = Object.assign({}, activeFilters)
      application.channel =  _.get(state.topology, 'fetchFilters.application[0].channel')
    }
  }
  return Object.assign({}, activeFilters, state.resourceToolbar.activeFilters)
}

export const saveTopologyState = (action, state) => {
  let ret
  let cookieKey
  switch (action.type) {
  case Actions.TOPOLOGY_RESOURCE_TYPE_UPDATE: {
    cookieKey = `${MCM_TOPOLOGY_STATE_COOKIE}--TOPOLOGY-RESOURCE-TYPE`
    ret = { topologyResourceType: action.topologyResourceType }
    break
  }
  case Actions.TOPOLOGY_FILTERS_UPDATE: {
    const topologyResourceType = state.topologyResourceType || RESOURCE_TYPES.HCM_CLUSTERS.name
    cookieKey = `${MCM_TOPOLOGY_STATE_COOKIE}--${topologyResourceType}`
    const activeFilters = Object.assign({}, {...state.activeFilters} || {})
    activeFilters[action.filterType] = action.filters
    ret = { activeFilters }
    break
  }
  }
  // to avoid conflict with getSavedTopologyState, don't save to local storage until getSavedTopologyState is called
  if (state.canSaveTopologyState) {
    saveStoredObject(cookieKey, JSON.stringify(ret))
  }
  return ret
}

export const getSavedTopologyState = () => {
  const ret = {activeFilters: {}, topologyResourceType: RESOURCE_TYPES.HCM_CLUSTERS.name}
  // get saved group by
  let cookieKey = `${MCM_TOPOLOGY_STATE_COOKIE}--TOPOLOGY-RESOURCE-TYPE`
  let savedState = getStoredObject(cookieKey)
  if (savedState) {
    try {
      savedState = JSON.parse(savedState)
      if (savedState.topologyResourceType) {
        ret.topologyResourceType = savedState.topologyResourceType
      }
    } catch (e) {
      //
    }
  }
  // get saved filters for this group by
  cookieKey = `${MCM_TOPOLOGY_STATE_COOKIE}--${ret.topologyResourceType}`
  savedState = getStoredObject(cookieKey)
  if (savedState) {
    try {
      savedState = JSON.parse(savedState)
      if (savedState.activeFilters) {
        ret.activeFilters = savedState.activeFilters
        if (ret.topologyResourceType === RESOURCE_TYPES.HCM_APPLICATIONS.name) {
          const application = _.get(ret.activeFilters, 'application[0]')
          if (application) {
            delete application.channel
          }
        }
      }
    } catch (e) {
      //
    }
  }
  return ret
}

export const getStoredKey = (fetchFilters) => {
  const keys = Object.keys(fetchFilters)
  if (Array.isArray(keys) && keys.length>0) {
    const key = keys[0]
    const uid = _.get(fetchFilters, `${key}[0].uid`)
    if (uid) {
      return `${MCM_TOPOLOGY_ELEMENTS_COOKIE}--${key}--${uid}`
    }
  }
  return undefined
}

export const getStoredObject = (storageKey) => {
  try {
    storageKey = `${storageKey}--${window.location.href}`
    const sessionObject = JSON.parse(sessionStorage.getItem(storageKey))
    if (sessionObject && sessionObject.expiresAt && sessionObject.expiresAt > Date.now()) {
      return sessionObject.sessionData
    } else {
      sessionStorage.removeItem(storageKey)
    }
  } catch (error) {
    // no privileges
  }
  return null
}

export const saveStoredObject = (storageKey, object, expiring=60) => {
  try {
    storageKey = `${storageKey}--${window.location.href}`
    const sessionObject = {
      expiresAt: Date.now() + expiring*60*1000, // expire in 60 minutes
      sessionData: object
    }
    sessionStorage.setItem(storageKey, JSON.stringify(sessionObject))
  } catch (error) {
    // no privileges
  }
}

