/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2018, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

import React from 'react'
import PropTypes from 'prop-types'
import AceEditor from 'react-ace'
import { Modal, DropdownV2, Loading } from 'carbon-components-react'
import { withRouter } from 'react-router-dom'
import { connect } from 'react-redux'
import { fetchLogs, resetLogs } from '../../actions/logs'
import { RESOURCE_TYPES } from '../../../lib/shared/constants'
import { REQUEST_STATUS } from '../../actions/index'
import resources from '../../../lib/shared/resources'
import { updateModal } from '../../actions/common'
import msgs from '../../../nls/platform.properties'
import config from '../../../lib/shared/config'

import 'brace/mode/asciidoc'
import 'brace/mode/powershell'
import 'brace/theme/sqlserver'

resources(() => {
  require('../../../scss/logs.scss')
})

class LogsModal extends React.Component {
  constructor(props) {
    super(props)
    this.state = {
      selectedContainer: props.containerName,
      xhrPoll: false
    }
  }

  UNSAFE_componentWillMount() {
    if (parseInt(config['featureFlags:liveUpdates'], 10) === 2) {
      const intervalId = setInterval(this.reload.bind(this), config['featureFlags:liveUpdatesPollInterval'])
      this.setState({ intervalId: intervalId })
    }
  }

  componentDidMount() {
    const { containerName, podName, podNamespace, clusterName } = this.props
    this.props.fetchLogs(containerName, podName, podNamespace, clusterName)
  }

  componentWillUnmount() {
    clearInterval(this.state.intervalId)
    this.props.resetLogs()
  }

  render() {
    const { logs, status, containers, containerName, podName, open } = this.props,
          { xhrPoll } = this.state,
          { locale } = this.context
    const containerChoices = containers.map(
      (container, index) =>
        ({
          id: `${container.name}-${index}`,
          label: container.name, value: container.name
        })
    )
    const idx = Math.max(0, containerChoices.findIndex(({value})=>{
      return value===containerName
    }))
    return (
      <Modal
        id='view-logs-modal'
        className='logs'
        open={open}
        passiveModal
        modalLabel='Pod'
        modalHeading={podName}
        primaryButtonDisabled
        ref={modal=>{
          if (modal && modal.outerModal) {
            modal.outerModal.current={focus: ()=>{/*This is intentional*/}}
          }
        }}
        onRequestClose={this.props.handleClose}
        role='region'
        aria-label='logs'>
        <div className='logs-container'>
          <div className='logs-container__actions'>
            <div className='dropdown-container'>
              <DropdownV2
                ariaLabel={msgs.get('dropdown.pod.label', locale)}
                label={containerName}
                inline={true}
                onChange={this.handleContainerChange.bind(this)}
                initialSelectedItem={containerChoices[idx].label}
                items={containerChoices} />
            </div>
          </div>

          {(() => {
            if (!xhrPoll && status !== REQUEST_STATUS.DONE) {
              return <Loading withOverlay={false} className='content-spinner' />
            }
            const setEditorRef = elem => {
              if (elem) {
                elem.editor.scrollToLine(1000000, true, true, () => {/*This is intentional*/})
                elem.editor.gotoLine(10000000, 0, true)
              }
            }
            return <div className='logs-container__content'>
              <AceEditor
                theme='sqlserver'
                mode={'powershell'}
                width={'100%'}
                height={'100%'}
                wrapEnabled={true}
                fontSize={12}
                showPrintMargin={false}
                showGutter={true}
                highlightActiveLine={true}
                value={logs}
                setOptions={{
                  readOnly: true,
                  showLineNumbers: true,
                }}
                ref={setEditorRef}
              />
            </div>
          })()}
        </div>
      </Modal>
    )
  }

  handleContainerChange(data) {
    const containerName = data.selectedItem.label,
          { fetchLogs:fetchLogsTemp, status } = this.props
    this.setState({
      xhrPoll: false,
      selectedContainer: containerName,
    })
    if (status === REQUEST_STATUS.DONE) {
      const { podName, podNamespace, clusterName } = this.props
      fetchLogsTemp(containerName, podName, podNamespace, clusterName)
    }
  }

  reload() {
    const { selectedContainer } = this.state
    const { podName, podNamespace, clusterName } = this.props
    if (this.props.status === REQUEST_STATUS.DONE) {
      this.setState({ xhrPoll: true })
      this.props.fetchLogs(selectedContainer, podName, podNamespace, clusterName)
    }
  }
}

LogsModal.contextTypes = {
  locale: PropTypes.string
}

LogsModal.propTypes = {
  clusterName: PropTypes.string,
  containerName: PropTypes.string,
  containers: PropTypes.array,
  fetchLogs: PropTypes.func,
  handleClose: PropTypes.func,
  logs: PropTypes.string,
  open: PropTypes.bool,
  podName: PropTypes.string,
  podNamespace: PropTypes.string,
  resetLogs: PropTypes.func,
  status: PropTypes.string,
}

const mapStateToProps = (state, ownProps) => {
  return {
    logs: state.logs && state.logs.data,
    status: state.logs && state.logs.status,
    containers: ownProps.data.containers,
    containerName: ownProps.data.containerName,
    podName: ownProps.data.name,
    podNamespace: ownProps.data.namespace,
    clusterName: ownProps.data.clusterName
  }
}

const mapDispatchToProps = (dispatch) => {
  return {
    handleClose: () => dispatch(updateModal({open: false, type: 'view-logs'})),
    fetchLogs: (containerName, podName, podNamespace, clusterName) =>
      dispatch(fetchLogs(containerName, podName, podNamespace, clusterName)),
    resetLogs: () => dispatch(resetLogs(RESOURCE_TYPES.LOGS)),
  }
}

export default withRouter(connect(mapStateToProps, mapDispatchToProps)(LogsModal))
