/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2018, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
import { getInitialBuffer, getBufferedResponse } from '../../lib/client/weave-helper'
import { getStoredKey, getStoredObject, saveStoredObject, requiresServerReload,
  getSavedTopologyState, saveTopologyState }
  from '../components/TopologyView/helpers/topology-helper'
import { RESOURCE_TYPES } from '../../lib/shared/constants'
import _ from 'lodash'
import {
  REQUEST_STATUS, RESOURCE_REQUEST, RESOURCE_RECEIVE_SUCCESS, RESOURCE_DETAILS_REQUEST,
  RESOURCE_DETAILS_RECEIVE_SUCCESS, RESOURCE_RECEIVE_FAILURE, TOPOLOGY_FILTERS_REQUEST,
  TOPOLOGY_FILTERS_RECEIVE_ERROR, TOPOLOGY_RESTORE_SAVED_STATE, TOPOLOGY_RESOURCE_TYPE_UPDATE,
  TOPOLOGY_FILTERS_UPDATE, TOPOLOGY_FILTERS_RECEIVE_SUCCESS
} from '../actions'

const initialState = {
  availableFilters: {},
  diagramFilters: [],
  links: [],
  nodes: [],
  status: REQUEST_STATUS.INCEPTION,
}

export const topology = (state = initialState, action) => {
  if (action.resourceType && action.resourceType.name === RESOURCE_TYPES.HCM_TOPOLOGY.name){
    switch (action.type) {
    case RESOURCE_REQUEST: {
      let channels=[]
      let activeChannel
      const { fetchFilters } = action
      const { buffer={} } = state
      if (!_.isEqual(buffer.fetchFilters, fetchFilters)) {

        // for applications we get the active channel first
        // then we use that as a key to the stored nodes/links
        let storedKey = getStoredKey(fetchFilters)
        if (state.topologyResourceType===RESOURCE_TYPES.HCM_APPLICATIONS.name) {
          const storedActiveChannel = getStoredObject(storedKey)
          if (storedActiveChannel) {
            activeChannel = _.get(storedActiveChannel, 'activeChannel', 'first')
            channels = _.get(storedActiveChannel, 'channels')
          }
          activeChannel = _.get(fetchFilters, 'application[0].channel', activeChannel)
          storedKey = `${storedKey}--${activeChannel}`
        }

        const savedElements = getStoredObject(storedKey)
        if (savedElements) {
          const {nodes, links} = savedElements
          const bufferTemp = (state.topologyResourceType===RESOURCE_TYPES.HCM_TOPOLOGY.name)
            ? getInitialBuffer(fetchFilters, nodes, links)
            : {}
          return {...state,
            nodes, links, bufferTemp, activeChannel, channels,
            fetchFilters: action.fetchFilters,
            status: REQUEST_STATUS.CACHE
          }
        }
      }
      return {...state,
        fetchFilters,
        status: REQUEST_STATUS.IN_PROGRESS}
    }
    case RESOURCE_RECEIVE_SUCCESS: {

      // ignore topologies that were fetched with a different set of active filters
      if (!requiresServerReload(state.topologyResourceType, action.fetchFilters, state.activeFilters)) {
        const {nodes, links, buffer} = getBufferedResponse(state, action)
        const {willLoadDetails} = action

        // for applications save active channel first
        let channels = []
        let activeChannel = 'first'
        let storedKey = getStoredKey(action.fetchFilters)
        if (state.topologyResourceType===RESOURCE_TYPES.HCM_APPLICATIONS.name) {
          nodes.some(({type, specs})=>{
            if (type==='application') {
              ({channels, activeChannel} = specs)
              return true
            }
            return false
          })
          saveStoredObject(storedKey, {activeChannel, channels})
          storedKey = `${storedKey}--${activeChannel||'first'}`
        }
        saveStoredObject(storedKey, {nodes, links})
        return { ...state,
          status: REQUEST_STATUS.DONE,
          nodes, links, buffer, channels, activeChannel,
          fetchFilters: action.fetchFilters,
          willLoadDetails,
          detailsLoaded: !willLoadDetails,
          detailsStatus: willLoadDetails? REQUEST_STATUS.IN_PROGRESS : REQUEST_STATUS.DONE,
          detailsReloading: willLoadDetails,
          reloading: false
        }
      } else {
        return { ...state }
      }
    }
    case RESOURCE_DETAILS_REQUEST: {
      return {
        ...state,
        detailsStatus: REQUEST_STATUS.IN_PROGRESS,
        detailsFetchFilters: action.detailsFetchFilters,
        detailsActiveFilters: {},
        detailsReloading: action.reloading,
        detailsLoaded: false,
        willLoadDetails: false,
      }
    }
    case RESOURCE_DETAILS_RECEIVE_SUCCESS: {

      // ignore topologies details that were fetched with a different set of active filters
      if (requiresServerReload(state.topologyResourceType, action.detailsFetchFilters, state.detailsActiveFilters)) {
        const { pods } = action
        return {
          ...state,
          detailsStatus: REQUEST_STATUS.DONE,
          pods,
          detailsActiveFilters: action.detailsFetchFilters,
          detailsLoaded: true,
          detailsReloading: false
        }
      } else {
        return { ...state }
      }
    }
    case RESOURCE_RECEIVE_FAILURE: {
      return { ...state, status: REQUEST_STATUS.ERROR, nodes: action.nodes, links: action.links }
    }
    }
  }

  switch (action.type){
  case '@@INIT':{
    return initialState
  }
  case TOPOLOGY_FILTERS_REQUEST: {
    return {...state,
      filtersStatus: REQUEST_STATUS.IN_PROGRESS,
    }
  }
  case TOPOLOGY_FILTERS_RECEIVE_ERROR: {
    return {...state,
      filtersStatus: REQUEST_STATUS.ERROR,
      err: action.err,
    }
  }
  case TOPOLOGY_RESTORE_SAVED_STATE: {
    const {activeFilters, topologyResourceType} = getSavedTopologyState()
    // once state has been initialized from local store,
    // we can start saving changes back to local store
    return {...state, activeFilters, topologyResourceType, canSaveTopologyState: true}
  }
  case TOPOLOGY_RESOURCE_TYPE_UPDATE: {
    return {...state, ...saveTopologyState(action, state), nodes:[], links:[], pods:[], channels:[]}
  }
  case TOPOLOGY_FILTERS_UPDATE: {
    return {...state, ...saveTopologyState(action, state), nodes:[], links:[], pods:[], channels:[] }
  }
  case TOPOLOGY_FILTERS_RECEIVE_SUCCESS: {
    let key, akey
    const availableFilters = {}
    const { topologyResourceType, filters: {filters}} = action
    const { activeFilters } = state
    switch (topologyResourceType) {
    case RESOURCE_TYPES.HCM_TOPOLOGY.name:
      key = 'clusters'
      akey = 'cluster'
      break

    case RESOURCE_TYPES.HCM_APPLICATIONS.name:
      key = 'applications'
      akey = 'application'
      break

    case RESOURCE_TYPES.HCM_POLICIES.name:
      key = 'policies'
      akey = 'policy'
      break
    }
    const availableSet = new Set()
    availableFilters[key] = filters.map(filter=>{
      availableSet.add(filter.uid)
      return {label: filter.name, value: filter}
    })
      .sort(({label:al}, {label:bl})=>{
        return al.localeCompare(bl)
      })
    // filter active if no longer available
    if (activeFilters && activeFilters[akey]) {
      activeFilters[akey] = activeFilters[akey].filter(filter=>{
        return availableSet.has(filter.uid)
      })
    }
    return {...state,
      activeFilters,
      availableFilters,
      filtersStatus: REQUEST_STATUS.DONE,
    }
  }
  default:
    return { ...state }
  }
}
