package tests

import (
	"fmt"
	"net/http"
	"testing"

	"github.com/hashicorp/terraform-plugin-sdk/helper/resource"
	"github.com/hashicorp/terraform-plugin-sdk/terraform"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/helpers/azure"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/acceptance"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/clients"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/utils"
)

func TestAccAzureRMMssqlServerVulnerabilityAssessment_basic(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_mssql_server_vulnerability_assessment", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMMssqlServerVulnerabilityAssessmentDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMMssqlServerVulnerabilityAssessment_basic(data),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMMssqlServerVulnerabilityAssessmentExists(data.ResourceName),
				),
			},
			data.ImportStep("storage_account_access_key"),
		},
	})
}

func TestAccAzureRMMssqlServerVulnerabilityAssessment_update(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_mssql_server_vulnerability_assessment", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMMssqlServerVulnerabilityAssessmentDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMMssqlServerVulnerabilityAssessment_basic(data),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMMssqlServerVulnerabilityAssessmentExists(data.ResourceName),
				),
			},
			data.ImportStep("storage_account_access_key"),
			{
				Config: testAccAzureRMMssqlServerVulnerabilityAssessment_update(data),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMMssqlServerVulnerabilityAssessmentExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "recurring_scans.0.enabled", "true"),
					resource.TestCheckResourceAttr(data.ResourceName, "recurring_scans.0.emails.#", "2"),
					resource.TestCheckResourceAttr(data.ResourceName, "recurring_scans.0.email_subscription_admins", "true"),
				),
			},
			data.ImportStep("storage_account_access_key"),
		},
	})
}

func testCheckAzureRMMssqlServerVulnerabilityAssessmentExists(resourceName string) resource.TestCheckFunc {
	return func(s *terraform.State) error {
		client := acceptance.AzureProvider.Meta().(*clients.Client).MSSQL.ServerVulnerabilityAssessmentsClient
		ctx := acceptance.AzureProvider.Meta().(*clients.Client).StopContext

		rs, ok := s.RootModule().Resources[resourceName]
		if !ok {
			return fmt.Errorf("resource not found: %s", resourceName)
		}

		id, err := azure.ParseAzureResourceID(rs.Primary.ID)
		if err != nil {
			return err
		}

		resourceGroupName := id.ResourceGroup
		serverName := id.Path["servers"]

		result, err := client.Get(ctx, resourceGroupName, serverName)
		if err != nil {
			if utils.ResponseWasNotFound(result.Response) {
				return fmt.Errorf("vulnerability assessment was not found for resource group %q, sql server %q",
					resourceGroupName, serverName)
			}

			return err
		}

		return nil
	}
}

func testCheckAzureRMMssqlServerVulnerabilityAssessmentDestroy(s *terraform.State) error {
	client := acceptance.AzureProvider.Meta().(*clients.Client).MSSQL.ServerVulnerabilityAssessmentsClient
	ctx := acceptance.AzureProvider.Meta().(*clients.Client).StopContext

	for _, rs := range s.RootModule().Resources {
		if rs.Type != "azurerm_mssql_server_vulnerability_assessment" {
			continue
		}

		id, err := azure.ParseAzureResourceID(rs.Primary.ID)
		if err != nil {
			return err
		}

		resourceGroupName := id.ResourceGroup
		serverName := id.Path["servers"]

		result, err := client.Get(ctx, resourceGroupName, serverName)
		if err != nil {
			return nil
		}

		if result.StatusCode != http.StatusNotFound {
			return fmt.Errorf("Server Vulnerability Assessment still exists:\n%#v", result.ServerVulnerabilityAssessmentProperties)
		}
	}

	return nil
}

func testAccAzureRMMssqlServerVulnerabilityAssessment_basic(data acceptance.TestData) string {
	server := testAccAzureRMMssqlServerVulnerabilityAssessment_server(data)

	return fmt.Sprintf(`
%s

resource "azurerm_mssql_server_vulnerability_assessment" "test" {
  server_security_alert_policy_id = azurerm_mssql_server_security_alert_policy.test.id
  storage_container_path          = "${azurerm_storage_account.test.primary_blob_endpoint}${azurerm_storage_container.test.name}/"
  storage_account_access_key      = azurerm_storage_account.test.primary_access_key
}
`, server)
}

func testAccAzureRMMssqlServerVulnerabilityAssessment_update(data acceptance.TestData) string {
	server := testAccAzureRMMssqlServerVulnerabilityAssessment_server(data)

	return fmt.Sprintf(`
%s

resource "azurerm_mssql_server_vulnerability_assessment" "test" {
  server_security_alert_policy_id = azurerm_mssql_server_security_alert_policy.test.id
  storage_container_path          = "${azurerm_storage_account.test.primary_blob_endpoint}${azurerm_storage_container.test.name}/"
  storage_account_access_key      = azurerm_storage_account.test.primary_access_key

  recurring_scans {
    enabled                   = true
    email_subscription_admins = true
    emails = [
      "email@example1.com",
      "email@example2.com"
    ]
  }
}
`, server)
}

func testAccAzureRMMssqlServerVulnerabilityAssessment_server(data acceptance.TestData) string {
	return fmt.Sprintf(`
provider "azurerm" {
  features {}
}

resource "azurerm_resource_group" "test" {
  name     = "acctestRG-mssql-%d"
  location = "%s"
}

resource "azurerm_sql_server" "test" {
  name                         = "acctestsqlserver%d"
  resource_group_name          = azurerm_resource_group.test.name
  location                     = azurerm_resource_group.test.location
  version                      = "12.0"
  administrator_login          = "mradministrator"
  administrator_login_password = "thisIsDog11"
}

resource "azurerm_storage_account" "test" {
  name                     = "accsa%d"
  resource_group_name      = azurerm_resource_group.test.name
  location                 = "%s"
  account_tier             = "Standard"
  account_replication_type = "GRS"
}

resource "azurerm_storage_container" "test" {
  name                  = "acctestsc%d"
  storage_account_name  = azurerm_storage_account.test.name
  container_access_type = "private"
}

resource "azurerm_mssql_server_security_alert_policy" "test" {
  resource_group_name = azurerm_resource_group.test.name
  server_name         = azurerm_sql_server.test.name
  state               = "Enabled"
}
`, data.RandomInteger, data.Locations.Primary, data.RandomInteger, data.RandomInteger, data.Locations.Primary, data.RandomInteger)
}
