package pkg

import (
	"context"
	"fmt"
	"strconv"

	"github.com/go-logr/logr"
	gerr "github.com/pkg/errors"
	"sigs.k8s.io/controller-runtime/pkg/client"
)

type Arg string

//type Match func(client.Client, Expect, Arg) bool

type Matcher interface {
	Match(client.Client, Expectation, logr.Logger) error
}

func MatcherRouter(name string) Matcher {
	switch name {
	case "byname":
		return ByName{}
	case "byannotation":
		return ByAnnotation{}
	case "byannotationcount":
		return ByAnnotationCount{}
	default:
		return nil
	}

	return nil
}

type ByName struct{}

func (b ByName) Match(clt client.Client, ep Expectation, logger logr.Logger) error {
	ins := ep.GetInstance()
	key := ep.GetKey()

	if err := clt.Get(context.TODO(), key, ins); err != nil {
		return gerr.Wrapf(err, "failed to get instance %s/%s of kind %s", ep.Namepsace, ep.Name, ep.Kind)
	}

	logger.Info(fmt.Sprintf("found %s of kind %s", key.String(), ep.Kind))
	return nil
}

type ByAnnotation struct{}

// Basiclly, list all the instance and check if there's instance generated by
// subscription
func (b ByAnnotation) Match(clt client.Client, ep Expectation, logger logr.Logger) error {
	if len(ep.Args) == 0 {
		return gerr.New("using the byannotation matcher, but NO annotation key:val is provided in the Args field")
	}

	ins := ep.GetInstance()
	key := ep.GetKey()

	if err := clt.Get(context.TODO(), key, ins); err != nil {
		return gerr.Wrapf(err, "failed to get instance %s of kind %s", key.String(), ep.Kind)
	}

	an := ins.GetAnnotations()

	for k, v := range ep.Args {
		if cv, ok := an[k]; !ok || v != cv {
			return fmt.Errorf("kind %s of %s has cluster annotation %s,  is not matching expectation %s", ep.Kind, key.String(), cv, v)
		}
	}

	logger.Info(fmt.Sprintf("found kind %s of %s which matched annotation %+v", ep.Kind, key.String(), ep.Args))
	return nil
}

type ByAnnotationCount struct{}

func (b ByAnnotationCount) Match(clt client.Client, ep Expectation, logger logr.Logger) error {
	if len(ep.Args) == 0 {
		return gerr.New("using the byannotationcount matcher, but NO annotation key:val is provided in the Args field")
	}

	cnt, ok := ep.Args["count"]
	if !ok {
		return gerr.New(fmt.Sprintf("%s, using the byannotationcount matcher, but NO count:val is provided in the Args field", ep.Desc))
	}

	icnt, err := strconv.Atoi(cnt)
	if err != nil {
		return gerr.New("can't parse the count filed to int")
	}

	insList := ep.GetInstanceList()
	key := ep.GetKey()

	if err := clt.List(context.TODO(), insList, &client.ListOptions{Namespace: ep.Namepsace}); err != nil {
		return gerr.Wrapf(err, "failed to get instanceList %s of kind %s", key.String(), ep.Kind)
	}

	c := 0
	for _, item := range insList.Items {
		if contains(ep.Args, item.GetAnnotations(), "count") {
			c += 1
		}
	}

	if icnt != c {
		return gerr.New(fmt.Sprintf("expecting kind %s has %d item(s), cluster has %d item(s)", ep.Kind, icnt, c))
	}

	logger.Info(fmt.Sprintf("found kind %s of %s which matched annotation %+v", ep.Kind, key.String(), ep.Args))
	return nil
}

func contains(small, big map[string]string, skip string) bool {
	for k, v := range small {
		if k == skip {
			continue
		}

		if bv, ok := big[k]; !ok || v != bv {
			return false
		}
	}

	return true
}
