// +build ignore

/* Copyright 2018 The Bazel Authors. All rights reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

// update_proto_csv reads the contents of the @go_googleapis repository
// and updates the proto.csv file. This must be done manually.

package main

import (
	"bytes"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"path"
	"path/filepath"
	"regexp"
	"strings"
)

var (
	progName           = filepath.Base(os.Args[0])
	protoCsvPath       = flag.String("proto_csv", "", "Path to proto.csv to update")
	googleapisRootPath = flag.String("go_googleapis", "", "Path to @go_googleapis repository root directory")
)

var prefix = `# This file lists special protos that Gazelle knows how to import. This is used to generate
# code for proto and Go resolvers.
#
# Generated by internal/language/proto/gen/update_proto_csv.go
# Do not edit directly.
#
# proto name,proto label,go import path,go proto label
google/protobuf/any.proto,@com_google_protobuf//:any_proto,github.com/golang/protobuf/ptypes/any,@io_bazel_rules_go//proto/wkt:any_go_proto
google/protobuf/api.proto,@com_google_protobuf//:api_proto,google.golang.org/genproto/protobuf/api,@io_bazel_rules_go//proto/wkt:api_go_proto
google/protobuf/compiler/plugin.proto,@com_google_protobuf//:compiler_plugin_proto,github.com/golang/protobuf/protoc-gen-go/plugin,@io_bazel_rules_go//proto/wkt:compiler_plugin_go_proto
google/protobuf/descriptor.proto,@com_google_protobuf//:descriptor_proto,github.com/golang/protobuf/protoc-gen-go/descriptor,@io_bazel_rules_go//proto/wkt:descriptor_go_proto
google/protobuf/duration.proto,@com_google_protobuf//:duration_proto,github.com/golang/protobuf/ptypes/duration,@io_bazel_rules_go//proto/wkt:duration_go_proto
google/protobuf/empty.proto,@com_google_protobuf//:empty_proto,github.com/golang/protobuf/ptypes/empty,@io_bazel_rules_go//proto/wkt:empty_go_proto
google/protobuf/field_mask.proto,@com_google_protobuf//:field_mask_proto,google.golang.org/genproto/protobuf/field_mask,@io_bazel_rules_go//proto/wkt:field_mask_go_proto
google/protobuf/source_context.proto,@com_google_protobuf//:source_context_proto,google.golang.org/genproto/protobuf/source_context,@io_bazel_rules_go//proto/wkt:source_context_go_proto
google/protobuf/struct.proto,@com_google_protobuf//:struct_proto,github.com/golang/protobuf/ptypes/struct,@io_bazel_rules_go//proto/wkt:struct_go_proto
google/protobuf/timestamp.proto,@com_google_protobuf//:timestamp_proto,github.com/golang/protobuf/ptypes/timestamp,@io_bazel_rules_go//proto/wkt:timestamp_go_proto
google/protobuf/type.proto,@com_google_protobuf//:type_proto,google.golang.org/genproto/protobuf/ptype,@io_bazel_rules_go//proto/wkt:type_go_proto
google/protobuf/wrappers.proto,@com_google_protobuf//:wrappers_proto,github.com/golang/protobuf/ptypes/wrappers,@io_bazel_rules_go//proto/wkt:wrappers_go_proto
`

func main() {
	log.SetPrefix(progName + ": ")
	log.SetFlags(0)
	flag.Parse()

	if *protoCsvPath == "" {
		log.Fatal("-proto must be set")
	}
	if *googleapisRootPath == "" {
		log.Fatal("-go_googleapis must be set")
	}

	protoContent := &bytes.Buffer{}
	protoContent.WriteString(prefix)

	err := filepath.Walk(*googleapisRootPath, func(path string, info os.FileInfo, err error) error {
		if err != nil {
			return err
		}
		if !strings.HasSuffix(path, ".proto") {
			return nil
		}
		relPath, err := filepath.Rel(*googleapisRootPath, path)
		if err != nil || strings.HasPrefix(relPath, "..") {
			log.Panicf("file %q not in googleapisRootPath %q", path, *googleapisRootPath)
		}
		relPath = filepath.ToSlash(relPath)

		if strings.HasPrefix(relPath, "google/api/experimental/") {
			// Special case: these protos need to be built together with protos in
			// google/api. They have the same 'option go_package'. The proto_library
			// rule requires them to be in the same Bazel package, so we don't
			// create a build file in experimental.
			packagePath := "google.golang.org/genproto/googleapis/api"
			protoLabel, goLabel := protoLabels("google/api/x", "api")
			fmt.Fprintf(protoContent, "%s,%s,%s,%s\n", relPath, protoLabel, packagePath, goLabel)
			return nil
		}

		packagePath, packageName, err := loadGoPackage(path, relPath)
		if err != nil {
			log.Print(err)
			return nil
		}

		protoLabel, goLabel := protoLabels(relPath, packageName)

		fmt.Fprintf(protoContent, "%s,%s,%s,%s\n", relPath, protoLabel, packagePath, goLabel)
		return nil
	})
	if err != nil {
		log.Fatal(err)
	}

	if err := ioutil.WriteFile(*protoCsvPath, protoContent.Bytes(), 0666); err != nil {
		log.Fatal(err)
	}
}

var goPackageRx = regexp.MustCompile(`option go_package = "([^"]*)"`)

func loadGoPackage(protoPath, relPath string) (packagePath, packageName string, err error) {
	data, err := ioutil.ReadFile(protoPath)
	if err != nil {
		return "", "", err
	}
	m := goPackageRx.FindSubmatch(data)
	if m == nil {
		return "", "", fmt.Errorf("%s: does not contain 'option go_package'", protoPath)
	}
	opt := string(m[1])
	if i := strings.LastIndexByte(opt, ';'); i >= 0 {
		return opt[:i], opt[i+1:], nil
	}
	if i := strings.LastIndexByte(opt, '/'); i >= 0 {
		return opt, opt[i+1:], nil
	}
	return opt, opt, nil
}

func protoLabels(relPath, packageName string) (protoLabel, goLabel string) {
	dir := path.Dir(relPath)
	protoLabel = fmt.Sprintf("@go_googleapis//%s:%s_proto", dir, packageName)
	goLabel = fmt.Sprintf("@go_googleapis//%s:%s_go_proto", dir, packageName)
	return protoLabel, goLabel
}
