"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const git_error_1 = require("./errors/git-error");
const task_1 = require("./tasks/task");
class GitExecutor {
    constructor(binary = 'git', cwd, env, outputHandler) {
        this.binary = binary;
        this.cwd = cwd;
        this.env = env;
        this.outputHandler = outputHandler;
        this._chain = Promise.resolve();
    }
    push(task) {
        return this._chain = this._chain.then(() => __awaiter(this, void 0, void 0, function* () {
            try {
                if (task_1.isEmptyTask(task)) {
                    return task.parser('');
                }
                const raw = yield this.gitResponse(this.binary, task.commands, this.outputHandler);
                const data = yield this.handleTaskData(task, raw);
                return task_1.isBufferTask(task) ? task.parser(data) : task.parser(data.toString(task.format));
            }
            catch (e) {
                this._chain = Promise.resolve();
                if (e instanceof git_error_1.GitError) {
                    e.task = task;
                    throw e;
                }
                throw new git_error_1.GitError(task, e && String(e));
            }
        }));
    }
    handleTaskData({ onError, concatStdErr }, { exitCode, stdOut, stdErr }) {
        return new Promise((done, fail) => {
            if (exitCode && stdErr.length && onError) {
                return onError(exitCode, Buffer.concat([...(concatStdErr ? stdOut : []), ...stdErr]).toString('utf-8'), (result) => {
                    done(Buffer.from(Buffer.isBuffer(result) ? result : String(result)));
                }, fail);
            }
            if (exitCode && stdErr.length) {
                return fail(Buffer.concat(stdErr).toString('utf-8'));
            }
            if (concatStdErr) {
                stdOut.push(...stdErr);
            }
            done(Buffer.concat(stdOut));
        });
    }
    gitResponse(command, args, outputHandler) {
        return __awaiter(this, void 0, void 0, function* () {
            const spawnOptions = {
                cwd: this.cwd,
                env: this.env,
                windowsHide: true,
            };
            return new Promise((done) => {
                const stdOut = [];
                const stdErr = [];
                let attempted = false;
                function attemptClose(exitCode) {
                    // closing when there is content, terminate immediately
                    if (attempted || stdErr.length || stdOut.length) {
                        done({
                            stdOut,
                            stdErr,
                            exitCode,
                        });
                        attempted = true;
                    }
                    // first attempt at closing but no content yet, wait briefly for the close/exit that may follow
                    if (!attempted) {
                        attempted = true;
                        setTimeout(() => attemptClose(exitCode), 50);
                    }
                }
                const spawned = child_process_1.spawn(command, args, spawnOptions);
                spawned.stdout.on('data', (buffer) => stdOut.push(buffer));
                spawned.stderr.on('data', (buffer) => stdErr.push(buffer));
                spawned.on('error', (err) => stdErr.push(Buffer.from(String(err.stack), 'ascii')));
                spawned.on('close', (code) => attemptClose(code));
                spawned.on('exit', attemptClose);
                if (outputHandler) {
                    outputHandler(command[0], spawned.stdout, spawned.stderr);
                }
            });
        });
    }
}
exports.GitExecutor = GitExecutor;
//# sourceMappingURL=git-executor.js.map