"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const FileStatusSummary_1 = require("./FileStatusSummary");
/**
 * The StatusSummary is returned as a response to getting `git().status()`
 */
class StatusSummary {
    constructor() {
        this.not_added = [];
        this.conflicted = [];
        this.created = [];
        this.deleted = [];
        this.modified = [];
        this.renamed = [];
        /**
         * All files represented as an array of objects containing the `path` and status in `index` and
         * in the `working_dir`.
         */
        this.files = [];
        this.staged = [];
        /**
         * Number of commits ahead of the tracked branch
         */
        this.ahead = 0;
        /**
         *Number of commits behind the tracked branch
         */
        this.behind = 0;
        /**
         * Name of the current branch
         */
        this.current = null;
        /**
         * Name of the branch being tracked
         */
        this.tracking = null;
    }
    /**
     * Gets whether this StatusSummary represents a clean working branch.
     */
    isClean() {
        return !this.files.length;
    }
}
exports.StatusSummary = StatusSummary;
exports.StatusSummaryParsers = {
    '##': function (line, status) {
        const aheadReg = /ahead (\d+)/;
        const behindReg = /behind (\d+)/;
        const currentReg = /^(.+?(?=(?:\.{3}|\s|$)))/;
        const trackingReg = /\.{3}(\S*)/;
        let regexResult;
        regexResult = aheadReg.exec(line);
        status.ahead = regexResult && +regexResult[1] || 0;
        regexResult = behindReg.exec(line);
        status.behind = regexResult && +regexResult[1] || 0;
        regexResult = currentReg.exec(line);
        status.current = regexResult && regexResult[1];
        regexResult = trackingReg.exec(line);
        status.tracking = regexResult && regexResult[1];
    },
    '??': function (line, status) {
        status.not_added.push(line);
    },
    A: function (line, status) {
        status.created.push(line);
    },
    AM: function (line, status) {
        status.created.push(line);
    },
    D: function (line, status) {
        status.deleted.push(line);
    },
    M: function (line, status, indexState) {
        status.modified.push(line);
        if (indexState === 'M') {
            status.staged.push(line);
        }
    },
    R: function (line, status) {
        const detail = /^(.+) -> (.+)$/.exec(line) || [null, line, line];
        status.renamed.push({
            from: String(detail[1]),
            to: String(detail[2])
        });
    },
    UU: function (line, status) {
        status.conflicted.push(line);
    }
};
exports.StatusSummaryParsers.MM = exports.StatusSummaryParsers.M;
/* Map all unmerged status code combinations to UU to mark as conflicted */
exports.StatusSummaryParsers.AA = exports.StatusSummaryParsers.UU;
exports.StatusSummaryParsers.UD = exports.StatusSummaryParsers.UU;
exports.StatusSummaryParsers.DU = exports.StatusSummaryParsers.UU;
exports.StatusSummaryParsers.DD = exports.StatusSummaryParsers.UU;
exports.StatusSummaryParsers.AU = exports.StatusSummaryParsers.UU;
exports.StatusSummaryParsers.UA = exports.StatusSummaryParsers.UU;
exports.parseStatusSummary = function (text) {
    let file;
    const lines = text.trim().split('\n');
    const status = new StatusSummary();
    for (let i = 0, l = lines.length; i < l; i++) {
        file = splitLine(lines[i]);
        if (!file) {
            continue;
        }
        if (file.handler) {
            file.handler(file.path, status, file.index, file.workingDir);
        }
        if (file.code !== '##') {
            status.files.push(new FileStatusSummary_1.FileStatusSummary(file.path, file.index, file.workingDir));
        }
    }
    return status;
};
function splitLine(lineStr) {
    let line = lineStr.trim().match(/(..?)(\s+)(.*)/);
    if (!line || !line[1].trim()) {
        line = lineStr.trim().match(/(..?)\s+(.*)/);
    }
    if (!line) {
        return;
    }
    let code = line[1];
    if (line[2].length > 1) {
        code += ' ';
    }
    if (code.length === 1 && line[2].length === 1) {
        code = ' ' + code;
    }
    return {
        raw: code,
        code: code.trim(),
        index: code.charAt(0),
        workingDir: code.charAt(1),
        handler: exports.StatusSummaryParsers[code.trim()],
        path: line[3]
    };
}
//# sourceMappingURL=StatusSummary.js.map