'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

function _interopDefault (ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

const tslib = require('tslib');
const graphql = require('graphql');
const stringify = _interopDefault(require('fast-json-stable-stringify'));
const utils = require('@graphql-tools/utils/es5');
const schema = require('@graphql-tools/schema/es5');

function isRef(maybeRef) {
    return !!(maybeRef && typeof maybeRef === 'object' && '$ref' in maybeRef);
}
function assertIsRef(maybeRef, message) {
    if (!isRef(maybeRef)) {
        throw new Error(message || "Expected " + maybeRef + " to be a valid Ref.");
    }
}
function isRecord(obj) {
    return typeof obj === 'object' && obj !== null;
}

function uuidv4() {
    return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
        var r = (Math.random() * 16) | 0;
        // eslint-disable-next-line eqeqeq
        var v = c == 'x' ? r : (r & 0x3) | 0x8;
        return v.toString(16);
    });
}
var randomListLength = function () {
    // Mocking has always returned list of length 2 by default
    // return 1 + Math.round(Math.random() * 10)
    return 2;
};
var takeRandom = function (arr) { return arr[Math.floor(Math.random() * arr.length)]; };
function makeRef(typeName, key) {
    return { $ref: { key: key, typeName: typeName } };
}
function isObject(thing) {
    return thing === Object(thing) && !Array.isArray(thing);
}
function copyOwnPropsIfNotPresent(target, source) {
    var e_1, _a;
    try {
        for (var _b = tslib.__values(Object.getOwnPropertyNames(source)), _c = _b.next(); !_c.done; _c = _b.next()) {
            var prop = _c.value;
            if (!Object.getOwnPropertyDescriptor(target, prop)) {
                var propertyDescriptor = Object.getOwnPropertyDescriptor(source, prop);
                Object.defineProperty(target, prop, propertyDescriptor == null ? {} : propertyDescriptor);
            }
        }
    }
    catch (e_1_1) { e_1 = { error: e_1_1 }; }
    finally {
        try {
            if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
        }
        finally { if (e_1) throw e_1.error; }
    }
}
function copyOwnProps(target) {
    var e_2, _a;
    var sources = [];
    for (var _i = 1; _i < arguments.length; _i++) {
        sources[_i - 1] = arguments[_i];
    }
    try {
        for (var sources_1 = tslib.__values(sources), sources_1_1 = sources_1.next(); !sources_1_1.done; sources_1_1 = sources_1.next()) {
            var source = sources_1_1.value;
            var chain = source;
            while (chain != null) {
                copyOwnPropsIfNotPresent(target, chain);
                chain = Object.getPrototypeOf(chain);
            }
        }
    }
    catch (e_2_1) { e_2 = { error: e_2_1 }; }
    finally {
        try {
            if (sources_1_1 && !sources_1_1.done && (_a = sources_1.return)) _a.call(sources_1);
        }
        finally { if (e_2) throw e_2.error; }
    }
    return target;
}

/**
 * @internal
 */
function isMockList(obj) {
    if (typeof (obj === null || obj === void 0 ? void 0 : obj.len) === 'number' || (Array.isArray(obj === null || obj === void 0 ? void 0 : obj.len) && typeof (obj === null || obj === void 0 ? void 0 : obj.len[0]) === 'number')) {
        if (typeof obj.wrappedFunction === 'undefined' || typeof obj.wrappedFunction === 'function') {
            return true;
        }
    }
    return false;
}
/**
 * This is an object you can return from your mock resolvers which calls the
 * provided `mockFunction` once for each list item.
 */
var MockList = /** @class */ (function () {
    /**
     * @param length Either the exact length of items to return or an inclusive
     * range of possible lengths.
     * @param mockFunction The function to call for each item in the list to
     * resolve it. It can return another MockList or a value.
     */
    function MockList(length, mockFunction) {
        this.len = length;
        if (typeof mockFunction !== 'undefined') {
            if (typeof mockFunction !== 'function') {
                throw new Error('Second argument to MockList must be a function or undefined');
            }
            this.wrappedFunction = mockFunction;
        }
    }
    /**
     * @internal
     */
    MockList.prototype.mock = function () {
        var arr;
        if (Array.isArray(this.len)) {
            arr = new Array(this.randint(this.len[0], this.len[1]));
        }
        else {
            arr = new Array(this.len);
        }
        for (var i = 0; i < arr.length; i++) {
            if (typeof this.wrappedFunction === 'function') {
                var res = this.wrappedFunction();
                if (isMockList(res)) {
                    arr[i] = res.mock();
                }
                else {
                    arr[i] = res;
                }
            }
            else {
                arr[i] = undefined;
            }
        }
        return arr;
    };
    MockList.prototype.randint = function (low, high) {
        return Math.floor(Math.random() * (high - low + 1) + low);
    };
    return MockList;
}());
function deepResolveMockList(mockList) {
    return mockList.mock().map(function (v) {
        if (isMockList(v))
            return deepResolveMockList(v);
        return v;
    });
}

var defaultMocks = {
    Int: function () { return Math.round(Math.random() * 200) - 100; },
    Float: function () { return Math.random() * 200 - 100; },
    String: function () { return 'Hello World'; },
    Boolean: function () { return Math.random() > 0.5; },
    ID: function () { return uuidv4(); },
};
var defaultKeyFieldNames = ['id', '_id'];
var MockStore = /** @class */ (function () {
    function MockStore(_a) {
        var schema = _a.schema, mocks = _a.mocks, typePolicies = _a.typePolicies;
        this.store = {};
        this.schema = schema;
        this.mocks = tslib.__assign(tslib.__assign({}, defaultMocks), mocks);
        this.typePolicies = typePolicies || {};
    }
    MockStore.prototype.get = function (_typeName, _key, _fieldName, _fieldArgs) {
        if (typeof _typeName !== 'string') {
            if (_key === undefined) {
                if (isRef(_typeName)) {
                    throw new Error("Can't provide a ref as first argument and no other argument");
                }
                // get({...})
                return this.getImpl(_typeName);
            }
            else {
                assertIsRef(_typeName);
                var $ref = _typeName.$ref;
                // arguments shift
                _fieldArgs = _fieldName;
                _fieldName = _key;
                _key = $ref.key;
                _typeName = $ref.typeName;
            }
        }
        var args = {
            typeName: _typeName,
        };
        if (isRecord(_key) || _key === undefined) {
            // get('User', { name: 'Alex'})
            args.defaultValue = _key;
            return this.getImpl(args);
        }
        args.key = _key;
        if (Array.isArray(_fieldName) && _fieldName.length === 1) {
            _fieldName = _fieldName[0];
        }
        if (typeof _fieldName !== 'string' && !Array.isArray(_fieldName)) {
            // get('User', 'me', { name: 'Alex'})
            args.defaultValue = _fieldName;
            return this.getImpl(args);
        }
        if (Array.isArray(_fieldName)) {
            // get('User', 'me', ['father', 'name'])
            var ref = this.get(_typeName, _key, _fieldName[0], _fieldArgs);
            assertIsRef(ref);
            return this.get(ref.$ref.typeName, ref.$ref.key, _fieldName.slice(1, _fieldName.length));
        }
        // get('User', 'me', 'name'...);
        args.fieldName = _fieldName;
        args.fieldArgs = _fieldArgs;
        return this.getImpl(args);
    };
    MockStore.prototype.set = function (_typeName, _key, _fieldName, _value) {
        if (typeof _typeName !== 'string') {
            if (_key === undefined) {
                if (isRef(_typeName)) {
                    throw new Error("Can't provide a ref as first argument and no other argument");
                }
                // set({...})
                return this.setImpl(_typeName);
            }
            else {
                assertIsRef(_typeName);
                var $ref = _typeName.$ref;
                // arguments shift
                _value = _fieldName;
                _fieldName = _key;
                _key = $ref.key;
                _typeName = $ref.typeName;
            }
        }
        assertIsDefined(_key, 'key was not provided');
        var args = {
            typeName: _typeName,
            key: _key,
        };
        if (typeof _fieldName !== 'string') {
            // set('User', 1, { name: 'Foo' })
            if (!isRecord(_fieldName))
                throw new Error('Expected value to be a record');
            args.value = _fieldName;
            return this.setImpl(args);
        }
        args.fieldName = _fieldName;
        args.value = _value;
        return this.setImpl(args);
    };
    MockStore.prototype.reset = function () {
        this.store = {};
    };
    MockStore.prototype.filter = function (key, predicate) {
        var entity = this.store[key];
        return Object.values(entity).filter(predicate);
    };
    MockStore.prototype.find = function (key, predicate) {
        var entity = this.store[key];
        return Object.values(entity).find(predicate);
    };
    MockStore.prototype.getImpl = function (args) {
        var _this = this;
        var typeName = args.typeName, key = args.key, fieldName = args.fieldName, fieldArgs = args.fieldArgs, defaultValue = args.defaultValue;
        if (!fieldName) {
            if (defaultValue !== undefined && !isRecord(defaultValue)) {
                throw new Error('`defaultValue` should be an object');
            }
            var valuesToInsert = defaultValue || {};
            if (key) {
                valuesToInsert = tslib.__assign(tslib.__assign({}, valuesToInsert), makeRef(typeName, key));
            }
            return this.insert(typeName, valuesToInsert, true);
        }
        assertIsDefined(key, 'key argument should be given when fieldName is given');
        var fieldNameInStore = getFieldNameInStore(fieldName, fieldArgs);
        if (this.store[typeName] === undefined ||
            this.store[typeName][key] === undefined ||
            this.store[typeName][key][fieldNameInStore] === undefined) {
            var value = void 0;
            if (defaultValue !== undefined) {
                value = defaultValue;
            }
            else if (this.isKeyField(typeName, fieldName)) {
                value = key;
            }
            else {
                value = this.generateFieldValue(typeName, fieldName, function (otherFieldName, otherValue) {
                    // if we get a key field in the mix we don't care
                    if (_this.isKeyField(typeName, otherFieldName))
                        return;
                    _this.set({ typeName: typeName, key: key, fieldName: otherFieldName, value: otherValue, noOverride: true });
                });
            }
            this.set({ typeName: typeName, key: key, fieldName: fieldName, fieldArgs: fieldArgs, value: value, noOverride: true });
        }
        return this.store[typeName][key][fieldNameInStore];
    };
    MockStore.prototype.setImpl = function (args) {
        var _a;
        var _this = this;
        var typeName = args.typeName, key = args.key, fieldName = args.fieldName, fieldArgs = args.fieldArgs, noOverride = args.noOverride;
        var value = args.value;
        if (isMockList(value)) {
            value = deepResolveMockList(value);
        }
        if (!fieldName) {
            if (!isRecord(value)) {
                throw new Error('When no `fieldName` is provided, `value` should be a record.');
            }
            for (var fieldName_1 in value) {
                this.setImpl({
                    typeName: typeName,
                    key: key,
                    fieldName: fieldName_1,
                    value: value[fieldName_1],
                    noOverride: noOverride,
                });
            }
            return;
        }
        var fieldNameInStore = getFieldNameInStore(fieldName, fieldArgs);
        if (this.isKeyField(typeName, fieldName) && value !== key) {
            throw new Error("Field " + fieldName + " is a key field of " + typeName + " and you are trying to set it to " + value + " while the key is " + key);
        }
        if (this.store[typeName] === undefined) {
            this.store[typeName] = {};
        }
        if (this.store[typeName][key] === undefined) {
            this.store[typeName][key] = {};
        }
        // if already set and we don't override
        if (this.store[typeName][key][fieldNameInStore] !== undefined && noOverride) {
            return;
        }
        var fieldType = this.getFieldType(typeName, fieldName);
        var currentValue = this.store[typeName][key][fieldNameInStore];
        var valueToStore;
        try {
            valueToStore = this.normalizeValueToStore(fieldType, value, currentValue, function (typeName, values) {
                return _this.insert(typeName, values, noOverride);
            });
        }
        catch (e) {
            throw new Error("Value to set in " + typeName + "." + fieldName + " in not normalizable: " + e.message);
        }
        this.store[typeName][key] = tslib.__assign(tslib.__assign({}, this.store[typeName][key]), (_a = {}, _a[fieldNameInStore] = valueToStore, _a));
    };
    MockStore.prototype.normalizeValueToStore = function (fieldType, value, currentValue, onInsertType) {
        var _this = this;
        var fieldTypeName = fieldType.toString();
        if (value === null) {
            if (!graphql.isNullableType(fieldType)) {
                throw new Error("should not be null because " + fieldTypeName + " is not nullable. Received null.");
            }
            return null;
        }
        var nullableFieldType = graphql.getNullableType(fieldType);
        if (value === undefined)
            return this.generateValueFromType(nullableFieldType);
        // deal with nesting insert
        if (graphql.isCompositeType(nullableFieldType)) {
            if (!isRecord(value))
                throw new Error("should be an object or null or undefined. Received " + value);
            var joinedTypeName = void 0;
            if (graphql.isAbstractType(nullableFieldType)) {
                if (isRef(value)) {
                    joinedTypeName = value.$ref.typeName;
                }
                else {
                    if (typeof value['__typename'] !== 'string') {
                        throw new Error("should contain a '__typename' because " + nullableFieldType.name + " an abstract type");
                    }
                    joinedTypeName = value['__typename'];
                }
            }
            else {
                joinedTypeName = nullableFieldType.name;
            }
            return onInsertType(joinedTypeName, isRef(currentValue) ? tslib.__assign(tslib.__assign({}, currentValue), value) : value);
        }
        if (graphql.isListType(nullableFieldType)) {
            if (!Array.isArray(value))
                throw new Error("should be an array or null or undefined. Received " + value);
            return value.map(function (v, index) {
                return _this.normalizeValueToStore(nullableFieldType.ofType, v, typeof currentValue === 'object' && currentValue != null && currentValue[index] ? currentValue : undefined, onInsertType);
            });
        }
        return value;
    };
    MockStore.prototype.insert = function (typeName, values, noOverride) {
        var keyFieldName = this.getKeyFieldName(typeName);
        var key;
        // when we generate a key for the type, we might produce
        // other associated values with it
        // We keep track of them and we'll insert them, with propririty
        // for the ones that we areasked to insert
        var otherValues = {};
        if (isRef(values)) {
            key = values.$ref.key;
        }
        else if (keyFieldName && keyFieldName in values) {
            key = values[keyFieldName];
        }
        else {
            key = this.generateKeyForType(typeName, function (otherFieldName, otherFieldValue) {
                otherValues[otherFieldName] = otherFieldValue;
            });
        }
        var toInsert = tslib.__assign(tslib.__assign({}, otherValues), values);
        for (var fieldName in toInsert) {
            if (fieldName === '$ref')
                continue;
            if (fieldName === '__typename')
                continue;
            this.set({
                typeName: typeName,
                key: key,
                fieldName: fieldName,
                value: toInsert[fieldName],
                noOverride: noOverride,
            });
        }
        return makeRef(typeName, key);
    };
    MockStore.prototype.generateFieldValue = function (typeName, fieldName, onOtherFieldsGenerated) {
        var mockedValue = this.generateFieldValueFromMocks(typeName, fieldName, onOtherFieldsGenerated);
        if (mockedValue !== undefined)
            return mockedValue;
        var fieldType = this.getFieldType(typeName, fieldName);
        return this.generateValueFromType(fieldType);
    };
    MockStore.prototype.generateFieldValueFromMocks = function (typeName, fieldName, onOtherFieldsGenerated) {
        var e_1, _a;
        var value;
        var mock = this.mocks ? this.mocks[typeName] : undefined;
        if (mock) {
            if (typeof mock === 'function') {
                var values = mock();
                if (typeof values !== 'object' || values == null) {
                    throw new Error("Value returned by the mock for " + typeName + " is not an object");
                }
                for (var otherFieldName in values) {
                    if (otherFieldName === fieldName)
                        continue;
                    if (typeof values[otherFieldName] === 'function')
                        continue;
                    onOtherFieldsGenerated && onOtherFieldsGenerated(otherFieldName, values[otherFieldName]);
                }
                value = values[fieldName];
                if (typeof value === 'function')
                    value = value();
            }
            else if (typeof mock === 'object' && mock != null && typeof mock[fieldName] === 'function') {
                value = mock[fieldName]();
            }
        }
        if (value !== undefined)
            return value;
        var type = this.getType(typeName);
        // GraphQL 14 Compatibility
        var interfaces = 'getInterfaces' in type ? type.getInterfaces() : [];
        if (interfaces.length > 0) {
            try {
                for (var interfaces_1 = tslib.__values(interfaces), interfaces_1_1 = interfaces_1.next(); !interfaces_1_1.done; interfaces_1_1 = interfaces_1.next()) {
                    var interface_ = interfaces_1_1.value;
                    if (value)
                        break;
                    value = this.generateFieldValueFromMocks(interface_.name, fieldName, onOtherFieldsGenerated);
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (interfaces_1_1 && !interfaces_1_1.done && (_a = interfaces_1.return)) _a.call(interfaces_1);
                }
                finally { if (e_1) throw e_1.error; }
            }
        }
        return value;
    };
    MockStore.prototype.generateKeyForType = function (typeName, onOtherFieldsGenerated) {
        var keyFieldName = this.getKeyFieldName(typeName);
        if (!keyFieldName)
            return uuidv4();
        return this.generateFieldValue(typeName, keyFieldName, onOtherFieldsGenerated);
    };
    MockStore.prototype.generateValueFromType = function (fieldType) {
        var _this = this;
        var nullableType = graphql.getNullableType(fieldType);
        if (graphql.isScalarType(nullableType)) {
            var mockFn = this.mocks[nullableType.name];
            if (typeof mockFn !== 'function')
                throw new Error("No mock defined for type \"" + nullableType.name + "\"");
            return mockFn();
        }
        else if (graphql.isEnumType(nullableType)) {
            var mockFn = this.mocks[nullableType.name];
            if (typeof mockFn === 'function')
                return mockFn();
            var values = nullableType.getValues().map(function (v) { return v.value; });
            return takeRandom(values);
        }
        else if (graphql.isObjectType(nullableType)) {
            // this will create a new random ref
            return this.insert(nullableType.name, {});
        }
        else if (graphql.isListType(nullableType)) {
            return tslib.__spreadArray([], tslib.__read(new Array(randomListLength())), false).map(function () { return _this.generateValueFromType(nullableType.ofType); });
        }
        else if (graphql.isAbstractType(nullableType)) {
            var mock = this.mocks[nullableType.name];
            var typeName = void 0;
            var values = {};
            if (!mock) {
                typeName = takeRandom(this.schema.getPossibleTypes(nullableType).map(function (t) { return t.name; }));
            }
            else if (typeof mock === 'function') {
                var mockRes = mock();
                if (mockRes === null)
                    return null;
                if (!isRecord(mockRes)) {
                    throw new Error("Value returned by the mock for " + nullableType.name + " is not an object or null");
                }
                values = mockRes;
                if (typeof values['__typename'] !== 'string') {
                    throw new Error("Please return a __typename in \"" + nullableType.name + "\"");
                }
                typeName = values['__typename'];
            }
            else if (typeof mock === 'object' && mock != null && typeof mock['__typename'] === 'function') {
                var mockRes = mock['__typename']();
                if (typeof mockRes !== 'string')
                    throw new Error("'__typename' returned by the mock for abstract type " + nullableType.name + " is not a string");
                typeName = mockRes;
            }
            else {
                throw new Error("Please return a __typename in \"" + nullableType.name + "\"");
            }
            var toInsert = {};
            for (var fieldName in values) {
                if (fieldName === '__typename')
                    continue;
                var fieldValue = values[fieldName];
                toInsert[fieldName] = typeof fieldValue === 'function' ? fieldValue() : fieldValue;
            }
            return this.insert(typeName, toInsert);
        }
        else {
            throw new Error(nullableType + " not implemented");
        }
    };
    MockStore.prototype.getFieldType = function (typeName, fieldName) {
        if (fieldName === '__typename') {
            return graphql.GraphQLString;
        }
        var type = this.getType(typeName);
        var field = type.getFields()[fieldName];
        if (!field) {
            throw new Error(fieldName + " does not exist on type " + typeName);
        }
        return field.type;
    };
    MockStore.prototype.getType = function (typeName) {
        var type = this.schema.getType(typeName);
        if (!type || !(graphql.isObjectType(type) || graphql.isInterfaceType(type))) {
            throw new Error(typeName + " does not exist on schema or is not an object or interface");
        }
        return type;
    };
    MockStore.prototype.isKeyField = function (typeName, fieldName) {
        return this.getKeyFieldName(typeName) === fieldName;
    };
    MockStore.prototype.getKeyFieldName = function (typeName) {
        var _a;
        var typePolicyKeyField = (_a = this.typePolicies[typeName]) === null || _a === void 0 ? void 0 : _a.keyFieldName;
        if (typePolicyKeyField !== undefined) {
            if (typePolicyKeyField === false)
                return null;
            return typePolicyKeyField;
        }
        // How about common key field names?
        var gqlType = this.getType(typeName);
        for (var fieldName in gqlType.getFields()) {
            if (defaultKeyFieldNames.includes(fieldName)) {
                return fieldName;
            }
        }
        return null;
    };
    return MockStore;
}());
var getFieldNameInStore = function (fieldName, fieldArgs) {
    if (!fieldArgs)
        return fieldName;
    if (typeof fieldArgs === 'string') {
        return fieldName + ":" + fieldArgs;
    }
    // empty args
    if (Object.keys(fieldArgs).length === 0) {
        return fieldName;
    }
    return fieldName + ":" + stringify(fieldArgs);
};
function assertIsDefined(value, message) {
    if (value !== undefined && value !== null) {
        return;
    }
    throw new Error(process.env['NODE_ENV'] === 'production' ? 'Invariant failed:' : "Invariant failed: " + (message || ''));
}
/**
 * Will create `MockStore` for the given `schema`.
 *
 * A `MockStore` will generate mock values for the given schem when queried.
 *
 * It will stores generated mocks, so that, provided with same arguments
 * the returned values will be the same.
 *
 * Its API also allows to modify the stored values.
 *
 * Basic example:
 * ```ts
 * store.get('User', 1, 'name');
 * // > "Hello World"
 * store.set('User', 1, 'name', 'Alexandre');
 * store.get('User', 1, 'name');
 * // > "Alexandre"
 * ```
 *
 * The storage key will correspond to the "key field"
 * of the type. Field with name `id` or `_id` will be
 * by default considered as the key field for the type.
 * However, use `typePolicies` to precise the field to use
 * as key.
 */
function createMockStore(options) {
    return new MockStore(options);
}

// todo: add option to preserve resolver
/**
 * Given a `schema` and a `MockStore`, returns an executable schema that
 * will use the provided `MockStore` to execute queries.
 *
 * ```ts
 * const schema = buildSchema(`
 *  type User {
 *    id: ID!
 *    name: String!
 *  }
 *  type Query {
 *    me: User!
 *  }
 * `)
 *
 * const store = createMockStore({ schema });
 * const mockedSchema = addMocksToSchema({ schema, store });
 * ```
 *
 *
 * If a `resolvers` parameter is passed, the query execution will use
 * the provided `resolvers` if, one exists, instead of the default mock
 * resolver.
 *
 *
 * ```ts
 * const schema = buildSchema(`
 *   type User {
 *     id: ID!
 *     name: String!
 *   }
 *   type Query {
 *     me: User!
 *   }
 *   type Mutation {
 *     setMyName(newName: String!): User!
 *   }
 * `)
 *
 * const store = createMockStore({ schema });
 * const mockedSchema = addMocksToSchema({
 *   schema,
 *   store,
 *   resolvers: {
 *     Mutation: {
 *       setMyName: (_, { newName }) => {
 *          const ref = store.get('Query', 'ROOT', 'viewer');
 *          store.set(ref, 'name', newName);
 *          return ref;
 *       }
 *     }
 *   }
 *  });
 * ```
 *
 *
 * `Query` and `Mutation` type will use `key` `'ROOT'`.
 */
function addMocksToSchema(_a) {
    var _b;
    var _this = this;
    var schema$1 = _a.schema, maybeStore = _a.store, mocks = _a.mocks, typePolicies = _a.typePolicies, resolversOrFnResolvers = _a.resolvers, _c = _a.preserveResolvers, preserveResolvers = _c === void 0 ? false : _c;
    if (!schema$1) {
        throw new Error('Must provide schema to mock');
    }
    if (!graphql.isSchema(schema$1)) {
        throw new Error('Value at "schema" must be of type GraphQLSchema');
    }
    if (mocks && !isObject(mocks)) {
        throw new Error('mocks must be of type Object');
    }
    var store = maybeStore ||
        createMockStore({
            schema: schema$1,
            mocks: mocks,
            typePolicies: typePolicies,
        });
    var resolvers = typeof resolversOrFnResolvers === 'function' ? resolversOrFnResolvers(store) : resolversOrFnResolvers;
    var mockResolver = function (source, args, contex, info) {
        var defaultResolvedValue = graphql.defaultFieldResolver(source, args, contex, info);
        // priority to default resolved value
        if (defaultResolvedValue !== undefined)
            return defaultResolvedValue;
        if (isRef(source)) {
            return store.get({
                typeName: source.$ref.typeName,
                key: source.$ref.key,
                fieldName: info.fieldName,
                fieldArgs: args,
            });
        }
        // we have to handle the root mutation, root query and root subscription types
        // differently, because no resolver is called at the root
        if (isRootType(info.parentType, info.schema)) {
            return store.get({
                typeName: info.parentType.name,
                key: 'ROOT',
                fieldName: info.fieldName,
                fieldArgs: args,
            });
        }
        return undefined;
    };
    var typeResolver = function (data) {
        if (isRef(data)) {
            return data.$ref.typeName;
        }
    };
    var mockSubscriber = function () {
        var _a;
        return (_a = {},
            _a[Symbol.asyncIterator] = function () {
                return {
                    next: function () {
                        return tslib.__awaiter(this, void 0, void 0, function () {
                            return tslib.__generator(this, function (_a) {
                                return [2 /*return*/, {
                                        done: true,
                                        value: {},
                                    }];
                            });
                        });
                    },
                };
            },
            _a);
    };
    var schemaWithMocks = utils.mapSchema(schema$1, (_b = {},
        _b[utils.MapperKind.OBJECT_FIELD] = function (fieldConfig) {
            var newFieldConfig = tslib.__assign({}, fieldConfig);
            var oldResolver = fieldConfig.resolve;
            if (!preserveResolvers || !oldResolver) {
                newFieldConfig.resolve = mockResolver;
            }
            else {
                newFieldConfig.resolve = function (rootObject, args, context, info) { return tslib.__awaiter(_this, void 0, void 0, function () {
                    var _a, mockedValue, resolvedValue, emptyObject;
                    return tslib.__generator(this, function (_b) {
                        switch (_b.label) {
                            case 0: return [4 /*yield*/, Promise.all([
                                    mockResolver(rootObject, args, context, info),
                                    oldResolver(rootObject, args, context, info),
                                ])];
                            case 1:
                                _a = tslib.__read.apply(void 0, [_b.sent(), 2]), mockedValue = _a[0], resolvedValue = _a[1];
                                // In case we couldn't mock
                                if (mockedValue instanceof Error) {
                                    // only if value was not resolved, populate the error.
                                    if (undefined === resolvedValue) {
                                        throw mockedValue;
                                    }
                                    return [2 /*return*/, resolvedValue];
                                }
                                if (resolvedValue instanceof Date && mockedValue instanceof Date) {
                                    return [2 /*return*/, undefined !== resolvedValue ? resolvedValue : mockedValue];
                                }
                                if (isObject(mockedValue) && isObject(resolvedValue)) {
                                    emptyObject = Object.create(Object.getPrototypeOf(resolvedValue));
                                    return [2 /*return*/, copyOwnProps(emptyObject, resolvedValue, mockedValue)];
                                }
                                return [2 /*return*/, undefined !== resolvedValue ? resolvedValue : mockedValue];
                        }
                    });
                }); };
            }
            var fieldSubscriber = fieldConfig.subscribe;
            if (!preserveResolvers || !fieldSubscriber) {
                newFieldConfig.subscribe = mockSubscriber;
            }
            else {
                newFieldConfig.subscribe = function (rootObject, args, context, info) { return tslib.__awaiter(_this, void 0, void 0, function () {
                    var _a, mockAsyncIterable, oldAsyncIterable;
                    return tslib.__generator(this, function (_b) {
                        switch (_b.label) {
                            case 0: return [4 /*yield*/, Promise.all([
                                    mockSubscriber(),
                                    fieldSubscriber(rootObject, args, context, info),
                                ])];
                            case 1:
                                _a = tslib.__read.apply(void 0, [_b.sent(), 2]), mockAsyncIterable = _a[0], oldAsyncIterable = _a[1];
                                return [2 /*return*/, oldAsyncIterable || mockAsyncIterable];
                        }
                    });
                }); };
            }
            return newFieldConfig;
        },
        _b[utils.MapperKind.ABSTRACT_TYPE] = function (type) {
            if (preserveResolvers && type.resolveType != null && type.resolveType.length) {
                return;
            }
            if (graphql.isUnionType(type)) {
                return new graphql.GraphQLUnionType(tslib.__assign(tslib.__assign({}, type.toConfig()), { resolveType: typeResolver }));
            }
            else {
                return new graphql.GraphQLInterfaceType(tslib.__assign(tslib.__assign({}, type.toConfig()), { resolveType: typeResolver }));
            }
        },
        _b));
    return resolvers ? schema.addResolversToSchema(schemaWithMocks, resolvers) : schemaWithMocks;
}
var isRootType = function (type, schema) {
    var rootTypeNames = utils.getRootTypeNames(schema);
    return rootTypeNames.has(type.name);
};

/**
 * A convenience wrapper on top of addMocksToSchema. It adds your mock resolvers
 * to your schema and returns a client that will correctly execute your query with
 * variables. Note: when executing queries from the returned server, context and
 * root will both equal `{}`.
 * @param schema The schema to which to add mocks. This can also be a set of type
 * definitions instead.
 * @param mocks The mocks to add to the schema.
 * @param preserveResolvers Set to `true` to prevent existing resolvers from being
 * overwritten to provide mock data. This can be used to mock some parts of the
 * server and not others.
 */
function mockServer(schema$1, mocks, preserveResolvers) {
    if (preserveResolvers === void 0) { preserveResolvers = false; }
    var mockedSchema = addMocksToSchema({
        schema: graphql.isSchema(schema$1)
            ? schema$1
            : schema.makeExecutableSchema({
                typeDefs: schema$1,
            }),
        mocks: mocks,
        preserveResolvers: preserveResolvers,
    });
    return {
        query: function (query, vars) {
            return graphql.graphql({
                schema: mockedSchema,
                source: query,
                rootValue: {},
                contextValue: {},
                variableValues: vars,
            });
        },
    };
}

exports.MockList = MockList;
exports.MockStore = MockStore;
exports.addMocksToSchema = addMocksToSchema;
exports.assertIsRef = assertIsRef;
exports.createMockStore = createMockStore;
exports.deepResolveMockList = deepResolveMockList;
exports.defaultMocks = defaultMocks;
exports.isMockList = isMockList;
exports.isRecord = isRecord;
exports.isRef = isRef;
exports.mockServer = mockServer;
