"use strict";

/**
 * Module dependencies.
 */
var util = require('util');

var Stream = require('stream');

var ResponseBase = require('../response-base');

var _require = require('../utils'),
    mixin = _require.mixin;
/**
 * Expose `Response`.
 */


module.exports = Response;
/**
 * Initialize a new `Response` with the given `xhr`.
 *
 *  - set flags (.ok, .error, etc)
 *  - parse header
 *
 * @param {Request} req
 * @param {Object} options
 * @constructor
 * @extends {Stream}
 * @implements {ReadableStream}
 * @api private
 */

function Response(request) {
  Stream.call(this);
  this.res = request.res;
  var res = this.res;
  this.request = request;
  this.req = request.req;
  this.text = res.text;
  this.files = res.files || {};
  this.buffered = request._resBuffered;
  this.headers = res.headers;
  this.header = this.headers;

  this._setStatusProperties(res.statusCode);

  this._setHeaderProperties(this.header);

  this.setEncoding = res.setEncoding.bind(res);
  res.on('data', this.emit.bind(this, 'data'));
  res.on('end', this.emit.bind(this, 'end'));
  res.on('close', this.emit.bind(this, 'close'));
  res.on('error', this.emit.bind(this, 'error'));
} // Lazy access res.body.
// https://github.com/nodejs/node/pull/39520#issuecomment-889697136


Object.defineProperty(Response.prototype, 'body', {
  get: function get() {
    return this._body !== undefined ? this._body : this.res.body !== undefined ? this.res.body : {};
  },
  set: function set(value) {
    this._body = value;
  }
});
/**
 * Inherit from `Stream`.
 */

util.inherits(Response, Stream);
mixin(Response.prototype, ResponseBase.prototype);
/**
 * Implements methods of a `ReadableStream`
 */

Response.prototype.destroy = function (error) {
  this.res.destroy(error);
};
/**
 * Pause.
 */


Response.prototype.pause = function () {
  this.res.pause();
};
/**
 * Resume.
 */


Response.prototype.resume = function () {
  this.res.resume();
};
/**
 * Return an `Error` representative of this response.
 *
 * @return {Error}
 * @api public
 */


Response.prototype.toError = function () {
  var req = this.req;
  var method = req.method;
  var path = req.path;
  var message = "cannot ".concat(method, " ").concat(path, " (").concat(this.status, ")");
  var error = new Error(message);
  error.status = this.status;
  error.text = this.text;
  error.method = method;
  error.path = path;
  return error;
};

Response.prototype.setStatusProperties = function (status) {
  console.warn('In superagent 2.x setStatusProperties is a private method');
  return this._setStatusProperties(status);
};
/**
 * To json.
 *
 * @return {Object}
 * @api public
 */


Response.prototype.toJSON = function () {
  return {
    req: this.request.toJSON(),
    header: this.header,
    status: this.status,
    text: this.text
  };
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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