// Copyright 2020 The Kubernetes Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package utils

import (
	"context"
	"encoding/json"
	"fmt"
	"net/url"
	"os"
	"path/filepath"
	"strings"

	"github.com/blang/semver"
	"github.com/ghodss/yaml"
	chnv1 "github.com/open-cluster-management/multicloud-operators-channel/pkg/apis/apps/v1"
	dplv1 "github.com/open-cluster-management/multicloud-operators-deployable/pkg/apis/apps/v1"
	dplutils "github.com/open-cluster-management/multicloud-operators-deployable/pkg/utils"
	releasev1 "github.com/open-cluster-management/multicloud-operators-subscription-release/pkg/apis/apps/v1"
	appv1 "github.com/open-cluster-management/multicloud-operators-subscription/pkg/apis/apps/v1"
	clientsetx "k8s.io/apiextensions-apiserver/pkg/client/clientset/clientset"
	"k8s.io/apimachinery/pkg/api/errors"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	v1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/apis/meta/v1/unstructured"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/types"
	"k8s.io/helm/pkg/chartutil"
	"k8s.io/helm/pkg/repo"
	"k8s.io/klog"
	"sigs.k8s.io/controller-runtime/pkg/client"
)

func GetPackageAlias(sub *appv1.Subscription, packageName string) string {
	for _, overrides := range sub.Spec.PackageOverrides {
		if overrides.PackageName == packageName {
			klog.Infof("Overrides for package %s found", packageName)

			if overrides.PackageAlias != "" {
				return overrides.PackageAlias
			}
		}
	}

	return ""
}

// GenerateHelmIndexFile generate helm repo index file
func GenerateHelmIndexFile(sub *appv1.Subscription, repoRoot string, chartDirs map[string]string) (*repo.IndexFile, error) {
	// Build a helm repo index file
	indexFile := repo.NewIndexFile()

	for chartDir := range chartDirs {
		chartFolderName := filepath.Base(chartDir)
		chartParentDir := strings.Split(chartDir, chartFolderName)[0]

		// Get the relative parent directory from the git repo root
		chartBaseDir := strings.SplitAfter(chartParentDir, repoRoot+"/")[1]

		chartMetadata, err := chartutil.LoadChartfile(chartDir + "Chart.yaml")

		if err != nil {
			klog.Error("There was a problem in generating helm charts index file: ", err.Error())
			return indexFile, err
		}

		indexFile.Add(chartMetadata, chartFolderName, chartBaseDir, "generated-by-multicloud-operators-subscription")
	}

	indexFile.SortEntries()

	err := FilterCharts(sub, indexFile)

	if err != nil {
		return indexFile, err
	}

	return indexFile, nil
}

func CreateOrUpdateHelmChart(
	packageName string,
	releaseCRName string,
	chartVersions repo.ChartVersions,
	client client.Client,
	channel *chnv1.Channel,
	sub *appv1.Subscription) (helmRelease *releasev1.HelmRelease, err error) {
	helmRelease = &releasev1.HelmRelease{}
	err = client.Get(context.TODO(),
		types.NamespacedName{Name: releaseCRName, Namespace: sub.Namespace}, helmRelease)

	source := &releasev1.Source{
		SourceType: releasev1.HelmRepoSourceType,
		HelmRepo: &releasev1.HelmRepo{
			Urls: chartVersions[0].URLs,
		},
	}

	if strings.EqualFold(string(channel.Spec.Type), chnv1.ChannelTypeGitHub) {
		source = &releasev1.Source{
			SourceType: releasev1.GitHubSourceType,
			GitHub: &releasev1.GitHub{
				Urls:      []string{channel.Spec.Pathname},
				ChartPath: chartVersions[0].URLs[0],
				Branch:    GetSubscriptionBranch(sub).Short(),
			},
		}
	}

	if err != nil {
		if errors.IsNotFound(err) {
			klog.V(2).Infof("Create helmRelease %s", releaseCRName)

			helmRelease = &releasev1.HelmRelease{
				TypeMeta: metav1.TypeMeta{
					APIVersion: "apps.open-cluster-management.io/v1",
					Kind:       "HelmRelease",
				},
				ObjectMeta: metav1.ObjectMeta{
					Name:      releaseCRName,
					Namespace: sub.Namespace,
					OwnerReferences: []metav1.OwnerReference{{
						APIVersion: "apps.open-cluster-management.io/v1",
						Kind:       "Subscription",
						Name:       sub.Name,
						UID:        sub.UID,
					}},
				},
				Repo: releasev1.HelmReleaseRepo{
					Source:       source,
					ConfigMapRef: channel.Spec.ConfigMapRef,
					SecretRef:    channel.Spec.SecretRef,
					ChartName:    packageName,
					Version:      chartVersions[0].GetVersion(),
				},
			}
		} else {
			klog.Error("Error in getting existing helm release", err)
			return nil, err
		}
	} else {
		// set kind and apiversion, coz it is not in the resource get from k8s
		helmRelease.APIVersion = "apps.open-cluster-management.io/v1"
		helmRelease.Kind = "HelmRelease"
		klog.V(2).Infof("Update helmRelease repo %s", helmRelease.Name)
		helmRelease.Repo = releasev1.HelmReleaseRepo{
			Source:       source,
			ConfigMapRef: channel.Spec.ConfigMapRef,
			SecretRef:    channel.Spec.SecretRef,
			ChartName:    packageName,
			Version:      chartVersions[0].GetVersion(),
		}
	}

	return helmRelease, nil
}

func Override(helmRelease *releasev1.HelmRelease, sub *appv1.Subscription) error {
	//Overrides with the values provided in the subscription for that package
	overrides := getOverrides(helmRelease.Repo.ChartName, sub)
	data, err := yaml.Marshal(helmRelease)

	if err != nil {
		klog.Error("Failed to mashall ", helmRelease.Name, " err:", err)
		return err
	}

	template := &unstructured.Unstructured{}
	err = yaml.Unmarshal(data, template)

	if err != nil {
		klog.Warning("Processing local deployable with error template:", helmRelease, err)
	}

	template, err = dplutils.OverrideTemplate(template, overrides.ClusterOverrides)

	if err != nil {
		klog.Error("Failed to apply override for instance: ")
		return err
	}

	data, err = yaml.Marshal(template)

	if err != nil {
		klog.Error("Failed to mashall ", helmRelease.Name, " err:", err)
		return err
	}

	err = yaml.Unmarshal(data, helmRelease)

	if err != nil {
		klog.Error("Failed to unmashall ", helmRelease.Name, " err:", err)
		return err
	}

	return nil
}

func CreateHelmCRDeployable(
	repoURL string,
	packageName string,
	chartVersions repo.ChartVersions,
	client client.Client,
	channel *chnv1.Channel,
	sub *appv1.Subscription) (*dplv1.Deployable, error) {
	releaseCRName := GetPackageAlias(sub, packageName)
	if releaseCRName == "" {
		releaseCRName = packageName
		subUID := string(sub.UID)

		if len(subUID) >= 5 {
			releaseCRName += "-" + subUID[:5]
		}
	}

	releaseCRName, err := GetReleaseName(releaseCRName)
	if err != nil {
		return nil, err
	}

	if channel == nil || !strings.EqualFold(string(channel.Spec.Type), chnv1.ChannelTypeGitHub) {
		for i := range chartVersions[0].URLs {
			parsedURL, err := url.Parse(chartVersions[0].URLs[i])

			if err != nil {
				klog.Error("Failed to parse url with error:", err)
				return nil, err
			}

			if parsedURL.Scheme == "local" {
				//make sure there is one and only one slash
				repoURL = strings.TrimSuffix(repoURL, "/") + "/"
				chartVersions[0].URLs[i] = strings.Replace(chartVersions[0].URLs[i], "local://", repoURL, -1)
			}
		}
	}

	helmRelease, err := CreateOrUpdateHelmChart(
		packageName, releaseCRName, chartVersions, client, channel, sub)

	if err != nil {
		klog.Error("Failed to create or update helm chart ", packageName, " err:", err)
		return nil, err
	}

	err = Override(helmRelease, sub)

	if err != nil {
		klog.Error("Failed to override ", helmRelease.Name, " err:", err)
		return nil, err
	}

	if helmRelease.Spec == nil {
		spec := make(map[string]interface{})

		err := yaml.Unmarshal([]byte("{\"\":\"\"}"), &spec)
		if err != nil {
			klog.Error("Failed to create an empty spec for helm release", helmRelease)
			return nil, err
		}

		helmRelease.Spec = spec
	}

	dpl := &dplv1.Deployable{}
	if channel == nil {
		dpl.Name = sub.Name + "-" + packageName + "-" + chartVersions[0].GetVersion()
		dpl.Namespace = sub.Namespace
	} else {
		dpl.Name = channel.Name + "-" + packageName + "-" + chartVersions[0].GetVersion()
		dpl.Namespace = channel.Namespace
	}

	dpl.Spec.Template = &runtime.RawExtension{}
	dpl.Spec.Template.Raw, err = json.Marshal(helmRelease)

	if err != nil {
		klog.Error("Failed to mashall helm release", helmRelease)
		return nil, err
	}

	dplanno := make(map[string]string)
	dplanno[dplv1.AnnotationLocal] = "true"
	dpl.SetAnnotations(dplanno)

	return dpl, nil
}

func getOverrides(packageName string, sub *appv1.Subscription) dplv1.Overrides {
	dploverrides := dplv1.Overrides{}

	for _, overrides := range sub.Spec.PackageOverrides {
		if overrides.PackageName == packageName {
			klog.Infof("Overrides for package %s found", packageName)
			dploverrides.ClusterName = packageName
			dploverrides.ClusterOverrides = make([]dplv1.ClusterOverride, 0)

			for _, override := range overrides.PackageOverrides {
				clusterOverride := dplv1.ClusterOverride{
					RawExtension: runtime.RawExtension{
						Raw: override.RawExtension.Raw,
					},
				}
				dploverrides.ClusterOverrides = append(dploverrides.ClusterOverrides, clusterOverride)
			}

			return dploverrides
		}
	}

	return dploverrides
}

//FilterCharts filters the indexFile by name, tillerVersion, version, digest
func FilterCharts(sub *appv1.Subscription, indexFile *repo.IndexFile) error {
	//Removes all entries from the indexFile with non matching name
	err := removeNoMatchingName(sub, indexFile)
	if err != nil {
		klog.Warning(err)
	}
	//Removes non matching version, tillerVersion, digest
	filterOnVersion(sub, indexFile)
	//Keep only the lastest version if multiple remains after filtering.
	err = takeLatestVersion(indexFile)
	if err != nil {
		klog.Error("Failed to filter on version with error: ", err)
		return err
	}

	return nil
}

//takeLatestVersion if the indexFile contains multiple versions for a given chart, then
//only the latest is kept.
func takeLatestVersion(indexFile *repo.IndexFile) (err error) {
	indexFile.SortEntries()

	for k := range indexFile.Entries {
		//Get return the latest version when version is empty but
		//there is a bug in the masterminds semver used by helm
		// "*" constraint is not working properly
		// "*" is equivalent to ">=0.0.0"
		chartVersion, err := indexFile.Get(k, ">=0.0.0")
		if err != nil {
			klog.Error(err)
			return err
		}

		indexFile.Entries[k] = []*repo.ChartVersion{chartVersion}
	}

	return nil
}

//checkDigest Checks if the digest matches
func checkDigest(sub *appv1.Subscription, chartVersion *repo.ChartVersion) bool {
	if sub != nil {
		if sub.Spec.PackageFilter != nil {
			if sub.Spec.PackageFilter.Annotations != nil {
				if filterDigest, ok := sub.Spec.PackageFilter.Annotations["digest"]; ok {
					return filterDigest == chartVersion.Digest
				}
			}
		}
	}

	klog.V(4).Info("Digest check passed for:", chartVersion)

	return true
}

//removeNoMatchingName Deletes entries that the name doesn't match the name provided in the subscription
func removeNoMatchingName(sub *appv1.Subscription, indexFile *repo.IndexFile) error {
	if sub.Spec.Package != "" {
		keys := make([]string, 0)
		for k := range indexFile.Entries {
			keys = append(keys, k)
		}

		for _, k := range keys {
			if k != sub.Spec.Package {
				delete(indexFile.Entries, k)
			}
		}
	} else {
		return fmt.Errorf("subsciption.spec.name is missing for subscription: %s/%s", sub.Namespace, sub.Name)
	}

	klog.V(4).Info("After name matching:", indexFile)

	return nil
}

//filterOnVersion filters the indexFile with the version, tillerVersion and Digest provided in the subscription
//The version provided in the subscription can be an expression like ">=1.2.3" (see https://github.com/blang/semver)
//The tillerVersion and the digest provided in the subscription must be literals.
func filterOnVersion(sub *appv1.Subscription, indexFile *repo.IndexFile) {
	keys := make([]string, 0)
	for k := range indexFile.Entries {
		keys = append(keys, k)
	}

	for _, k := range keys {
		chartVersions := indexFile.Entries[k]
		newChartVersions := make([]*repo.ChartVersion, 0)

		for index, chartVersion := range chartVersions {
			if checkKeywords(sub, chartVersion) && checkDigest(sub, chartVersion) && checkTillerVersion(sub, chartVersion) && checkVersion(sub, chartVersion) {
				newChartVersions = append(newChartVersions, chartVersions[index])
			}
		}

		if len(newChartVersions) > 0 {
			indexFile.Entries[k] = newChartVersions
		} else {
			delete(indexFile.Entries, k)
		}
	}

	klog.V(4).Info("After version matching:", indexFile)
}

//checkKeywords Checks if the charts has at least 1 keyword from the packageFilter.Keywords array
func checkKeywords(sub *appv1.Subscription, chartVersion *repo.ChartVersion) bool {
	var labelSelector *metav1.LabelSelector
	if sub.Spec.PackageFilter != nil {
		labelSelector = sub.Spec.PackageFilter.LabelSelector
	}

	return KeywordsChecker(labelSelector, chartVersion.Keywords)
}

//checkTillerVersion Checks if the TillerVersion matches
func checkTillerVersion(sub *appv1.Subscription, chartVersion *repo.ChartVersion) bool {
	if sub.Spec.PackageFilter != nil {
		if sub.Spec.PackageFilter.Annotations != nil {
			if filterTillerVersion, ok := sub.Spec.PackageFilter.Annotations["tillerVersion"]; ok {
				tillerVersion := chartVersion.GetTillerVersion()
				if tillerVersion != "" {
					tillerVersionVersion, err := semver.ParseRange(tillerVersion)
					if err != nil {
						klog.Errorf("Error while parsing tillerVersion: %s of %s Error: %s", tillerVersion, chartVersion.GetName(), err.Error())
						return false
					}

					filterTillerVersion, err := semver.Parse(filterTillerVersion)

					if err != nil {
						klog.Error(err)
						return false
					}

					return tillerVersionVersion(filterTillerVersion)
				}
			}
		}
	}

	klog.V(4).Info("Tiller check passed for:", chartVersion)

	return true
}

//checkVersion checks if the version matches
func checkVersion(sub *appv1.Subscription, chartVersion *repo.ChartVersion) bool {
	if sub.Spec.PackageFilter != nil {
		if sub.Spec.PackageFilter.Version != "" {
			version := chartVersion.GetVersion()
			versionVersion, err := semver.Parse(version)

			if err != nil {
				klog.Error(err)
				return false
			}

			filterVersion, err := semver.ParseRange(sub.Spec.PackageFilter.Version)

			if err != nil {
				klog.Error(err)
				return false
			}

			return filterVersion(versionVersion)
		}
	}

	klog.V(4).Info("Version check passed for:", chartVersion)

	return true
}

//DeleteHelmReleaseCRD deletes the HelmRelease CRD
func DeleteHelmReleaseCRD(runtimeClient client.Client, crdx *clientsetx.Clientset) {
	hrlist := &releasev1.HelmReleaseList{}
	err := runtimeClient.List(context.TODO(), hrlist, &client.ListOptions{})

	if err != nil && !errors.IsNotFound(err) {
		klog.Infof("HelmRelease kind is gone. err: %s", err.Error())
		os.Exit(0)
	} else {
		for _, hr := range hrlist.Items {
			klog.V(1).Infof("Found %s", hr.SelfLink)
			// remove all finalizers
			hr = *hr.DeepCopy()
			hr.SetFinalizers([]string{})
			err = runtimeClient.Update(context.TODO(), &hr)
			if err != nil {
				klog.Warning(err)
			}
		}
		// now get rid of the crd
		err = crdx.ApiextensionsV1().CustomResourceDefinitions().Delete("helmreleases.apps.open-cluster-management.io", &v1.DeleteOptions{})
		if err != nil {
			klog.Infof("Deleting helmrelease CRD failed. err: %s", err.Error())
		} else {
			klog.Info("helmrelease CRD removed")
		}
	}
}
