//
//  StoreAPITests.swift
//  SwaggerClient
//
//  Created by Tony Wang on 7/31/16.
//  Copyright © 2016 Swagger. All rights reserved.
//

import PetstoreClient
import RxSwift
import XCTest
@testable import SwaggerClient

class StoreAPITests: XCTestCase {

    let isoDateFormat = "yyyy-MM-dd'T'HH:mm:ss.SSSZZZZZ"

    let testTimeout = 10.0
    let disposeBag = DisposeBag()

    func test1PlaceOrder() {
        let order = Order()
        let shipDate = NSDate()
        order.id = 1000
        order.petId = 1000
        order.complete = false
        order.quantity = 10
        order.shipDate = shipDate
        // use explicit naming to reference the enum so that we test we don't regress on enum naming
        order.status = Order.Status.Placed
        let expectation = self.expectationWithDescription("testPlaceOrder")
        StoreAPI.placeOrder(body: order).subscribe(onNext: { order in
            XCTAssert(order.id == 1000, "invalid id")
            XCTAssert(order.quantity == 10, "invalid quantity")
            XCTAssert(order.status == .Placed, "invalid status")
            XCTAssert(order.shipDate!.isEqual(shipDate, format: self.isoDateFormat),
                "Date should be idempotent")

            expectation.fulfill()
        }, onError: { _ in
            XCTFail("error placing order")
        }, onCompleted: nil, onDisposed: nil).addDisposableTo(disposeBag)
        self.waitForExpectationsWithTimeout(testTimeout, handler: nil)
    }

    func test2GetOrder() {
        let expectation = self.expectationWithDescription("testGetOrder")
        StoreAPI.getOrderById(orderId: "1000").subscribe(onNext: { order -> Void in
            XCTAssert(order.id == 1000, "invalid id")
            XCTAssert(order.quantity == 10, "invalid quantity")
            XCTAssert(order.status == .Placed, "invalid status")
            expectation.fulfill()
            }, onError: { _ in
                XCTFail("error placing order")
            }, onCompleted: nil, onDisposed: nil).addDisposableTo(disposeBag)
        self.waitForExpectationsWithTimeout(testTimeout, handler: nil)
    }

    func test3DeleteOrder() {
        let expectation = self.expectationWithDescription("testDeleteOrder")
        StoreAPI.deleteOrder(orderId: "1000").subscribe(onNext: {
            expectation.fulfill()
            }, onError: { errorType -> Void in
                // The server gives us no data back so alamofire parsing fails - at least
                // verify that is the error we get here
                // Error Domain=com.alamofire.error Code=-6006 "JSON could not be serialized. Input data was nil or zero
                // length." UserInfo={NSLocalizedFailureReason=JSON could not be serialized. Input data was nil or zero
                // length.}
                let error = errorType as NSError
                if error.code == -6006 {
                    expectation.fulfill()
                } else {
                    XCTFail("error deleting order")
                }
            }, onCompleted: nil, onDisposed: nil).addDisposableTo(disposeBag)
        self.waitForExpectationsWithTimeout(testTimeout, handler: nil)
    }

}

private extension NSDate {

    /**
     Returns true if the dates are equal given the format string.

     - parameter date:   The date to compare to.
     - parameter format: The format string to use to compare.

     - returns: true if the dates are equal, given the format string.
     */
    func isEqual(date: NSDate, format: String) -> Bool {
        let fmt = NSDateFormatter()
        fmt.dateFormat = format
        return fmt.stringFromDate(self).isEqual(fmt.stringFromDate(date))
    }

}
