import { getNodeAttributes, uniqueArray } from '../../core/utils';
import { getRole, allowedAttr, validateAttr } from '../../commons/aria';

/**
 * Check if each ARIA attribute on an element is allowed for its semantic role.
 *
 * Allowed ARIA attributes are taken from the `ariaRoles` standards object combining the roles `requiredAttrs` and `allowedAttrs` properties, as well as any global ARIA attributes from the `ariaAttrs` standards object.
 *
 * ##### Data:
 * <table class="props">
 *   <thead>
 *     <tr>
 *       <th>Type</th>
 *       <th>Description</th>
 *     </tr>
 *   </thead>
 *   <tbody>
 *     <tr>
 *       <td><code>String[]</code></td>
 *       <td>List of all unallowed aria attributes and their value</td>
 *     </tr>
 *   </tbody>
 * </table>
 *
 * @memberof checks
 * @return {Boolean} True if each aria attribute is allowed. False otherwise.
 */
function ariaAllowedAttrEvaluate(node, options) {
	const invalid = [];

	const role = getRole(node);
	const attrs = getNodeAttributes(node);
	let allowed = allowedAttr(role);

	// @deprecated: allowed attr options to pass more attrs.
	// configure the standards spec instead
	if (Array.isArray(options[role])) {
		allowed = uniqueArray(options[role].concat(allowed));
	}

	if (role && allowed) {
		for (let i = 0; i < attrs.length; i++) {
			const attr = attrs[i];
			const attrName = attr.name;
			if (validateAttr(attrName) && !allowed.includes(attrName)) {
				invalid.push(attrName + '="' + attr.nodeValue + '"');
			}
		}
	}

	if (invalid.length) {
		this.data(invalid);
		return false;
	}

	return true;
}

export default ariaAllowedAttrEvaluate;
