import { ResizeObserver } from '@juggle/resize-observer';
import useLayoutEffect from '@react-hook/passive-layout-effect';
import useLatest from '@react-hook/latest';
import rafSchd from 'raf-schd';

/**
 * A React hook that fires a callback whenever ResizeObserver detects a change to its size
 *
 * @param target A React ref created by `useRef()` or an HTML element
 * @param callback Invoked with a single `ResizeObserverEntry` any time
 *   the `target` resizes
 */

function useResizeObserver(target, callback) {
  var resizeObserver = getResizeObserver();
  var storedCallback = useLatest(callback);
  useLayoutEffect(() => {
    var didUnsubscribe = false;
    var targetEl = target && 'current' in target ? target.current : target;
    if (!targetEl) return;
    resizeObserver.subscribe(targetEl, (entry, observer) => {
      if (didUnsubscribe) return;
      storedCallback.current(entry, observer);
    });
    return () => {
      didUnsubscribe = true;
      resizeObserver.unsubscribe(targetEl);
    };
  }, [target, resizeObserver, storedCallback]);
  return resizeObserver.observer;
}

function createResizeObserver() {
  var callbacks = new Map();
  var observer = new ResizeObserver(rafSchd((entries, observer) => {
    if (entries.length === 1) {
      var _callbacks$get;

      (_callbacks$get = callbacks.get(entries[0].target)) === null || _callbacks$get === void 0 ? void 0 : _callbacks$get(entries[0], observer);
    } else {
      for (var i = 0; i < entries.length; i++) {
        var _callbacks$get2;

        (_callbacks$get2 = callbacks.get(entries[i].target)) === null || _callbacks$get2 === void 0 ? void 0 : _callbacks$get2(entries[i], observer);
      }
    }
  }));
  return {
    observer,

    subscribe(target, callback) {
      observer.observe(target);
      callbacks.set(target, callback);
    },

    unsubscribe(target) {
      observer.unobserve(target);
      callbacks.delete(target);
    }

  };
}

var _resizeObserver;

var getResizeObserver = () => !_resizeObserver ? _resizeObserver = createResizeObserver() : _resizeObserver;

export default useResizeObserver;
//# sourceMappingURL=index.dev.mjs.map
