'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

function _interopDefault (ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

function _interopNamespace(e) {
    if (e && e.__esModule) { return e; } else {
        var n = {};
        if (e) {
            Object.keys(e).forEach(function (k) {
                var d = Object.getOwnPropertyDescriptor(e, k);
                Object.defineProperty(n, k, d.get ? d : {
                    enumerable: true,
                    get: function () {
                        return e[k];
                    }
                });
            });
        }
        n['default'] = e;
        return n;
    }
}

const graphql = require('graphql');
const utils = require('@graphql-tools/utils');
const validUrl = require('valid-url');
const wrap = require('@graphql-tools/wrap');
const graphqlWs = require('graphql-ws');
const graphqlSse = require('graphql-sse');
const WebSocket = _interopDefault(require('isomorphic-ws'));
const extractFiles = require('extract-files');
const subscriptionsTransportWs = require('subscriptions-transport-ws');
const valueOrPromise = require('value-or-promise');
const graphqlLiveQuery = require('@n1ru4l/graphql-live-query');
const crossUndiciFetch = require('cross-undici-fetch');
const syncFetchImported = _interopDefault(require('sync-fetch'));
const node = require('meros/node');
const browser = require('meros/browser');
const merge = require('dset/merge');

const defaultAsyncFetch = async (input, init) => {
    return crossUndiciFetch.fetch(input, init);
};

const defaultSyncFetch = (input, init) => {
    if (typeof input === 'string') {
        init === null || init === void 0 ? true : delete init.signal;
    }
    else {
        delete input.signal;
    }
    return syncFetchImported(input, init);
};

function isIncomingMessage(body) {
    return body != null && typeof body === 'object' && 'pipe' in body;
}
async function handleMultipartMixedResponse(response) {
    const body = await response.body;
    const contentType = response.headers.get('content-type') || '';
    let asyncIterator;
    if (isIncomingMessage(body)) {
        // Meros/node expects headers as an object map with the content-type prop
        body.headers = {
            'content-type': contentType,
        };
        // And it expects `IncomingMessage` and `node-fetch` returns `body` as `Promise<PassThrough>`
        asyncIterator = (await node.meros(body));
    }
    else {
        // Nothing is needed for regular `Response`.
        asyncIterator = (await browser.meros(response));
    }
    const executionResult = {};
    return utils.mapAsyncIterator(asyncIterator, (part) => {
        if (part.json) {
            const chunk = part.body;
            if (chunk.path) {
                if (chunk.data) {
                    const path = ['data'];
                    merge.dset(executionResult, path.concat(chunk.path), chunk.data);
                }
                if (chunk.errors) {
                    executionResult.errors = (executionResult.errors || []).concat(chunk.errors);
                }
            }
            else {
                if (chunk.data) {
                    executionResult.data = chunk.data;
                }
                if (chunk.errors) {
                    executionResult.errors = chunk.errors;
                }
            }
            return executionResult;
        }
    });
}

/* eslint-disable no-labels */
async function* handleReadable(readable) {
    const decoder = new TextDecoder();
    outer: for await (const chunk of readable) {
        const chunkStr = typeof chunk === 'string' ? chunk : decoder.decode(chunk);
        for (const part of chunkStr.split('\n\n')) {
            if (part) {
                const eventStr = part.split('event: ')[1];
                const dataStr = part.split('data: ')[1];
                if (eventStr === 'complete') {
                    break outer;
                }
                if (dataStr) {
                    const data = JSON.parse(dataStr);
                    yield data.payload || data;
                }
            }
        }
    }
}

// Based on https://github.com/Azure/fetch-event-source/blob/main/src/parse.ts
async function* handleReadableStream(stream) {
    const decoder = new TextDecoder();
    const reader = stream.getReader();
    let buffer;
    let position = 0; // current read position
    let fieldLength = -1; // length of the `field` portion of the line
    let discardTrailingNewline = false;
    try {
        let result;
        let message = {
            data: '',
            event: '',
            id: '',
            retry: undefined,
        };
        while (!(result = await reader.read()).done) {
            const arr = result.value;
            if (buffer === undefined) {
                buffer = arr;
                position = 0;
                fieldLength = -1;
            }
            else {
                // we're still parsing the old line. Append the new bytes into buffer:
                buffer = concat(buffer, arr);
            }
            const bufLength = buffer.length;
            let lineStart = 0; // index where the current line starts
            while (position < bufLength) {
                if (discardTrailingNewline) {
                    if (buffer[position] === 10 /* NewLine */) {
                        lineStart = ++position; // skip to next char
                    }
                    discardTrailingNewline = false;
                }
                // start looking forward till the end of line:
                let lineEnd = -1; // index of the \r or \n char
                for (; position < bufLength && lineEnd === -1; ++position) {
                    switch (buffer[position]) {
                        case 58 /* Colon */: {
                            if (fieldLength === -1) {
                                // first colon in line
                                fieldLength = position - lineStart;
                            }
                            break;
                        }
                        case 13 /* CarriageReturn */: {
                            discardTrailingNewline = true;
                            break;
                        }
                        case 10 /* NewLine */: {
                            lineEnd = position;
                            break;
                        }
                    }
                }
                if (lineEnd === -1) {
                    // We reached the end of the buffer but the line hasn't ended.
                    // Wait for the next arr and then continue parsing:
                    break;
                }
                // we've reached the line end, send it out:
                const line = buffer.subarray(lineStart, lineEnd);
                if (line.length === 0) {
                    // empty line denotes end of message. Trigger the callback and start a new message:
                    if (message.event || message.data) {
                        // NOT a server ping (":\n\n")
                        yield JSON.parse(message.data);
                        message = {
                            data: '',
                            event: '',
                            id: '',
                            retry: undefined,
                        };
                    }
                }
                else if (fieldLength > 0) {
                    // exclude comments and lines with no values
                    // line is of format "<field>:<value>" or "<field>: <value>"
                    // https://html.spec.whatwg.org/multipage/server-sent-events.html#event-stream-interpretation
                    const field = decoder.decode(line.subarray(0, fieldLength));
                    const valueOffset = fieldLength + (line[fieldLength + 1] === 32 /* Space */ ? 2 : 1);
                    const value = decoder.decode(line.subarray(valueOffset));
                    switch (field) {
                        case 'data':
                            // if this message already has data, append the new value to the old.
                            // otherwise, just set to the new value:
                            message.data = message.data ? message.data + '\n' + value : value; // otherwise,
                            break;
                        case 'event':
                            message.event = value;
                            break;
                        case 'id':
                            message.id = value;
                            break;
                        case 'retry': {
                            const retry = parseInt(value, 10);
                            message.retry = retry;
                            break;
                        }
                    }
                }
                lineStart = position; // we're now on the next line
                fieldLength = -1;
            }
            if (lineStart === bufLength) {
                buffer = undefined; // we've finished reading it
            }
            else if (lineStart !== 0) {
                // Create a new view into buffer beginning at lineStart so we don't
                // need to copy over the previous lines when we get the new arr:
                buffer = buffer.subarray(lineStart);
                position -= lineStart;
            }
        }
    }
    finally {
        reader.releaseLock();
    }
}
function concat(a, b) {
    const res = new Uint8Array(a.length + b.length);
    res.set(a);
    res.set(b, a.length);
    return res;
}

async function handleEventStreamResponse(response) {
    // node-fetch returns body as a promise so we need to resolve it
    const body = await response.body;
    if (body) {
        if (utils.isAsyncIterable(body)) {
            return handleReadable(body);
        }
        return handleReadableStream(body);
    }
    throw new Error('Response body is expected to be a readable stream but got; ' + utils.inspect(body));
}

function addCancelToResponseStream(resultStream, controller) {
    return utils.withCancel(resultStream, () => {
        if (!controller.signal.aborted) {
            controller.abort();
        }
    });
}

function isBlob(obj) {
    return typeof obj.arrayBuffer === 'function';
}
function isGraphQLUpload(upload) {
    return typeof upload.createReadStream === 'function';
}
function isPromiseLike(obj) {
    return typeof obj.then === 'function';
}

/* eslint-disable no-case-declarations */
const asyncImport = (moduleName) => new Promise(function (resolve) { resolve(_interopNamespace(require(moduleName))); });
const syncImport = (moduleName) => require(moduleName);
(function (SubscriptionProtocol) {
    SubscriptionProtocol["WS"] = "WS";
    /**
     * Use legacy web socket protocol `graphql-ws` instead of the more current standard `graphql-transport-ws`
     */
    SubscriptionProtocol["LEGACY_WS"] = "LEGACY_WS";
    /**
     * Use SSE for subscription instead of WebSocket
     */
    SubscriptionProtocol["SSE"] = "SSE";
    /**
     * Use `graphql-sse` for subscriptions
     */
    SubscriptionProtocol["GRAPHQL_SSE"] = "GRAPHQL_SSE";
})(exports.SubscriptionProtocol || (exports.SubscriptionProtocol = {}));
const isCompatibleUri = (uri) => {
    if (validUrl.isWebUri(uri)) {
        return true;
    }
    // we just replace the url part, the remaining validation is the same
    const wsUri = uri.replace('wss://', 'http://').replace('ws://', 'http://');
    return !!validUrl.isWebUri(wsUri);
};
/**
 * This loader loads a schema from a URL. The loaded schema is a fully-executable,
 * remote schema since it's created using [@graphql-tools/wrap](/docs/remote-schemas).
 *
 * ```
 * const schema = await loadSchema('http://localhost:3000/graphql', {
 *   loaders: [
 *     new UrlLoader(),
 *   ]
 * });
 * ```
 */
class UrlLoader {
    async canLoad(pointer, options) {
        return this.canLoadSync(pointer, options);
    }
    canLoadSync(pointer, _options) {
        return isCompatibleUri(pointer);
    }
    createFormDataFromVariables({ query, variables, operationName, extensions, }) {
        const vars = Object.assign({}, variables);
        const { clone, files } = extractFiles.extractFiles(vars, 'variables', ((v) => extractFiles.isExtractableFile(v) ||
            (v === null || v === void 0 ? void 0 : v.promise) ||
            utils.isAsyncIterable(v) ||
            (v === null || v === void 0 ? void 0 : v.then) ||
            typeof (v === null || v === void 0 ? void 0 : v.arrayBuffer) === 'function'));
        const map = {};
        const uploads = [];
        let currIndex = 0;
        for (const [file, curr] of files) {
            map[currIndex] = curr;
            uploads[currIndex] = file;
            currIndex++;
        }
        const form = new crossUndiciFetch.FormData();
        form.append('operations', JSON.stringify({
            query,
            variables: clone,
            operationName,
            extensions,
        }));
        form.append('map', JSON.stringify(map));
        function handleUpload(upload, i) {
            const indexStr = i.toString();
            if (upload != null) {
                const filename = upload.filename || upload.name || upload.path || `blob-${indexStr}`;
                if (isPromiseLike(upload)) {
                    return upload.then((resolvedUpload) => handleUpload(resolvedUpload, i));
                    // If Blob
                }
                else if (isBlob(upload)) {
                    return upload.arrayBuffer().then((arrayBuffer) => {
                        form.append(indexStr, new crossUndiciFetch.File([arrayBuffer], filename, { type: upload.type }), filename);
                    });
                }
                else if (isGraphQLUpload(upload)) {
                    const stream = upload.createReadStream();
                    const chunks = [];
                    return Promise.resolve().then(async () => {
                        for await (const chunk of stream) {
                            if (chunk) {
                                chunks.push(...chunk);
                            }
                        }
                        const blobPart = new Uint8Array(chunks);
                        form.append(indexStr, new crossUndiciFetch.File([blobPart], filename, { type: upload.mimetype }), filename);
                    });
                }
                else {
                    form.append(indexStr, new crossUndiciFetch.File([upload], filename), filename);
                }
            }
        }
        return valueOrPromise.ValueOrPromise.all(uploads.map((upload, i) => new valueOrPromise.ValueOrPromise(() => handleUpload(upload, i))))
            .then(() => form)
            .resolve();
    }
    prepareGETUrl({ baseUrl, query, variables, operationName, extensions, }) {
        const HTTP_URL = switchProtocols(baseUrl, {
            wss: 'https',
            ws: 'http',
        });
        const dummyHostname = 'https://dummyhostname.com';
        const validUrl = HTTP_URL.startsWith('http')
            ? HTTP_URL
            : HTTP_URL.startsWith('/')
                ? `${dummyHostname}${HTTP_URL}`
                : `${dummyHostname}/${HTTP_URL}`;
        const urlObj = new URL(validUrl);
        urlObj.searchParams.set('query', query);
        if (variables && Object.keys(variables).length > 0) {
            urlObj.searchParams.set('variables', JSON.stringify(variables));
        }
        if (operationName) {
            urlObj.searchParams.set('operationName', operationName);
        }
        if (extensions) {
            urlObj.searchParams.set('extensions', JSON.stringify(extensions));
        }
        const finalUrl = urlObj.toString().replace(dummyHostname, '');
        return finalUrl;
    }
    buildHTTPExecutor(initialEndpoint, fetch, options) {
        const defaultMethod = this.getDefaultMethodFromOptions(options === null || options === void 0 ? void 0 : options.method, 'POST');
        const HTTP_URL = switchProtocols(initialEndpoint, {
            wss: 'https',
            ws: 'http',
        });
        const executor = (request) => {
            var _a, _b;
            const controller = new crossUndiciFetch.AbortController();
            let method = defaultMethod;
            if (options === null || options === void 0 ? void 0 : options.useGETForQueries) {
                const operationAst = utils.getOperationASTFromRequest(request);
                const operationType = operationAst.operation;
                if (operationType === 'query') {
                    method = 'GET';
                }
            }
            const endpoint = ((_a = request.extensions) === null || _a === void 0 ? void 0 : _a.endpoint) || initialEndpoint;
            const headers = Object.assign({}, options === null || options === void 0 ? void 0 : options.headers, ((_b = request.extensions) === null || _b === void 0 ? void 0 : _b.headers) || {});
            const acceptedProtocols = [`application/json`];
            if (method === 'GET' && (options === null || options === void 0 ? void 0 : options.subscriptionsProtocol) === exports.SubscriptionProtocol.SSE) {
                acceptedProtocols.push('text/event-stream');
            }
            else {
                acceptedProtocols.push('multipart/mixed');
            }
            const accept = acceptedProtocols.join(', ');
            const query = graphql.print(request.document);
            const requestBody = {
                query,
                variables: request.variables,
                operationName: request.operationName,
                extensions: request.extensions,
            };
            let timeoutId;
            if (options === null || options === void 0 ? void 0 : options.timeout) {
                timeoutId = setTimeout(() => {
                    if (!controller.signal.aborted) {
                        controller.abort();
                    }
                }, options.timeout);
            }
            const credentials = (options === null || options === void 0 ? void 0 : options.credentials) || 'same-origin';
            return new valueOrPromise.ValueOrPromise(() => {
                switch (method) {
                    case 'GET':
                        const finalUrl = this.prepareGETUrl({
                            baseUrl: endpoint,
                            ...requestBody,
                        });
                        return fetch(finalUrl, {
                            method: 'GET',
                            credentials,
                            headers: {
                                accept,
                                ...headers,
                            },
                            signal: controller.signal,
                        });
                    case 'POST':
                        if (options === null || options === void 0 ? void 0 : options.multipart) {
                            return new valueOrPromise.ValueOrPromise(() => this.createFormDataFromVariables(requestBody))
                                .then(form => fetch(HTTP_URL, {
                                method: 'POST',
                                credentials,
                                body: form,
                                headers: {
                                    accept,
                                    ...headers,
                                },
                                signal: controller.signal,
                            }))
                                .resolve();
                        }
                        else {
                            return fetch(HTTP_URL, {
                                method: 'POST',
                                credentials,
                                body: JSON.stringify(requestBody),
                                headers: {
                                    accept,
                                    'content-type': 'application/json',
                                    ...headers,
                                },
                                signal: controller.signal,
                            });
                        }
                }
            })
                .then((fetchResult) => {
                if (timeoutId != null) {
                    clearTimeout(timeoutId);
                }
                // Retry should respect HTTP Errors
                if ((options === null || options === void 0 ? void 0 : options.retry) != null && !fetchResult.status.toString().startsWith('2')) {
                    throw new Error(fetchResult.statusText || `HTTP Error: ${fetchResult.status}`);
                }
                const contentType = fetchResult.headers.get('content-type');
                if (contentType === null || contentType === void 0 ? void 0 : contentType.includes('text/event-stream')) {
                    return handleEventStreamResponse(fetchResult).then(resultStream => addCancelToResponseStream(resultStream, controller));
                }
                else if (contentType === null || contentType === void 0 ? void 0 : contentType.includes('multipart/mixed')) {
                    return handleMultipartMixedResponse(fetchResult).then(resultStream => addCancelToResponseStream(resultStream, controller));
                }
                return fetchResult.json();
            })
                .resolve();
        };
        if ((options === null || options === void 0 ? void 0 : options.retry) != null) {
            return function retryExecutor(request) {
                let result;
                let error;
                let attempt = 0;
                function retryAttempt() {
                    attempt++;
                    if (attempt > options.retry) {
                        if (result != null) {
                            return result;
                        }
                        if (error != null) {
                            throw error;
                        }
                        throw new Error('No result');
                    }
                    return new valueOrPromise.ValueOrPromise(() => executor(request))
                        .then(res => {
                        var _a;
                        result = res;
                        if ((_a = result === null || result === void 0 ? void 0 : result.errors) === null || _a === void 0 ? void 0 : _a.length) {
                            return retryAttempt();
                        }
                        return result;
                    })
                        .catch((e) => {
                        error = e;
                        return retryAttempt();
                    })
                        .resolve();
                }
                return retryAttempt();
            };
        }
        return executor;
    }
    buildWSExecutor(subscriptionsEndpoint, webSocketImpl, connectionParams) {
        const WS_URL = switchProtocols(subscriptionsEndpoint, {
            https: 'wss',
            http: 'ws',
        });
        const subscriptionClient = graphqlWs.createClient({
            url: WS_URL,
            webSocketImpl,
            connectionParams,
            lazy: true,
        });
        return async ({ document, variables, operationName, extensions }) => {
            const query = graphql.print(document);
            return utils.observableToAsyncIterable({
                subscribe: observer => {
                    const unsubscribe = subscriptionClient.subscribe({
                        query,
                        variables: variables,
                        operationName,
                        extensions,
                    }, observer);
                    return {
                        unsubscribe,
                    };
                },
            });
        };
    }
    buildWSLegacyExecutor(subscriptionsEndpoint, webSocketImpl, connectionParams) {
        const WS_URL = switchProtocols(subscriptionsEndpoint, {
            https: 'wss',
            http: 'ws',
        });
        const subscriptionClient = new subscriptionsTransportWs.SubscriptionClient(WS_URL, {
            connectionParams,
            lazy: true,
        }, webSocketImpl);
        return async ({ document, variables, operationName }) => {
            return utils.observableToAsyncIterable(subscriptionClient.request({
                query: document,
                variables,
                operationName,
            }));
        };
    }
    buildGraphQLSSEExecutor(endpoint, fetch, options = {}) {
        const { headers } = options;
        const client = graphqlSse.createClient({
            ...options.graphqlSseOptions,
            url: endpoint,
            fetchFn: fetch,
            abortControllerImpl: crossUndiciFetch.AbortController,
            headers,
        });
        return async ({ document, variables, operationName, extensions }) => {
            return utils.observableToAsyncIterable({
                subscribe: observer => {
                    const unsubscribe = client.subscribe({
                        query: document,
                        variables: variables,
                        operationName,
                        extensions,
                    }, observer);
                    return {
                        unsubscribe,
                    };
                },
            });
        };
    }
    getFetch(customFetch, importFn) {
        if (customFetch) {
            if (typeof customFetch === 'string') {
                const [moduleName, fetchFnName] = customFetch.split('#');
                return new valueOrPromise.ValueOrPromise(() => importFn(moduleName))
                    .then(module => (fetchFnName ? module[fetchFnName] : module))
                    .resolve();
            }
            else if (typeof customFetch === 'function') {
                return customFetch;
            }
        }
        if (importFn === asyncImport) {
            return defaultAsyncFetch;
        }
        else {
            return defaultSyncFetch;
        }
    }
    getDefaultMethodFromOptions(method, defaultMethod) {
        if (method) {
            defaultMethod = method;
        }
        return defaultMethod;
    }
    getWebSocketImpl(importFn, options) {
        if (typeof (options === null || options === void 0 ? void 0 : options.webSocketImpl) === 'string') {
            const [moduleName, webSocketImplName] = options.webSocketImpl.split('#');
            return new valueOrPromise.ValueOrPromise(() => importFn(moduleName))
                .then(importedModule => (webSocketImplName ? importedModule[webSocketImplName] : importedModule))
                .resolve();
        }
        else {
            const websocketImpl = (options === null || options === void 0 ? void 0 : options.webSocketImpl) || WebSocket;
            return websocketImpl;
        }
    }
    async buildSubscriptionExecutor(subscriptionsEndpoint, fetch, options) {
        if ((options === null || options === void 0 ? void 0 : options.subscriptionsProtocol) === exports.SubscriptionProtocol.SSE) {
            return this.buildHTTPExecutor(subscriptionsEndpoint, fetch, {
                ...options,
                method: 'GET',
            });
        }
        else if ((options === null || options === void 0 ? void 0 : options.subscriptionsProtocol) === exports.SubscriptionProtocol.GRAPHQL_SSE) {
            if (!(options === null || options === void 0 ? void 0 : options.subscriptionsEndpoint)) {
                // when no custom subscriptions endpoint is specified,
                // graphql-sse is recommended to be used on `/graphql/stream`
                subscriptionsEndpoint += '/stream';
            }
            return this.buildGraphQLSSEExecutor(subscriptionsEndpoint, fetch, options);
        }
        else {
            const webSocketImpl = await this.getWebSocketImpl(asyncImport, options);
            const connectionParams = () => ({ headers: options === null || options === void 0 ? void 0 : options.headers });
            if ((options === null || options === void 0 ? void 0 : options.subscriptionsProtocol) === exports.SubscriptionProtocol.LEGACY_WS) {
                return this.buildWSLegacyExecutor(subscriptionsEndpoint, webSocketImpl, connectionParams);
            }
            else {
                return this.buildWSExecutor(subscriptionsEndpoint, webSocketImpl, connectionParams);
            }
        }
    }
    async getExecutorAsync(endpoint, options) {
        const fetch = await this.getFetch(options === null || options === void 0 ? void 0 : options.customFetch, asyncImport);
        const httpExecutor = this.buildHTTPExecutor(endpoint, fetch, options);
        const subscriptionsEndpoint = (options === null || options === void 0 ? void 0 : options.subscriptionsEndpoint) || endpoint;
        const subscriptionExecutor = await this.buildSubscriptionExecutor(subscriptionsEndpoint, fetch, options);
        return request => {
            const operationAst = utils.getOperationASTFromRequest(request);
            if (operationAst.operation === 'subscription' ||
                graphqlLiveQuery.isLiveQueryOperationDefinitionNode(operationAst, request.variables)) {
                return subscriptionExecutor(request);
            }
            return httpExecutor(request);
        };
    }
    getExecutorSync(endpoint, options) {
        const fetch = this.getFetch(options === null || options === void 0 ? void 0 : options.customFetch, syncImport);
        const executor = this.buildHTTPExecutor(endpoint, fetch, options);
        return executor;
    }
    handleSDL(pointer, fetch, options) {
        const defaultMethod = this.getDefaultMethodFromOptions(options === null || options === void 0 ? void 0 : options.method, 'GET');
        return new valueOrPromise.ValueOrPromise(() => fetch(pointer, {
            method: defaultMethod,
            headers: options.headers,
        }))
            .then(response => response.text())
            .then(schemaString => utils.parseGraphQLSDL(pointer, schemaString, options))
            .resolve();
    }
    async load(pointer, options) {
        if (!(await this.canLoad(pointer, options))) {
            return [];
        }
        let source = {
            location: pointer,
        };
        const fetch = await this.getFetch(options === null || options === void 0 ? void 0 : options.customFetch, asyncImport);
        let executor = await this.getExecutorAsync(pointer, options);
        if ((options === null || options === void 0 ? void 0 : options.handleAsSDL) || pointer.endsWith('.graphql') || pointer.endsWith('.graphqls')) {
            source = await this.handleSDL(pointer, fetch, options);
            if (!source.schema && !source.document && !source.rawSDL) {
                throw new Error(`Invalid SDL response`);
            }
            source.schema =
                source.schema ||
                    (source.document
                        ? graphql.buildASTSchema(source.document, options)
                        : source.rawSDL
                            ? graphql.buildSchema(source.rawSDL, options)
                            : undefined);
        }
        else {
            source.schema = await wrap.introspectSchema(executor, {}, options);
        }
        if (!source.schema) {
            throw new Error(`Invalid introspected schema`);
        }
        if (options === null || options === void 0 ? void 0 : options.endpoint) {
            executor = await this.getExecutorAsync(options.endpoint, options);
        }
        source.schema = wrap.wrapSchema({
            schema: source.schema,
            executor,
        });
        return [source];
    }
    loadSync(pointer, options) {
        if (!this.canLoadSync(pointer, options)) {
            return [];
        }
        let source = {
            location: pointer,
        };
        const fetch = this.getFetch(options === null || options === void 0 ? void 0 : options.customFetch, syncImport);
        let executor = this.getExecutorSync(pointer, options);
        if ((options === null || options === void 0 ? void 0 : options.handleAsSDL) || pointer.endsWith('.graphql') || pointer.endsWith('.graphqls')) {
            source = this.handleSDL(pointer, fetch, options);
            if (!source.schema && !source.document && !source.rawSDL) {
                throw new Error(`Invalid SDL response`);
            }
            source.schema =
                source.schema ||
                    (source.document
                        ? graphql.buildASTSchema(source.document, options)
                        : source.rawSDL
                            ? graphql.buildSchema(source.rawSDL, options)
                            : undefined);
        }
        else {
            source.schema = wrap.introspectSchema(executor, {}, options);
        }
        if (!source.schema) {
            throw new Error(`Invalid introspected schema`);
        }
        if (options === null || options === void 0 ? void 0 : options.endpoint) {
            executor = this.getExecutorSync(options.endpoint, options);
        }
        source.schema = wrap.wrapSchema({
            schema: source.schema,
            executor,
        });
        return [source];
    }
}
function switchProtocols(pointer, protocolMap) {
    return Object.entries(protocolMap).reduce((prev, [source, target]) => prev.replace(`${source}://`, `${target}://`).replace(`${source}:\\`, `${target}:\\`), pointer);
}

exports.UrlLoader = UrlLoader;
