"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.load = exports.resolve = exports.resolveSync = void 0;
const fs_1 = require("fs");
const globby_1 = __importDefault(require("globby"));
const path_1 = require("path");
const gather_deps_1 = require("./gather-deps");
const file_1 = require("./file");
const enhanced_resolve_1 = require("enhanced-resolve");
const readFile = fs_1.promises.readFile;
const lstat = fs_1.promises.lstat;
const fileSystem = new enhanced_resolve_1.CachedInputFileSystem(new enhanced_resolve_1.NodeJsInputFileSystem(), 4000), resolver = enhanced_resolve_1.ResolverFactory.createResolver({
    extensions: ['.js', '.json', '.node'],
    //symlinks: false,
    fileSystem: new enhanced_resolve_1.CachedInputFileSystem(new enhanced_resolve_1.NodeJsInputFileSystem(), 4000),
}), syncResolver = enhanced_resolve_1.ResolverFactory.createResolver({
    extensions: ['.js', '.json', '.node'],
    useSyncFileSystemCalls: true,
    //symlinks: false,
    fileSystem,
}), defaultOptions = { loadContent: true, expand: 'none', isEntry: false };
function resolveSync(from, request) {
    const result = {
        absPath: '',
        pkgPath: '',
        pkg: null,
        warning: '',
    };
    syncResolver.resolve({}, from, request, {}, (err, path, data) => {
        if (err) {
            result.warning = err.message;
            return;
        }
        result.absPath = path;
        result.pkgPath = data.descriptionFilePath;
        result.pkg = data.descriptionFileData;
        return;
    });
    return result;
}
exports.resolveSync = resolveSync;
function resolve(from, request) {
    const result = {
        absPath: '',
        pkgPath: '',
        pkg: null,
        warning: '',
    };
    return new Promise((resolve) => {
        resolver.resolve({}, from, request, {}, (err, path, data) => {
            if (err) {
                result.warning = err.message;
                return resolve(result);
            }
            result.absPath = path;
            result.pkgPath = data.descriptionFilePath;
            result.pkg = data.descriptionFileData;
            resolve(result);
        });
    });
}
exports.resolve = resolve;
async function expand(file, fileDir, baseDir, globs) {
    const files = await globby_1.default(globs, { cwd: baseDir, unique: true, dot: true });
    files
        .map((dep) => file_1.ensureDottedRelative(fileDir, path_1.join(baseDir, dep)))
        .filter((relDep) => file.absPath !== path_1.join(baseDir, relDep))
        .forEach((relDep) => {
        file.deps[relDep] = file.deps[relDep] || null;
    });
    const currentDeps = Object.keys(file.deps);
    file.package &&
        file.package.dependencies &&
        Object.keys(file.package.dependencies || {}).forEach((dependency) => {
            if (!currentDeps.some((curDep) => curDep.startsWith(dependency))) {
                file.deps[dependency] = file.deps[dependency] || null;
            }
        });
}
async function load(workingDirectory, request, options = defaultOptions) {
    var _a;
    const { absPath, pkg, pkgPath, warning } = await resolve(workingDirectory, request);
    if (!absPath) {
        return { warning: warning };
    }
    const file = file_1.createFile(absPath), isJs = absPath.endsWith('.js') || absPath.endsWith('.mjs') || options.isEntry;
    file.absPath = absPath;
    const stats = await lstat(absPath);
    if (stats.isSymbolicLink()) {
        file.symlink = true;
    }
    if (isJs || absPath.endsWith('json')) {
        file.contents = await readFile(absPath, 'utf-8');
    }
    if (isJs) {
        try {
            const parseResult = gather_deps_1.gatherDependencies(file.contents, absPath.endsWith('.mjs'));
            Object.assign(file.deps, parseResult.deps);
            file.variableImports = parseResult.variable;
        }
        catch (e) {
            return { warning: `Error parsing file: "${file.absPath}"\n${e.stack}` };
        }
    }
    const fileDir = path_1.dirname(file.absPath), expandVariable = Boolean(options.expand === 'variable' && file.variableImports);
    if (file_1.isNodeModule(request) && pkg && pkgPath) {
        file.package = pkg;
        file.deps[file_1.ensureDottedRelative(fileDir, pkgPath)] = null;
        const pkgDir = (file.moduleRoot = path_1.dirname(pkgPath)), expandAll = options.expand === 'all';
        if (expandVariable || expandAll) {
            await expand(file, fileDir, pkgDir, file_1.nodeModuleGlobs(file));
            file.contextExpanded = true;
        }
        if (file_1.extraGlobs(file).length) {
            await expand(file, fileDir, pkgDir, file_1.extraGlobs(file));
        }
    }
    else if (expandVariable && ((_a = options.context) === null || _a === void 0 ? void 0 : _a.moduleRoot) && !options.context.expanded) {
        await expand(file, fileDir, options.context.moduleRoot, file_1.nodeModuleGlobs({ package: { files: options.context.globs } }));
        file.contextExpanded = true;
    }
    if (!options.loadContent) {
        file.contents = null;
    }
    return file;
}
exports.load = load;
//# sourceMappingURL=data:application/json;base64,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