/**********************************************************************
 * Copyright (c) 2019-2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 ***********************************************************************/

import { inject, injectable } from 'inversify';

import { ActivityTrackerService } from '../common/activity-tracker-protocol';
import { TelemetryService } from '@eclipse-che/theia-remote-api/lib/common/telemetry-service';
import { WorkspaceService } from '@eclipse-che/theia-remote-api/lib/common/workspace-service';

/**
 * Server side part of Theia activity tracker.
 * Receives activity updates from clients and sends reset inactivity requests to Che workspace master.
 * To avoid duplicate requests from different frontend clients may send requests periodically. This mean
 * that, in the worst case, it might keep user's workspace alive for the period longer.
 * Che master API url for resetting inactive timeout: che-host[:port]/api/activity/<workspace-ID>
 */
@injectable()
export class ActivityTrackerServiceImpl implements ActivityTrackerService {
  // Time before sending next request. If a few requests from frontend(s) are received during this period,
  // only one request to workspace master will be sent.
  private static REQUEST_PERIOD_MS = 1 * 60 * 1000;
  // Time before resending request to workspace master if a network error occurs.
  private static RETRY_REQUEST_PERIOD_MS = 5 * 1000;
  // Number of retries before give up if a network error occurs.
  private static RETRY_COUNT = 5;

  // Indicates state of the timer. If true timer is running.
  private isTimerRunning: boolean;
  // Flag which is used to check if new requests were received during timer awaiting.
  private isNewRequest: boolean;

  @inject(WorkspaceService)
  protected workspaceService: WorkspaceService;

  @inject(TelemetryService)
  protected telemetryService: TelemetryService;

  constructor() {
    this.isTimerRunning = false;
    this.isNewRequest = false;
  }

  /**
   * Invoked each time when a client sends an activity request.
   */
  resetTimeout(): void {
    if (this.isTimerRunning) {
      this.isNewRequest = true;
      return;
    }

    this.sendRequestAndSetTimer();
  }

  private sendRequestAndSetTimer(): void {
    this.sendRequest();
    this.isNewRequest = false;

    setTimeout(() => this.checkNewRequestsTimerCallback(), ActivityTrackerServiceImpl.REQUEST_PERIOD_MS);
    this.isTimerRunning = true;
  }

  private checkNewRequestsTimerCallback(): void {
    this.isTimerRunning = false;

    if (this.isNewRequest) {
      this.sendRequestAndSetTimer();
    }
  }

  private sendRequest(attemptsLeft: number = ActivityTrackerServiceImpl.RETRY_COUNT): void {
    this.telemetryService.submitTelemetryActivity();
    try {
      this.workspaceService.updateWorkspaceActivity();
    } catch (error) {
      if (attemptsLeft > 0) {
        setTimeout(() => this.sendRequest(), ActivityTrackerServiceImpl.RETRY_REQUEST_PERIOD_MS, --attemptsLeft);
      } else {
        console.error('Activity tracker: Failed to ping workspace master: ', error.message);
      }
    }
  }
}
