/**********************************************************************
 * Copyright (c) 2019-2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 ***********************************************************************/

import 'reflect-metadata';

import * as che from '@eclipse-che/plugin';
import * as theia from '@theia/plugin';

import { CHE_TASK_SCHEMA } from './schema/che-task-schema';
import { CHE_TASK_TYPE } from './task/task-protocol';
import { CheTaskEventsHandler } from './preview/task-events-handler';
import { CheTaskRunner } from './task/che-task-runner';
import { ExportConfigurationsManager } from './export/export-configs-manager';
import { PreviewUrlVariableResolver } from './variable/preview-url-variable-resolver';
import { ProjectPathVariableResolver } from './variable/project-path-variable-resolver';
import { ServerVariableResolver } from './variable/server-variable-resolver';
import { TaskStatusHandler } from './task/task-status';
import { TasksPreviewManager } from './preview/tasks-preview-manager';
import { container } from './che-task-backend-module';

let pluginContext: theia.PluginContext;
let outputChannel: theia.OutputChannel | undefined;

export async function start(context: theia.PluginContext): Promise<void> {
  pluginContext = context;

  const сheTaskEventsHandler = container.get<CheTaskEventsHandler>(CheTaskEventsHandler);
  сheTaskEventsHandler.init();

  const tasksPreviewManager = container.get<TasksPreviewManager>(TasksPreviewManager);
  tasksPreviewManager.init();

  const serverVariableResolver = container.get<ServerVariableResolver>(ServerVariableResolver);
  serverVariableResolver.registerVariables();

  const previewUrlVariableResolver = container.get<PreviewUrlVariableResolver>(PreviewUrlVariableResolver);
  previewUrlVariableResolver.registerVariables();

  const projectPathVariableResolver = container.get<ProjectPathVariableResolver>(ProjectPathVariableResolver);
  projectPathVariableResolver.registerVariables();

  const cheTaskRunner = container.get<CheTaskRunner>(CheTaskRunner);
  const taskRunnerSubscription = await che.task.registerTaskRunner(CHE_TASK_TYPE, cheTaskRunner);
  getSubscriptions().push(taskRunnerSubscription);

  await che.task.addTaskSubschema(CHE_TASK_SCHEMA);

  const exportConfigurationsManager = container.get<ExportConfigurationsManager>(ExportConfigurationsManager);
  exportConfigurationsManager.init();

  const taskStatusHandler = container.get<TaskStatusHandler>(TaskStatusHandler);
  taskStatusHandler.init();
}

export function stop(): void {}

export function getContext(): theia.PluginContext {
  return pluginContext;
}

// eslint-disable-next-line @typescript-eslint/no-explicit-any
export function getSubscriptions(): { dispose(): any }[] {
  return pluginContext.subscriptions;
}

export function getOutputChannel(): theia.OutputChannel {
  if (outputChannel) {
    return outputChannel;
  }

  outputChannel = theia.window.createOutputChannel('task-plugin-log');
  return outputChannel;
}
