/********************************************************************************
 * Copyright (C) 2020 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
import URI from '@theia/core/lib/common/uri';
import { CancellationToken } from '@theia/core/lib/common/cancellation';
import { Disposable } from '@theia/core/lib/common/disposable';
import { WaitUntilEvent } from '@theia/core/lib/common/event';
import { ContributionProvider } from '@theia/core/lib/common/contribution-provider';
import { FileChangesEvent, FileOperation, FileOperationError, FileOperationEvent, FileSystemProviderCapabilities, CreateFileOptions, FileContent, FileStat, FileStatWithMetadata, FileStreamContent, FileSystemProvider, ReadFileOptions, ResolveFileOptions, ResolveMetadataFileOptions, Stat, WatchOptions, WriteFileOptions, ResolveFileResult, ResolveFileResultWithMetadata, MoveFileOptions, CopyFileOptions, BaseStatWithMetadata, FileDeleteOptions, FileOperationOptions } from '../common/files';
import { BinaryBuffer, BinaryBufferReadable, BinaryBufferReadableStream } from '@theia/core/lib/common/buffer';
import { ReadableStream, Readable } from '@theia/core/lib/common/stream';
import { LabelProvider } from '@theia/core/lib/browser/label-provider';
import { FileSystemPreferences } from './filesystem-preferences';
import { ProgressService } from '@theia/core/lib/common/progress-service';
import type { TextDocumentContentChangeEvent } from '@theia/core/shared/vscode-languageserver-protocol';
import { EncodingRegistry } from '@theia/core/lib/browser/encoding-registry';
import { EncodingService, ResourceEncoding } from '@theia/core/lib/common/encoding-service';
import { FileSystemWatcherErrorHandler } from './filesystem-watcher-error-handler';
export interface FileOperationParticipant {
    /**
     * Participate in a file operation of a working copy. Allows to
     * change the working copy before it is being saved to disk.
     */
    participate(target: URI, source: URI | undefined, operation: FileOperation, timeout: number, token: CancellationToken): Promise<void>;
}
export interface ReadEncodingOptions {
    /**
     * The optional encoding parameter allows to specify the desired encoding when resolving
     * the contents of the file.
     */
    encoding?: string;
    /**
     * The optional guessEncoding parameter allows to guess encoding from content of the file.
     */
    autoGuessEncoding?: boolean;
}
export interface WriteEncodingOptions {
    /**
     * The encoding to use when updating a file.
     */
    encoding?: string;
    /**
     * If set to true, will enforce the selected encoding and not perform any detection using BOMs.
     */
    overwriteEncoding?: boolean;
}
export interface ReadTextFileOptions extends ReadEncodingOptions, ReadFileOptions {
    /**
     * The optional acceptTextOnly parameter allows to fail this request early if the file
     * contents are not textual.
     */
    acceptTextOnly?: boolean;
}
interface BaseTextFileContent extends BaseStatWithMetadata {
    /**
     * The encoding of the content if known.
     */
    encoding: string;
}
export interface TextFileContent extends BaseTextFileContent {
    /**
     * The content of a text file.
     */
    value: string;
}
export interface TextFileStreamContent extends BaseTextFileContent {
    /**
     * The line grouped content of a text file.
     */
    value: ReadableStream<string>;
}
export interface CreateTextFileOptions extends WriteEncodingOptions, CreateFileOptions {
}
export interface WriteTextFileOptions extends WriteEncodingOptions, WriteFileOptions {
}
export interface UpdateTextFileOptions extends WriteEncodingOptions, WriteFileOptions {
    readEncoding: string;
}
export interface UserFileOperationEvent extends WaitUntilEvent {
    /**
     * An identifier to correlate the operation through the
     * different event types (before, after, error).
     */
    readonly correlationId: number;
    /**
     * The file operation that is taking place.
     */
    readonly operation: FileOperation;
    /**
     * The resource the event is about.
     */
    readonly target: URI;
    /**
     * A property that is defined for move operations.
     */
    readonly source?: URI;
}
export declare const FileServiceContribution: unique symbol;
/**
 * A {@link FileServiceContribution} can be used to add custom {@link FileSystemProvider}s.
 * For this, the contribution has to listen to the {@link FileSystemProviderActivationEvent} and register
 * the custom {@link FileSystemProvider}s according to the scheme when this event is fired.
 *
 * ### Example usage
 * ```ts
 * export class MyFileServiceContribution implements FileServiceContribution {
 *     registerFileSystemProviders(service: FileService): void {
 *         service.onWillActivateFileSystemProvider(event => {
 *             if (event.scheme === 'mySyncProviderScheme') {
 *                 service.registerProvider('mySyncProviderScheme', this.mySyncProvider);
 *             }
 *             if (event.scheme === 'myAsyncProviderScheme') {
 *                 event.waitUntil((async () => {
 *                     const myAsyncProvider = await this.createAsyncProvider();
 *                     service.registerProvider('myAsyncProviderScheme', myAsyncProvider);
 *                 })());
 *             }
 *         });
 *
 *     }
 *```
 */
export interface FileServiceContribution {
    /**
     * Register custom file system providers for the given {@link FileService}.
     * @param service The file service for which the providers should be registered.
     */
    registerFileSystemProviders(service: FileService): void;
}
/**
 * Represents the `FileSystemProviderRegistration` event.
 * This event is fired by the {@link FileService} if a {@link FileSystemProvider} is
 * registered to or unregistered from the service.
 */
export interface FileSystemProviderRegistrationEvent {
    /** `True` if a new provider has been registered, `false` if a provider has been unregistered. */
    added: boolean;
    /** The (uri) scheme for which the provider was (previously) registered */
    scheme: string;
    /** The affected file system provider for which this event was fired. */
    provider?: FileSystemProvider;
}
/**
 * Represents the `FileSystemProviderCapabilitiesChange` event.
 * This event is fired by the {@link FileService} if the capabilities of one of its managed
 * {@link FileSystemProvider}s have changed.
 */
export interface FileSystemProviderCapabilitiesChangeEvent {
    /** The affected file system provider for which this event was fired. */
    provider: FileSystemProvider;
    /** The (uri) scheme for which the provider is registered */
    scheme: string;
}
/**
 * Represents the `FileSystemProviderActivation` event.
 * This event is fired by the {@link FileService} if it wants to activate the
 * {@link FileSystemProvider} for a specific scheme.
 */
export interface FileSystemProviderActivationEvent extends WaitUntilEvent {
    /** The (uri) scheme for which the provider should be activated */
    scheme: string;
}
export declare const enum TextFileOperationResult {
    FILE_IS_BINARY = 0
}
export declare class TextFileOperationError extends FileOperationError {
    textFileOperationResult: TextFileOperationResult;
    options?: (ReadTextFileOptions & WriteTextFileOptions) | undefined;
    constructor(message: string, textFileOperationResult: TextFileOperationResult, options?: (ReadTextFileOptions & WriteTextFileOptions) | undefined);
}
/**
 * The {@link FileService} is the common facade responsible for all interactions with file systems.
 * It manages all registered {@link FileSystemProvider}s and
 *  forwards calls to the responsible {@link FileSystemProvider}, determined by the scheme.
 * For additional documentation regarding the provided functions see also {@link FileSystemProvider}.
 */
export declare class FileService {
    private readonly BUFFER_SIZE;
    protected readonly labelProvider: LabelProvider;
    protected readonly preferences: FileSystemPreferences;
    protected readonly progressService: ProgressService;
    protected readonly encodingRegistry: EncodingRegistry;
    protected readonly encodingService: EncodingService;
    protected readonly contributions: ContributionProvider<FileServiceContribution>;
    protected readonly watcherErrorHandler: FileSystemWatcherErrorHandler;
    protected init(): void;
    private correlationIds;
    private readonly onWillRunUserOperationEmitter;
    /**
     * An event that is emitted when file operation is being performed.
     * This event is triggered by user gestures.
     */
    readonly onWillRunUserOperation: import("@theia/core/lib/common/event").Event<UserFileOperationEvent>;
    private readonly onDidFailUserOperationEmitter;
    /**
     * An event that is emitted when file operation is failed.
     * This event is triggered by user gestures.
     */
    readonly onDidFailUserOperation: import("@theia/core/lib/common/event").Event<UserFileOperationEvent>;
    private readonly onDidRunUserOperationEmitter;
    /**
     * An event that is emitted when file operation is finished.
     * This event is triggered by user gestures.
     */
    readonly onDidRunUserOperation: import("@theia/core/lib/common/event").Event<UserFileOperationEvent>;
    private onDidChangeFileSystemProviderRegistrationsEmitter;
    readonly onDidChangeFileSystemProviderRegistrations: import("@theia/core/lib/common/event").Event<FileSystemProviderRegistrationEvent>;
    private onWillActivateFileSystemProviderEmitter;
    /**
     * See `FileServiceContribution.registerProviders`.
     */
    readonly onWillActivateFileSystemProvider: import("@theia/core/lib/common/event").Event<FileSystemProviderActivationEvent>;
    private onDidChangeFileSystemProviderCapabilitiesEmitter;
    readonly onDidChangeFileSystemProviderCapabilities: import("@theia/core/lib/common/event").Event<FileSystemProviderCapabilitiesChangeEvent>;
    private readonly providers;
    private readonly activations;
    /**
     * Registers a new {@link FileSystemProvider} for the given scheme.
     * @param scheme The (uri) scheme for which the provider should be registered.
     * @param provider The file system provider that should be registered.
     *
     * @returns A `Disposable` that can be invoked to unregister the given provider.
     */
    registerProvider(scheme: string, provider: FileSystemProvider): Disposable;
    /**
     * Try to activate the registered provider for the given scheme
     * @param scheme  The uri scheme for which the responsible provider should be activated.
     *
     * @returns A promise of the activated file system provider. Only resolves if a provider is available for this scheme, gets rejected otherwise.
     */
    activateProvider(scheme: string): Promise<FileSystemProvider>;
    /**
     * Tests if the service (i.e. any of its registered {@link FileSystemProvider}s) can handle the given resource.
     * @param resource `URI` of the resource to test.
     *
     * @returns `true` if the resource can be handled, `false` otherwise.
     */
    canHandleResource(resource: URI): boolean;
    /**
     * Tests if the service (i.e the {@link FileSystemProvider} registered for the given uri scheme) provides the given capability.
     * @param resource `URI` of the resource to test.
     * @param capability The required capability.
     *
     * @returns `true` if the resource can be handled and the required capability can be provided.
     */
    hasCapability(resource: URI, capability: FileSystemProviderCapabilities): boolean;
    protected withProvider(resource: URI): Promise<FileSystemProvider>;
    private withReadProvider;
    private withWriteProvider;
    private onDidRunOperationEmitter;
    /**
     * An event that is emitted when operation is finished.
     * This event is triggered by user gestures and programmatically.
     */
    readonly onDidRunOperation: import("@theia/core/lib/common/event").Event<FileOperationEvent>;
    /**
     * Try to resolve file information and metadata for the given resource.
     * @param resource `URI` of the resource that should be resolved.
     * @param options  Options to customize the resolvement process.
     *
     * @return A promise that resolves if the resource could be successfully resolved.
     */
    resolve(resource: URI, options: ResolveMetadataFileOptions): Promise<FileStatWithMetadata>;
    resolve(resource: URI, options?: ResolveFileOptions | undefined): Promise<FileStat>;
    private doResolveFile;
    private toFileStat;
    /**
     * Try to resolve file information and metadata for all given resource.
     * @param toResolve An array of all the resources (and corresponding resolvement options) that should be resolved.
     *
     * @returns A promise of all resolved resources. The promise is not rejected if any of the given resources cannot be resolved.
     * Instead this is reflected with the `success` flag of the corresponding {@link ResolveFileResult}.
     */
    resolveAll(toResolve: {
        resource: URI;
        options?: ResolveFileOptions;
    }[]): Promise<ResolveFileResult[]>;
    resolveAll(toResolve: {
        resource: URI;
        options: ResolveMetadataFileOptions;
    }[]): Promise<ResolveFileResultWithMetadata[]>;
    /**
     * Tests if the given resource exists in the filesystem.
     * @param resource `URI` of the resource which should be tested.
     * @throws Will throw an error if no {@link FileSystemProvider} is registered for the given resource.
     *
     * @returns A promise that resolves to `true` if the resource exists.
     */
    exists(resource: URI): Promise<boolean>;
    /**
     * Tests a user's permissions for the given resource.
     */
    access(resource: URI, mode?: number): Promise<boolean>;
    /**
     * Resolves the fs path of the given URI.
     *
     * USE WITH CAUTION: You should always prefer URIs to paths if possible, as they are
     * portable and platform independent. Paths should only be used in cases you directly
     * interact with the OS, e.g. when running a command on the shell.
     *
     * If you need to display human readable simple or long names then use `LabelProvider` instead.
     * @param resource `URI` of the resource that should be resolved.
     * @throws Will throw an error if no {@link FileSystemProvider} is registered for the given resource.
     *
     * @returns A promise of the resolved fs path.
     */
    fsPath(resource: URI): Promise<string>;
    create(resource: URI, value?: string | Readable<string>, options?: CreateTextFileOptions): Promise<FileStatWithMetadata>;
    protected doCreate(resource: URI, value?: string | Readable<string>, options?: CreateTextFileOptions): Promise<FileStatWithMetadata>;
    write(resource: URI, value: string | Readable<string>, options?: WriteTextFileOptions): Promise<FileStatWithMetadata & {
        encoding: string;
    }>;
    read(resource: URI, options?: ReadTextFileOptions): Promise<TextFileContent>;
    readStream(resource: URI, options?: ReadTextFileOptions): Promise<TextFileStreamContent>;
    private doRead;
    protected resolveReadOptions(options?: ReadTextFileOptions): ReadTextFileOptions;
    update(resource: URI, changes: TextDocumentContentChangeEvent[], options: UpdateTextFileOptions): Promise<FileStatWithMetadata & {
        encoding: string;
    }>;
    createFile(resource: URI, bufferOrReadableOrStream?: BinaryBuffer | BinaryBufferReadable | BinaryBufferReadableStream, options?: CreateFileOptions): Promise<FileStatWithMetadata>;
    writeFile(resource: URI, bufferOrReadableOrStream: BinaryBuffer | BinaryBufferReadable | BinaryBufferReadableStream, options?: WriteFileOptions): Promise<FileStatWithMetadata>;
    private validateWriteFile;
    /**
     * Dirty write prevention: if the file on disk has been changed and does not match our expected
     * mtime and etag, we bail out to prevent dirty writing.
     *
     * First, we check for a mtime that is in the future before we do more checks. The assumption is
     * that only the mtime is an indicator for a file that has changed on disk.
     *
     * Second, if the mtime has advanced, we compare the size of the file on disk with our previous
     * one using the etag() function. Relying only on the mtime check has proven to produce false
     * positives due to file system weirdness (especially around remote file systems). As such, the
     * check for size is a weaker check because it can return a false negative if the file has changed
     * but to the same length. This is a compromise we take to avoid having to produce checksums of
     * the file content for comparison which would be much slower to compute.
     */
    protected modifiedSince(stat: Stat, options?: WriteFileOptions): boolean;
    readFile(resource: URI, options?: ReadFileOptions): Promise<FileContent>;
    readFileStream(resource: URI, options?: ReadFileOptions): Promise<FileStreamContent>;
    private doReadAsFileStream;
    private readFileStreamed;
    private readFileBuffered;
    protected rethrowAsFileOperationError(message: string, resource: URI, error: Error, options?: ReadFileOptions & WriteFileOptions & CreateFileOptions): never;
    protected asFileOperationError(message: string, resource: URI, error: Error, options?: ReadFileOptions & WriteFileOptions & CreateFileOptions): FileOperationError;
    private readFileUnbuffered;
    private validateReadFile;
    private validateReadFileLimits;
    move(source: URI, target: URI, options?: MoveFileOptions): Promise<FileStatWithMetadata>;
    protected doMove(source: URI, target: URI, overwrite?: boolean): Promise<FileStatWithMetadata>;
    copy(source: URI, target: URI, options?: CopyFileOptions): Promise<FileStatWithMetadata>;
    protected doCopy(source: URI, target: URI, overwrite?: boolean): Promise<FileStatWithMetadata>;
    private doMoveCopy;
    private doCopyFile;
    private doCopyFolder;
    private doValidateMoveCopy;
    createFolder(resource: URI): Promise<FileStatWithMetadata>;
    private mkdirp;
    delete(resource: URI, options?: FileOperationOptions & Partial<FileDeleteOptions>): Promise<void>;
    protected doDelete(resource: URI, options?: Partial<FileDeleteOptions>): Promise<void>;
    private onDidFilesChangeEmitter;
    /**
     * An event that is emitted when files are changed on the disk.
     */
    readonly onDidFilesChange: import("@theia/core/lib/common/event").Event<FileChangesEvent>;
    private activeWatchers;
    watch(resource: URI, options?: WatchOptions): Disposable;
    doWatch(resource: URI, options: WatchOptions): Promise<Disposable>;
    private toWatchKey;
    private writeQueues;
    private ensureWriteQueue;
    private toMapKey;
    private doWriteBuffered;
    private doWriteStreamBufferedQueued;
    private doWriteReadableBufferedQueued;
    private doWriteBuffer;
    private doWriteUnbuffered;
    private doWriteUnbufferedQueued;
    private doPipeBuffered;
    private doPipeBufferedQueued;
    private doPipeUnbuffered;
    private doPipeUnbufferedQueued;
    private doPipeUnbufferedToBuffered;
    private doPipeUnbufferedToBufferedQueued;
    private doPipeBufferedToUnbuffered;
    protected throwIfFileSystemIsReadonly<T extends FileSystemProvider>(provider: T, resource: URI): T;
    private resourceForError;
    private readonly participants;
    addFileOperationParticipant(participant: FileOperationParticipant): Disposable;
    runFileOperationParticipants(target: URI, source: URI | undefined, operation: FileOperation): Promise<void>;
    private progressLabel;
    protected getWriteEncoding(resource: URI, options?: WriteEncodingOptions): Promise<ResourceEncoding>;
    protected getReadEncoding(resource: URI, options?: ReadEncodingOptions, detectedEncoding?: string): Promise<string>;
    protected getEncodingForResource(resource: URI, preferredEncoding?: string): Promise<string>;
    /**
     * Converts to an underlying fs provider resource format.
     *
     * For example converting `user-storage` resources to `file` resources under a user home:
     * user-storage:/user/settings.json => file://home/.theia/settings.json
     */
    toUnderlyingResource(resource: URI): Promise<URI>;
    protected handleFileWatchError(): void;
}
export {};
//# sourceMappingURL=file-service.d.ts.map