/********************************************************************************
 * Copyright (C) 2017-2018 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
/// <reference types="node" />
import * as nsfw from '@theia/core/shared/nsfw';
import { IMinimatch } from 'minimatch';
import { FileChangeType, FileSystemWatcherService, FileSystemWatcherServiceClient, WatchOptions } from '../../common/filesystem-watcher-protocol';
import { FileChangeCollection } from '../file-change-collection';
import { Deferred } from '@theia/core/lib/common/promise-util';
export interface NsfwWatcherOptions {
    ignored: IMinimatch[];
}
export declare const NsfwFileSystemWatcherServerOptions: unique symbol;
export interface NsfwFileSystemWatcherServerOptions {
    verbose: boolean;
    info: (message: string, ...args: any[]) => void;
    error: (message: string, ...args: any[]) => void;
    nsfwOptions: nsfw.Options;
}
/**
 * This is a flag value passed around upon disposal.
 */
export declare const WatcherDisposal: unique symbol;
/**
 * Because URIs can be watched by different clients, we'll track
 * how many are listening for a given URI.
 *
 * This component wraps the whole start/stop process given some
 * reference count.
 *
 * Once there are no more references the handle
 * will wait for some time before destroying its resources.
 */
export declare class NsfwWatcher {
    /** Filesystem path to be watched. */
    readonly fsPath: string;
    /** Watcher-specific options */
    readonly watcherOptions: NsfwWatcherOptions;
    /** Logging and Nsfw options */
    protected readonly nsfwFileSystemWatchServerOptions: NsfwFileSystemWatcherServerOptions;
    /** The client to forward events to. */
    protected readonly fileSystemWatcherClient: FileSystemWatcherServiceClient;
    /** Amount of time in ms to wait once this handle is not referenced anymore. */
    protected readonly deferredDisposalTimeout: number;
    protected static debugIdSequence: number;
    protected disposed: boolean;
    /**
     * Used for debugging to keep track of the watchers.
     */
    protected debugId: number;
    /**
     * When this field is set, it means the nsfw instance was successfully started.
     */
    protected nsfw: nsfw.NSFW | undefined;
    /**
     * When the ref count hits zero, we schedule this watch handle to be disposed.
     */
    protected deferredDisposalTimer: NodeJS.Timer | undefined;
    /**
     * This deferred only rejects with `WatcherDisposal` and never resolves.
     */
    protected readonly deferredDisposalDeferred: Deferred<never>;
    /**
     * We count each reference made to this watcher, per client.
     *
     * We do this to know where to send events via the network.
     *
     * An entry should be removed when its value hits zero.
     */
    protected readonly refsPerClient: Map<number, {
        value: number;
    }>;
    /**
     * Ensures that events are processed in the order they are emitted,
     * despite being processed async.
     */
    protected nsfwEventProcessingQueue: Promise<void>;
    /**
     * Resolves once this handle disposed itself and its resources. Never throws.
     */
    readonly whenDisposed: Promise<void>;
    /**
     * Promise that resolves when the watcher is fully started, or got disposed.
     *
     * Will reject if an error occurred while starting.
     *
     * @returns `true` if successfully started, `false` if disposed early.
     */
    readonly whenStarted: Promise<boolean>;
    constructor(
    /** Initial reference to this handle. */
    initialClientId: number, 
    /** Filesystem path to be watched. */
    fsPath: string, 
    /** Watcher-specific options */
    watcherOptions: NsfwWatcherOptions, 
    /** Logging and Nsfw options */
    nsfwFileSystemWatchServerOptions: NsfwFileSystemWatcherServerOptions, 
    /** The client to forward events to. */
    fileSystemWatcherClient: FileSystemWatcherServiceClient, 
    /** Amount of time in ms to wait once this handle is not referenced anymore. */
    deferredDisposalTimeout?: number);
    addRef(clientId: number): void;
    removeRef(clientId: number): void;
    /**
     * All clients with at least one active reference.
     */
    getClientIds(): number[];
    /**
     * Add the references for each client together.
     */
    getTotalReferences(): number;
    /**
     * Returns true if at least one client listens to this handle.
     */
    isInUse(): boolean;
    /**
     * When starting a watcher, we'll first check and wait for the path to exists
     * before running an NSFW watcher.
     */
    protected start(): Promise<void>;
    /**
     * Given a started nsfw instance, gracefully shut it down.
     */
    protected stopNsfw(watcher: nsfw.NSFW): Promise<void>;
    protected createNsfw(): Promise<nsfw.NSFW>;
    protected handleNsfwEvents(events: nsfw.FileChangeEvent[]): void;
    protected resolveEventPath(directory: string, file: string): Promise<string>;
    protected pushFileChange(changes: FileChangeCollection, type: FileChangeType, path: string): void;
    protected fireError(): void;
    /**
     * Wrap a promise to reject as soon as this handle gets disposed.
     */
    protected orCancel<T>(promise: Promise<T>): Promise<T>;
    /**
     * When references hit zero, we'll schedule disposal for a bit later.
     *
     * This allows new references to reuse this watcher instead of creating a new one.
     *
     * e.g. A frontend disconnects for a few milliseconds before reconnecting again.
     */
    protected onRefsReachZero(): void;
    /**
     * If we get new references after hitting zero, let's unschedule our disposal and keep watching.
     */
    protected onRefsRevive(): void;
    protected isIgnored(path: string): boolean;
    /**
     * Internal disposal mechanism.
     */
    protected _dispose(): Promise<void>;
    protected info(prefix: string, ...params: any[]): void;
    protected debug(prefix: string, ...params: any[]): void;
}
/**
 * Each time a client makes a watchRequest, we generate a unique watcherId for it.
 *
 * This watcherId will map to this handle type which keeps track of the clientId that made the request.
 */
export interface NsfwWatcherHandle {
    clientId: number;
    watcher: NsfwWatcher;
}
export declare class NsfwFileSystemWatcherService implements FileSystemWatcherService {
    protected client: FileSystemWatcherServiceClient | undefined;
    protected watcherId: number;
    protected readonly watchers: Map<string, NsfwWatcher>;
    protected readonly watcherHandles: Map<number, NsfwWatcherHandle>;
    protected readonly options: NsfwFileSystemWatcherServerOptions;
    /**
     * `this.client` is undefined until someone sets it.
     */
    protected readonly maybeClient: FileSystemWatcherServiceClient;
    constructor(options?: Partial<NsfwFileSystemWatcherServerOptions>);
    setClient(client: FileSystemWatcherServiceClient | undefined): void;
    /**
     * A specific client requests us to watch a given `uri` according to some `options`.
     *
     * We internally re-use all the same `(uri, options)` pairs.
     */
    watchFileChanges(clientId: number, uri: string, options?: WatchOptions): Promise<number>;
    protected createWatcher(clientId: number, fsPath: string, options: WatchOptions): NsfwWatcher;
    unwatchFileChanges(watcherId: number): Promise<void>;
    /**
     * Given some `URI` and some `WatchOptions`, generate a unique key.
     */
    protected getWatcherKey(uri: string, options: WatchOptions): string;
    /**
     * Return fully qualified options.
     */
    protected resolveWatchOptions(options?: WatchOptions): WatchOptions;
    protected debug(message: string, ...params: any[]): void;
    dispose(): void;
}
//# sourceMappingURL=nsfw-filesystem-service.d.ts.map