"use strict";
/********************************************************************************
 * Copyright (C) 2017 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
var jsdom_1 = require("@theia/core/lib/browser/test/jsdom");
var disableJSDOM = jsdom_1.enableJSDOM();
var chai = require("chai");
var uri_1 = require("@theia/core/lib/common/uri");
var inversify_1 = require("@theia/core/shared/inversify");
var problem_manager_1 = require("./problem-manager");
var event_1 = require("@theia/core/lib/common/event");
var logger_1 = require("@theia/core/lib/common/logger");
var vscode_languageserver_types_1 = require("@theia/core/shared/vscode-languageserver-types");
var mock_logger_1 = require("@theia/core/lib/common/test/mock-logger");
var file_service_1 = require("@theia/filesystem/lib/browser/file-service");
var storage_service_1 = require("@theia/core/lib/browser/storage-service");
disableJSDOM();
var expect = chai.expect;
var manager;
var container;
/**
 * The default range for test purposes.
 */
var range = { start: { line: 0, character: 10 }, end: { line: 0, character: 10 } };
describe('problem-manager', function () {
    beforeEach(function () {
        container = new inversify_1.Container();
        container.bind(logger_1.ILogger).to(mock_logger_1.MockLogger);
        container.bind(storage_service_1.StorageService).to(storage_service_1.LocalStorageService).inSingletonScope();
        container.bind(storage_service_1.LocalStorageService).toSelf().inSingletonScope();
        container.bind(file_service_1.FileService).toConstantValue({
            onDidFilesChange: event_1.Event.None
        });
        container.bind(problem_manager_1.ProblemManager).toSelf();
        manager = container.get(problem_manager_1.ProblemManager);
    });
    describe('#setMarkers', function () {
        it('should successfully set new markers', function () {
            expect(Array.from(manager.getUris()).length).to.equal(0);
            manager.setMarkers(new uri_1.default('a'), 'a', [{ message: 'a', range: range }]);
            expect(Array.from(manager.getUris()).length).to.equal(1);
        });
        it('should replace markers', function () {
            var uri = new uri_1.default('a');
            var events = 0;
            manager.onDidChangeMarkers(function () { return events++; });
            expect(events).equal(0);
            var initial = manager.setMarkers(uri, 'a', [{ message: 'a', range: range }]);
            expect(initial.length).equal(0);
            expect(events).equal(1);
            var updated = manager.setMarkers(uri, 'a', [{ message: 'a', range: range }]);
            expect(updated.length).equal(1);
            expect(events).equal(2);
            expect(manager.findMarkers({ uri: uri }).length).equal(1);
        });
    });
    describe('#cleanAllMarkers', function () {
        it('should successfully clean all markers', function () {
            // Create mock markers.
            manager.setMarkers(new uri_1.default('a'), 'a', [{ message: 'a', range: range }]);
            manager.setMarkers(new uri_1.default('b'), 'b', [{ message: 'a', range: range }]);
            manager.setMarkers(new uri_1.default('c'), 'c', [{ message: 'a', range: range }]);
            expect(Array.from(manager.getUris()).length).to.equal(3);
            // Clean the markers.
            manager.cleanAllMarkers();
            expect(Array.from(manager.getUris()).length).to.equal(0);
        });
    });
    describe('#findMarkers', function () {
        it('should find markers by `owner`', function () {
            var owner = 'foo';
            manager.setMarkers(new uri_1.default('a'), owner, [{ message: 'a', range: range }]);
            manager.setMarkers(new uri_1.default('b'), owner, [{ message: 'a', range: range }]);
            expect(manager.findMarkers({ owner: owner }).length).equal(2);
            expect(manager.findMarkers({ owner: 'unknown' }).length).equal(0);
        });
        it('should find markers by `owner` and `uri`', function () {
            var owner = 'foo';
            var uri = new uri_1.default('bar');
            // Create a marker to match the filter.
            manager.setMarkers(uri, owner, [{ message: 'a', range: range }]);
            // Create 2 markers that do not match the filter.
            manager.setMarkers(new uri_1.default('invalid'), 'invalid-owner', [{ message: 'a', range: range }]);
            manager.setMarkers(new uri_1.default('invalid'), 'invalid-owner', [{ message: 'a', range: range }]);
            // Expect to find the markers which satisfy the filter only.
            expect(manager.findMarkers({ owner: owner, uri: uri }).length).equal(1);
        });
        describe('dataFilter', function () {
            it('should find markers that satisfy filter for `severity`', function () {
                manager.setMarkers(new uri_1.default('a'), 'a', [{ message: 'a', range: range, severity: vscode_languageserver_types_1.DiagnosticSeverity.Error }]);
                expect(manager.findMarkers({ dataFilter: function (d) { return d.severity === vscode_languageserver_types_1.DiagnosticSeverity.Error; } }).length).equal(1);
                expect(manager.findMarkers({ dataFilter: function (d) { return d.severity !== vscode_languageserver_types_1.DiagnosticSeverity.Error; } }).length).equal(0);
            });
            it('should find markers that satisfy filter for `code`', function () {
                var code = 100;
                manager.setMarkers(new uri_1.default('a'), 'a', [{ message: 'a', range: range, code: code }]);
                expect(manager.findMarkers({ dataFilter: function (d) { return d.code === code; } }).length).equal(1);
                expect(manager.findMarkers({ dataFilter: function (d) { return d.code !== code; } }).length).equal(0);
            });
            it('should find markers that satisfy filter for `message`', function () {
                var message = 'foo';
                manager.setMarkers(new uri_1.default('a'), 'a', [{ message: message, range: range }]);
                expect(manager.findMarkers({ dataFilter: function (d) { return d.message === message; } }).length).equal(1);
                expect(manager.findMarkers({ dataFilter: function (d) { return d.message !== message; } }).length).equal(0);
            });
            it('should find markers that satisfy filter for `source`', function () {
                var source = 'typescript';
                manager.setMarkers(new uri_1.default('a'), 'a', [{ message: 'a', range: range, source: source }]);
                expect(manager.findMarkers({ dataFilter: function (d) { return d.source === source; } }).length).equal(1);
                expect(manager.findMarkers({ dataFilter: function (d) { return d.source !== source; } }).length).equal(0);
            });
            it('should find markers that satisfy filter for `range`', function () {
                manager.setMarkers(new uri_1.default('a'), 'a', [{ message: 'a', range: range }]);
                // The default `range` has a start line number of 0.
                expect(manager.findMarkers({ dataFilter: function (d) { return d.range.start.line === 0; } }).length).equal(1);
                expect(manager.findMarkers({ dataFilter: function (d) { return d.range.start.line > 0; } }).length).equal(0);
            });
        });
    });
    describe('#getUris', function () {
        it('should return 0 uris when no markers are present', function () {
            expect(Array.from(manager.getUris()).length).to.equal(0);
        });
        it('should return the list of uris', function () {
            manager.setMarkers(new uri_1.default('a'), 'a', [{ message: 'a', range: range, severity: vscode_languageserver_types_1.DiagnosticSeverity.Error }]);
            manager.setMarkers(new uri_1.default('b'), 'b', [{ message: 'a', range: range, severity: vscode_languageserver_types_1.DiagnosticSeverity.Error }]);
            expect(Array.from(manager.getUris()).length).to.equal(2);
        });
    });
    describe('#getProblemStat', function () {
        it('should return 0 stats when no markers are present', function () {
            var _a = manager.getProblemStat(), errors = _a.errors, warnings = _a.warnings, infos = _a.infos;
            expect(errors).to.equal(0);
            expect(warnings).to.equal(0);
            expect(infos).to.equal(0);
        });
        it('should return the proper problem stats', function () {
            // Create 3 error markers.
            manager.setMarkers(new uri_1.default('error-1'), 'error-1', [{ message: 'a', range: range, severity: vscode_languageserver_types_1.DiagnosticSeverity.Error }]);
            manager.setMarkers(new uri_1.default('error-2'), 'error-2', [{ message: 'a', range: range, severity: vscode_languageserver_types_1.DiagnosticSeverity.Error }]);
            manager.setMarkers(new uri_1.default('error-3'), 'error-3', [{ message: 'a', range: range, severity: vscode_languageserver_types_1.DiagnosticSeverity.Error }]);
            // Create 2 warning markers.
            manager.setMarkers(new uri_1.default('warning-1'), 'warning-1', [{ message: 'a', range: range, severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning }]);
            manager.setMarkers(new uri_1.default('warning-2'), 'warning-2', [{ message: 'a', range: range, severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning }]);
            // Create 1 info marker.
            manager.setMarkers(new uri_1.default('info-1'), 'info-1', [{ message: 'a', range: range, severity: vscode_languageserver_types_1.DiagnosticSeverity.Information }]);
            // Collect the total problem stats for the application.
            var _a = manager.getProblemStat(), errors = _a.errors, warnings = _a.warnings, infos = _a.infos;
            expect(errors).to.equal(3);
            expect(warnings).to.equal(2);
            expect(infos).to.equal(1);
        });
    });
});
//# sourceMappingURL=problem-manager.spec.js.map