/********************************************************************************
 * Copyright (C) 2017 Ericsson and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
import { LoggerWatcher } from './logger-watcher';
import { ILoggerServer, LogLevel, rootLoggerName } from './logger-protocol';
export { LogLevel, rootLoggerName };
export declare let logger: ILogger;
/**
 * Counterpart of the `#setRootLogger(ILogger)`. Restores the `console.xxx` bindings to the original one.
 * Invoking has no side-effect if `setRootLogger` was not called before. Multiple function invocation has
 * no side-effect either.
 */
export declare function unsetRootLogger(): void;
export declare function setRootLogger(aLogger: ILogger): void;
export declare type Log = (message: any, ...params: any[]) => void;
export declare type Loggable = (log: Log) => void;
export declare const LoggerFactory: unique symbol;
export declare type LoggerFactory = (name: string) => ILogger;
export declare const LoggerName: unique symbol;
export declare const ILogger: unique symbol;
export interface ILogger {
    /**
     * Set the log level.
     *
     * @param loglevel - The loglevel to set. see Logger.LogLevel for
     * possible options.
     */
    setLogLevel(logLevel: number): Promise<void>;
    /**
     * Get the log level.
     *
     * @returns a Promise to the log level.
     */
    getLogLevel(): Promise<number>;
    /**
     * Test whether the given log level is enabled.
     */
    isEnabled(logLevel: number): Promise<boolean>;
    /**
     * Resolve if the given log is enabled.
     */
    ifEnabled(logLevel: number): Promise<void>;
    /**
     * Log a loggable with the given level if it is enabled.
     */
    log(logLevel: number, loggable: Loggable): Promise<void>;
    /**
     * Log a message with the given level if it is enabled.
     *
     * @param logLevel - The loglevel to use.
     * @param message - The message format string.
     * @param params - The format string variables.
     */
    log(logLevel: number, message: any, ...params: any[]): void;
    /**
     * Test whether the trace level is enabled.
     */
    isTrace(): Promise<boolean>;
    /**
     * Resolve if the trace level is enabled.
     */
    ifTrace(): Promise<void>;
    /**
     * Log a loggable with the trace level if it is enabled.
     */
    trace(loggable: Loggable): Promise<void>;
    /**
     * Log a message with the trace level if it is enabled.
     *
     * @param message - The message format string.
     * @param params - The format string variables.
     */
    trace(message: any, ...params: any[]): Promise<void>;
    /**
     * Test whether the debug level is enabled.
     */
    isDebug(): Promise<boolean>;
    /**
     * Resolve if the debug level is enabled.
     */
    ifDebug(): Promise<void>;
    /**
     * Log a loggable with the debug level if it is enabled.
     */
    debug(loggable: Loggable): Promise<void>;
    /**
     * Log a message with the debug level if it is enabled.
     *
     * @param message - The message format string.
     * @param params - The format string variables.
     */
    debug(message: any, ...params: any[]): Promise<void>;
    /**
     * Test whether the info level is enabled.
     */
    isInfo(): Promise<boolean>;
    /**
     * Resolve if the info level is enabled.
     */
    ifInfo(): Promise<void>;
    /**
     * Log a loggable with the info level if it is enabled.
     */
    info(loggable: Loggable): Promise<void>;
    /**
     * Log a message with the info level if it is enabled.
     *
     * @param message - The message format string.
     * @param params - The format string variables.
     */
    info(message: any, ...params: any[]): Promise<void>;
    /**
     * Test whether the warn level is enabled.
     */
    isWarn(): Promise<boolean>;
    /**
     * Resolve if the warn level is enabled.
     */
    ifWarn(): Promise<void>;
    /**
     * Log a loggable with the warn level if it is enabled.
     */
    warn(loggable: Loggable): Promise<void>;
    /**
     * Log a message with the warn level if it is enabled.
     *
     * @param message - The message format string.
     * @param params - The format string variables.
     */
    warn(message: any, ...params: any[]): Promise<void>;
    /**
     * Test whether the error level is enabled.
     */
    isError(): Promise<boolean>;
    /**
     * Resolve if the error level is enabled.
     */
    ifError(): Promise<void>;
    /**
     * Log a loggable with the error level if it is enabled.
     */
    error(loggable: Loggable): Promise<void>;
    /**
     * Log a message with the error level.
     *
     * @param message - The message format string.
     * @param params - The format string variables.
     */
    error(message: any, ...params: any[]): Promise<void>;
    /**
     * Test whether the fatal level is enabled.
     */
    isFatal(): Promise<boolean>;
    /**
     * Resolve if the fatal level is enabled.
     */
    ifFatal(): Promise<void>;
    /**
     * Log a loggable with the fatal level if it is enabled.
     */
    fatal(loggable: Loggable): Promise<void>;
    /**
     * Log a message with the fatal level if it is enabled.
     *
     * @param message - The message format string.
     * @param params - The format string variables.
     */
    fatal(message: any, ...params: any[]): Promise<void>;
    /**
     * Create a child logger from this logger.
     *
     * @param name - The name of the child logger.
     */
    child(name: string): ILogger;
}
export declare class Logger implements ILogger {
    protected readonly server: ILoggerServer;
    protected readonly loggerWatcher: LoggerWatcher;
    protected readonly factory: LoggerFactory;
    protected name: string;
    protected _logLevel: Promise<number>;
    protected created: Promise<void>;
    /**
     * Build a new Logger.
     */
    constructor(server: ILoggerServer, loggerWatcher: LoggerWatcher, factory: LoggerFactory, name: string);
    setLogLevel(logLevel: number): Promise<void>;
    getLogLevel(): Promise<number>;
    isEnabled(logLevel: number): Promise<boolean>;
    ifEnabled(logLevel: number): Promise<void>;
    log(logLevel: number, arg2: any | Loggable, ...params: any[]): Promise<void>;
    protected getLog(logLevel: number): Promise<Log>;
    protected format(value: any): any;
    isTrace(): Promise<boolean>;
    ifTrace(): Promise<void>;
    trace(arg: any | Loggable, ...params: any[]): Promise<void>;
    isDebug(): Promise<boolean>;
    ifDebug(): Promise<void>;
    debug(arg: any | Loggable, ...params: any[]): Promise<void>;
    isInfo(): Promise<boolean>;
    ifInfo(): Promise<void>;
    info(arg: any | Loggable, ...params: any[]): Promise<void>;
    isWarn(): Promise<boolean>;
    ifWarn(): Promise<void>;
    warn(arg: any | Loggable, ...params: any[]): Promise<void>;
    isError(): Promise<boolean>;
    ifError(): Promise<void>;
    error(arg: any | Loggable, ...params: any[]): Promise<void>;
    isFatal(): Promise<boolean>;
    ifFatal(): Promise<void>;
    fatal(arg: any | Loggable, ...params: any[]): Promise<void>;
    child(name: string): ILogger;
}
//# sourceMappingURL=logger.d.ts.map