"use strict";
/**********************************************************************
 * Copyright (c) 2021 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 ***********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.Main = void 0;
const axios = require("axios");
const generate_1 = require("./generate");
const inversify_binding_1 = require("./inversify/inversify-binding");
const devfile_context_1 = require("./api/devfile-context");
class Main {
    async doStart() {
        let devfileUrl;
        let outputFile;
        let pluginRegistryUrl;
        let editor;
        let sidecarPolicy;
        const args = process.argv.slice(2);
        args.forEach(arg => {
            if (arg.startsWith('--devfile-url:')) {
                devfileUrl = arg.substring('--devfile-url:'.length);
            }
            if (arg.startsWith('--plugin-registry-url:')) {
                pluginRegistryUrl = arg.substring('--plugin-registry-url:'.length);
            }
            if (arg.startsWith('--editor:')) {
                editor = arg.substring('--editor:'.length);
            }
            if (arg.startsWith('--output-file:')) {
                outputFile = arg.substring('--output-file:'.length);
            }
            if (arg.startsWith('--sidecar-policy:')) {
                const policy = arg.substring('--sidecar-policy:'.length);
                if (policy === devfile_context_1.SidecarPolicy.MERGE_IMAGE.toString()) {
                    sidecarPolicy = devfile_context_1.SidecarPolicy.MERGE_IMAGE;
                }
                else if (policy === devfile_context_1.SidecarPolicy.USE_DEV_CONTAINER.toString()) {
                    sidecarPolicy = devfile_context_1.SidecarPolicy.USE_DEV_CONTAINER;
                }
                else {
                    throw new Error(`${policy} is not a valid sidecar policy. Available values are ${Object.values(devfile_context_1.SidecarPolicy)}`);
                }
            }
        });
        if (!pluginRegistryUrl) {
            pluginRegistryUrl = 'https://eclipse-che.github.io/che-plugin-registry/main/v3';
            console.log(`No plug-in registry url. Setting to ${pluginRegistryUrl}`);
        }
        if (!editor) {
            editor = 'eclipse/che-theia/next';
            console.log(`No editor. Setting to ${editor}`);
        }
        if (!sidecarPolicy) {
            sidecarPolicy = devfile_context_1.SidecarPolicy.USE_DEV_CONTAINER;
            console.log(`No sidecar policy. Setting to ${sidecarPolicy}`);
        }
        if (!devfileUrl) {
            throw new Error('missing --devfile-url: parameter');
        }
        if (!outputFile) {
            throw new Error('missing --output-file: parameter');
        }
        const axiosInstance = axios.default;
        const inversifyBinbding = new inversify_binding_1.InversifyBinding();
        const container = await inversifyBinbding.initBindings({
            pluginRegistryUrl,
            insertTemplates: true,
            axiosInstance,
        });
        container.bind(generate_1.Generate).toSelf().inSingletonScope();
        const generate = container.get(generate_1.Generate);
        return generate.generate(devfileUrl, editor, sidecarPolicy, outputFile);
    }
    async start() {
        try {
            await this.doStart();
            return true;
        }
        catch (error) {
            console.error('stack=' + error.stack);
            console.error('Unable to start', error);
            return false;
        }
    }
}
exports.Main = Main;
//# sourceMappingURL=main.js.map