"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = getStylelint;

var _os = require("os");

var _jestWorker = require("jest-worker");

var _worker = require("./worker");

var _utils = require("./utils");

var _options = require("./options");

// @ts-ignore

/** @type {{[key: string]: any}} */
const cache = {};
/** @typedef {import('stylelint')} Stylelint */

/** @typedef {import('stylelint').LintResult} LintResult */

/** @typedef {import('./options').Options} Options */

/** @typedef {() => Promise<void>} AsyncTask */

/** @typedef {(files: string|string[]) => Promise<LintResult[]>} LintTask */

/** @typedef {JestWorker & {lintFiles: LintTask}} Worker */

/** @typedef {{stylelint: Stylelint, lintFiles: LintTask, cleanup: AsyncTask, threads: number, }} Linter */

/**
 * @param {Options} options
 * @returns {Linter}
 */

function loadStylelint(options) {
  const stylelint = (0, _worker.setup)(options, (0, _options.getStylelintOptions)(options));
  return {
    stylelint,
    lintFiles: _worker.lintFiles,
    cleanup: async () => {},
    threads: 1
  };
}
/**
 * @param {string|undefined} key
 * @param {number} poolSize
 * @param {Options} options
 * @returns {Linter}
 */


function loadStylelintThreaded(key, poolSize, options) {
  const cacheKey = getCacheKey(key, options);

  const source = require.resolve('./worker');

  const workerOptions = {
    enableWorkerThreads: true,
    numWorkers: poolSize,
    setupArgs: [options, (0, _options.getStylelintOptions)(options)]
  };
  const local = loadStylelint(options);
  /** @type {Worker?} */
  // prettier-ignore

  let worker =
  /** @type {Worker} */
  new _jestWorker.Worker(source, workerOptions);
  /** @type {Linter} */

  const context = { ...local,
    threads: poolSize,
    lintFiles: async files =>
    /* istanbul ignore next */
    worker ? worker.lintFiles(files) : local.lintFiles(files),
    cleanup: async () => {
      cache[cacheKey] = local;

      context.lintFiles = files => local.lintFiles(files);
      /* istanbul ignore next */


      if (worker) {
        worker.end();
        worker = null;
      }
    }
  };
  return context;
}
/**
 * @param {string|undefined} key
 * @param {Options} options
 * @returns {Linter}
 */


function getStylelint(key, {
  threads,
  ...options
}) {
  const max = typeof threads !== 'number' ? threads ? (0, _os.cpus)().length - 1 : 1 : threads;
  const cacheKey = getCacheKey(key, {
    threads,
    ...options
  });

  if (!cache[cacheKey]) {
    cache[cacheKey] = max > 1 ? loadStylelintThreaded(key, max, options) : loadStylelint(options);
  }

  return cache[cacheKey];
}
/**
 * @param {string|undefined} key
 * @param {Options} options
 * @returns {string}
 */


function getCacheKey(key, options) {
  return JSON.stringify({
    key,
    options
  }, _utils.jsonStringifyReplacerSortKeys);
}